﻿/*
 * Copyright 2013 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using JetBrains.Annotations;
using Nehta.VendorLibrary.CDA.Common.Enums;
using Nehta.VendorLibrary.Common;

namespace Nehta.VendorLibrary.CDA.SCSModel.Common
{
    /// <summary>
    /// Medication class
    /// 
    /// This class encapsulates medication information that is used to populate medication 
    /// related sections / information with CDA documents
    /// </summary>
    [Serializable]
    [DataContract]
    [KnownType(typeof(CodableText))]
    internal class Medication : IMedicationInstruction, IMedication, IMedicationItem
    {
        #region Properties
        /// <summary>
        /// The medicne as a codableText
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ICodableText Medicine { get; set; }

        /// <summary>
        /// The directions of use for this medication
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String Directions { get; set; }

        /// <summary>
        /// The directions NullFlavour
        /// </summary>
        [CanBeNull]
        public NullFlavour? DirectionsNullFlavour { get; set; }

        /// <summary>
        /// The clinical indication associated with this medication
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String ClinicalIndication { get; set; }

        /// <summary>
        /// Any comments associated with this medication
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String Comment { get; set; }

        /// <summary>
        /// Change type
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ChangeType? ChangeType { get; set; }

        /// <summary>
        /// ChangeType NullFlavour
        /// </summary>
        [CanBeNull]
        [DataMember]
        public NullFlavour? ChangeTypeNullFlavour { get; set; }

        /// <summary>
        /// Recommendation or change
        /// </summary>
        [CanBeNull]
        [DataMember]
        public RecomendationOrChange? RecommendationOrChange { get; set; }

        /// <summary>
        /// Recommendation or change NullFlavour
        /// </summary>
        [CanBeNull]
        public NullFlavour? RecommendationOrChangeNullFlavour { get; set; }

        /// <summary>
        /// Change description
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String ChangeDescription { get; set; }

        /// <summary>
        /// Change reason
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String ChangeReason { get; set; }
        #endregion

        #region Constructors
        internal Medication()
        {
        }
        #endregion

        #region Validation
        /// <summary>
        /// Validates this Medication
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        void IMedicationInstruction.Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("Medicine", Medicine))
            {
                if (Medicine != null) Medicine.Validate(vb.Path + "Medicine", vb.Messages);
            }

            // Directions can only contain a Directions or a Null Flavor 
            var directionsNullFlavour = new Dictionary<string, object>()
            {
                { "Directions", Directions },
                { "DirectionsNullFlavour", DirectionsNullFlavour }
            };
            vb.ChoiceCheck(directionsNullFlavour);
        }

        /// <summary>
        /// Validates this Medication
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        void IMedication.Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("Medicine", Medicine))
            {
                Medicine.ValidateMandatory(vb.Path + "Medicine", vb.Messages);
            }

            // Directions can only contain a Directions or a Null Flavor 
            var directionsNullFlavour = new Dictionary<string, object>()
            {
                { "Directions", Directions },
                { "DirectionsNullFlavour", DirectionsNullFlavour }
            };
            vb.ChoiceCheck(directionsNullFlavour);
        }

        /// <summary>
        /// Validates this Medication
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        void IMedicationItem.Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("Medicine", Medicine))
            {
                if (Medicine != null) Medicine.ValidateMandatory(vb.Path, vb.Messages);
            }

            // ChangeType can only contain a changeType or NullFlavor 
            var changeType = new Dictionary<string, object>()
            {
                { "ChangeType", ChangeType },
                { "ChangeTypeNullFlavour", ChangeTypeNullFlavour }
            };
            vb.ChoiceCheck(changeType);


            // Recommendation Or Change can only contain a recommendationOrChange or a NullFlavor 
            var recommendationOrChange = new Dictionary<string, object>()
            {
                { "RecommendationOrChange", RecommendationOrChange },
                { "RecommendationOrChangeNullFlavour", RecommendationOrChangeNullFlavour }
            };
            vb.ChoiceCheck(recommendationOrChange);

              // Directions can only contain a Directions or a Null Flavor 
            var directionsNullFlavour = new Dictionary<string, object>()
            {
                  { "Directions", Directions },
                  { "DirectionsNullFlavour", DirectionsNullFlavour }
            };
            vb.ChoiceCheck(directionsNullFlavour);
        }

        #endregion
    }
}