﻿/*
 * Copyright 2013 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Runtime.Serialization;
using JetBrains.Annotations;
using Nehta.VendorLibrary.CDA.Common;
using Nehta.VendorLibrary.Common;
using System.Collections.Generic;

namespace Nehta.VendorLibrary.CDA.SCSModel.Common
{
    /// <summary>
    /// This class is designed to encapsulate the properties within a CDA document that make up 
    /// an problem diagnosis
    /// </summary>
    [Serializable]
    [DataContract]
    [KnownType(typeof(CodableText))]
    [KnownType(typeof(Statement))]
    [KnownType(typeof(Nehta.VendorLibrary.CDA.SCSModel.DischargeSummary.ProblemDiagnosis))]
    public class ProblemDiagnosis : IProblemDiagnosis, IProblemDiagnosisEventSummary
    {
        #region Properties
        /// <summary>
        /// Problem or diagnosis ID
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ICodableText ProblemDiagnosisIdentification { get; set; }

        /// <summary>
        /// Date of onset
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ISO8601DateTime DateOfOnset { get; set; }

        /// <summary>
        /// Date of remission
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ISO8601DateTime DateOfResolutionRemission { get; set; }

        /// <summary>
        /// Comments
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String Comment { get; set; }

        /// <summary>
        /// Problem Diagnosis Type
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ICodableText ProblemDiagnosisType { get; set; }

        /// <summary>
        /// Problem Diagnosis Description
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ICodableText ProblemDiagnosisDescription { get; set; }

        #endregion

        #region Constructors
        internal ProblemDiagnosis()
        {
        }
        #endregion

        #region Validation

        /// <summary>
        /// Validates this problem / diagnosis
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        public void Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("ProblemDiagnosisIdentification", ProblemDiagnosisIdentification))
            {
                if (ProblemDiagnosisIdentification != null)
                    ProblemDiagnosisIdentification.ValidateMandatory(vb.Path + "ProblemDiagnosisIdentification", vb.Messages);
            }

            if (DateOfResolutionRemission != null)
            {
              if (DateOfResolutionRemission.PrecisionIndicator.HasValue && DateOfResolutionRemission.TimeZone == null)
              {
                if (!(DateOfResolutionRemission.PrecisionIndicator.Value == ISO8601DateTime.Precision.Day) ||
                    (DateOfResolutionRemission.PrecisionIndicator.Value == ISO8601DateTime.Precision.Year))
                    {
                      vb.AddValidationMessage(vb.PathName, null, string.Format("DateOfResolutionRemission must have a precision of 'Day' or 'Year' with no Time Zone specified"));
                    }
              }
              else
              {
                vb.AddValidationMessage(vb.PathName, null, string.Format("DateOfResolutionRemission must have a precision of 'Day' or 'Year' with no Time Zone specified"));
              }
            }
        }

        /// <summary>
        /// Validates this problem / diagnosis
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        void IProblemDiagnosis.Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("ProblemDiagnosisIdentification", ProblemDiagnosisIdentification))
            {
                if (ProblemDiagnosisIdentification != null) ProblemDiagnosisIdentification.Validate(vb.Path + "ProblemDiagnosisIdentification", vb.Messages);
            }

            if (DateOfResolutionRemission != null)
            {
              if (DateOfResolutionRemission.PrecisionIndicator.HasValue && DateOfResolutionRemission.TimeZone == null)
              {
                if (!(DateOfResolutionRemission.PrecisionIndicator.Value == ISO8601DateTime.Precision.Day ||
                    DateOfResolutionRemission.PrecisionIndicator.Value == ISO8601DateTime.Precision.Hour ||
                    DateOfResolutionRemission.PrecisionIndicator.Value == ISO8601DateTime.Precision.Year))
                    {
                      vb.AddValidationMessage(vb.PathName, null, string.Format("DateOfResolutionRemission must have a precision of 'Day', 'Hour' or 'Year' with no Time Zone specified"));
                    }
              }
              else
              {
                vb.AddValidationMessage(vb.PathName, null, string.Format("DateOfResolutionRemission must have a precision of 'Day', 'Hour' or 'Year' with no Time Zone specified"));
              }
            }
        }

        /// <summary>
        /// Validates this problem / diagnosis
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        void IProblemDiagnosisEventSummary.Validate(string path, List<ValidationMessage> messages)
        {
          var vb = new ValidationBuilder(path, messages);

          if (vb.ArgumentRequiredCheck("ProblemDiagnosisIdentification", ProblemDiagnosisIdentification))
          {
            if (ProblemDiagnosisIdentification != null) ProblemDiagnosisIdentification.Validate(vb.Path + "ProblemDiagnosisIdentification", vb.Messages);
          }
        }
        
        #endregion
    }
}