﻿/*
 * Copyright 2011 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Runtime.Serialization;
using JetBrains.Annotations;
using Nehta.VendorLibrary.CDA.SCSModel.Common;
using Nehta.VendorLibrary.CDA.Common.Enums;
using System.Collections.Generic;
using Nehta.VendorLibrary.Common;

namespace Nehta.VendorLibrary.CDA.SCSModel
{
    /// <summary>
    /// The PrescriptionRequestItem class contains all the properties that CDA has identified for 
    /// a prescription request item
    /// 
    /// Please use the CreatePrescriptionRequestItem() method on the appropriate parent SCS object to 
    /// instantiate this class.
    /// </summary>
    [Serializable]
    [DataContract]
    public class PrescriptionRequestItem 
    {
        #region Properties

        /// <summary>
        /// Therapeutic good ID
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ICodableText TherapeuticGoodId { get; set; }

        /// <summary>
        /// Formula
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String Formula { get; set; }

        /// <summary>
        /// The quantity of the therapeutic good
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String QuantityOfTherapeuticGood { get; set; }

        /// <summary>
        /// A boolean indicating if brand substitution is allowed
        /// </summary>
        [CanBeNull]
        [DataMember]
        public Boolean? BrandSubstituteAllowed { get; set; }

        /// <summary>
        /// Additional comments
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String AdditionalComments { get; set; }

        /// <summary>
        /// The instructions for use of the item, E.g. take twice daily
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String Instructions { get; set; }

        /// <summary>
        /// The medical benefit category type, E.g. PBS
        /// </summary>
        [CanBeNull]
        [DataMember]
        public MedicalBenefitCategoryType MedicalBenefitCategoryType { get; set; }

        /// <summary>
        /// PBS RPBS authority approval number
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String PBSRPBSAuthorityApprovalNumber { get; set; }

        /// <summary>
        /// State authority number
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String StateAuthorityNumber { get; set; }

        #endregion

        #region Constructors
        internal PrescriptionRequestItem()
        {
        }
        #endregion

        #region Validation
        /// <summary>
        /// Validates this Prescription Request Item
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        public void Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("TherapeuticGoodId", TherapeuticGoodId))
            {
                if (TherapeuticGoodId != null) TherapeuticGoodId.Validate(vb.Path + "TherapeuticGoodId", vb.Messages);
            }

            vb.ArgumentRequiredCheck("QuantityOfTherapeuticGood", QuantityOfTherapeuticGood);
        }

        #endregion
    }
}