﻿using System;
using System.Diagnostics;
using System.Threading;
using System.Web.Hosting;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.AppServer.HIPSServiceHost.BackgroundThread
{
    /// <summary>
    /// Custom Service Host to allow a separate thread to Clean up IHI.
    /// </summary>
    public class IhiCleanUpServiceHost : System.ServiceModel.ServiceHost, IRegisteredObject
    {
        private static bool _isThreadRunning;
        private bool _appDomainClosing = false;
        private TimerState _state = new TimerState();
        private Object _threadLock = new object();
        private Timer _timer;
        private UserDetails ihiCleanupUser = new UserDetails();

        /// <summary>
        /// Constructor where only the service type is known. The base addresses from
        /// configuration are used instead.
        /// </summary>
        /// <param name="serviceType"></param>
        public IhiCleanUpServiceHost(Type serviceType)
            : base(serviceType)
        {
            HostingEnvironment.RegisterObject(this);
        }

        /// <summary>
        /// Constructor that passes both the service type And the base addresses
        /// </summary>
        /// <param name="serviceType"></param>
        /// <param name="baseAddresses"></param>
        public IhiCleanUpServiceHost(Type serviceType, params Uri[] baseAddresses)
            : base(serviceType, baseAddresses)
        {
            HostingEnvironment.RegisterObject(this);
        }

        /// <summary>
        /// Callback event when Timer "ticks"
        /// </summary>
        public event TimerCallback IhiCleanUpCallback;

        /// <summary>
        /// Called when the appdomain is closing to stop the registered object
        /// Need to ensure that the timer can't start if stopping, but needs to wait for a running process
        /// </summary>
        /// <param name="immediate"></param>
        public void Stop(bool immediate)
        {
            lock (_threadLock)
            {
                _appDomainClosing = true;
            }
            HostingEnvironment.UnregisterObject(this);
        }

        /// <summary>
        /// Override InitializeRuntime to start the Thread
        /// </summary>
        protected override void InitializeRuntime()
        {
            base.InitializeRuntime();

            ihiCleanupUser.Role = UserRole.AuthorisedEmployee;
            ihiCleanupUser.AuthorisedEmployeeUserId = "PatientIhiCleanupUserId";
            ihiCleanupUser.Name = "PatientIhiCleanupUserName";

            RunThread();
            Debug.WriteLine(String.Format("{0} - (InitializeRuntime) Background Thread timer started", DateTime.Now.ToString()));

            EventLogger.WriteLog(String.Format("(INFO) (InitializeRuntime) Background Thread timer started"), null, ihiCleanupUser, LogMessage.HIPS_MESSAGE_116);
        }

        /// <summary>
        /// Override OnOpened to start the thread when the application is opened.
        /// </summary>
        protected override void OnOpened()
        {
            base.OnOpened();
            RunThread();
            Debug.WriteLine(String.Format("{0} - (OnOpened) Background Thread timer started", DateTime.Now.ToString()));
        }

        /// <summary>
        /// Runs on a separate thread:
        /// Invokes the Deliver() method and checks if any mail is left to send.
        /// If not the thread sleeps for a specified period of time.
        /// </summary>
        /// <param name="state"></param>
        private void IhiCleanUpServiceHost_IhiCleanUpCallback(object state)
        {
            System.Diagnostics.Debug.WriteLine("IhiCleanUpServiceHost_IhiCleanUpCallback:" + DateTime.Now.ToString("dd/MM/yyyy hh:mm:ss.ttt"));
            Debug.WriteLine("IhiCleanupServiceHost.IhiCleanUpServiceHost_IhiCleanUpCallback Started: " + DateTime.Now.ToString("dd/MM/yyyy hh:mm:ss.ttt"));
            if (!_isThreadRunning)
            {
                _isThreadRunning = true;

                // Lock the resources
                lock (_threadLock)
                {
                    if (_appDomainClosing)
                    {
                        //halt immediately if appdomain closure is detected
                        return;
                    }

                    TimerState timerState = (TimerState)state;
                    int result = -1;

                    try
                    {
                        Debug.WriteLine(String.Format("{0} - Ihi CleanUp started", DateTime.Now.ToString()));

                        PatientIhiCleanup cleanup = new PatientIhiCleanup();
                        cleanup.Cleanup();

                        result = 0; //all ok

                        Debug.WriteLine(String.Format("{0} - Ihi CleanUp Completed with result of {1}", DateTime.Now.ToString(), result.ToString()));
                    }
                    catch (Exception e)
                    {
                        try
                        {
                            Debug.WriteLine(String.Format("{0} - Exception: {1}", DateTime.Now.ToString(), e.Message));
                        }
                        catch
                        {
                            // Do something!
                        }
                    }
                    finally
                    {
                        try
                        {
                            Debug.WriteLine(String.Format("{0} - Resetting Timer.", DateTime.Now.ToString()));
                            int intervalInMinutes = Settings.Instance.IhiCleanupProcessMinutes;
                            if (intervalInMinutes == 0)
                            {
                                timerState.timer.Change(Timeout.Infinite, Timeout.Infinite);
                                throw new Exception("Interval not configured");
                            }
                            int intervalInMilliseconds = (int)new TimeSpan(0, intervalInMinutes, 0).TotalMilliseconds;
                            timerState.UpdateTimerFrequency(intervalInMilliseconds, true);

                            Debug.WriteLine(String.Format("{0} - Timer Reset to {1} minutes.", DateTime.Now, intervalInMinutes));
                        }
                        catch (Exception exception)
                        {
                            try
                            {
                                Debug.WriteLine(String.Format("{0} - Resetting Timer, Exception: {1}", DateTime.Now.ToString(), exception.Message));
                                Debug.WriteLine(String.Format("{0} - Timer Stopped. Could not Restart.", DateTime.Now.ToString()));

                                EventLogger.WriteLog("(ERROR) Resetting Timer Exception", exception, ihiCleanupUser, LogMessage.HIPS_MESSAGE_118);
                                EventLogger.WriteLog("(ERROR) Timer Stopped. Could not Restart.", exception, ihiCleanupUser, LogMessage.HIPS_MESSAGE_119);
                            }
                            catch
                            {
                                // Do something!
                                _isThreadRunning = false;
                            }
                        }

                        _isThreadRunning = false;
                    }
                }
            }
        }

        /// <summary>
        /// Creates a new timer which launches a new thread for each tick.
        /// </summary>
        private void RunThread()
        {
            if (_timer == null)
            {
                EventLogger.WriteLog(String.Format("(INFO) Background Thread timer interval set every {0} minutes", Settings.Instance.IhiCleanupProcessMinutes), null, ihiCleanupUser, LogMessage.HIPS_MESSAGE_117);
                IhiCleanUpCallback += new TimerCallback(IhiCleanUpServiceHost_IhiCleanUpCallback);
                _state = new TimerState();

                //start after 5 seconds then will follow intervalInMilliseconds configuration item
                _timer = new Timer(IhiCleanUpCallback, _state, 0, 5000);
                _state.SetTimerReference(_timer);
                _state.isInitialised = true;
            }
        }
    }
}