﻿using System;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.CommonBusinessLogic.Consent
{
    /// <summary>
    /// Business logic to record when a patient withdraws or reinstates
    /// consent to upload clinical documents to the PCEHR system.
    /// </summary>
    public class RecordUploadConsent
    {
        /// <summary>
        /// Record the withdrawal of consent to upload a discharge summary for a particular episode.
        /// In order to check or record the consent to upload the discharge summary to PCEHR for an episode,
        /// the clinical system must specify the admission date/time of the episode within a configured tolerance.
        /// If there is no unique match to an episode then an error will be returned and the situation
        /// may be investigated by the system administrator.
        /// </summary>
        /// <param name="consentWithdrawn">True if the patient withdrew consent to upload, false if the patient reinstated consent to upload.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The date and time when the episode started (Episode.AdmissionDate)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="auditInformation">Additional audit information to store</param>
        /// <returns>Indicator of any error that occurred</returns>
        public HipsResponse RecordConsent(bool consentWithdrawn, PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user, byte[] auditInformation)
        {
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                return status;
            }

            // When recording consent, it's OK if the patient has an invalid IHI or an unresolved IHI alert.
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi && status.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                return status;
            }
            if (!User.PopulateAndValidateUser(hospital, user))
            {
                return new HipsResponse(HipsResponseIndicator.InvalidUser);
            }

            // A Specific Episode that has not been cancelled must be found in this case
            Episode episode = patientAccess.GetEpisodeWithoutCancelled(patientIdentifier, admissionDate, hospitalPatient);
            if (episode == null)
            {
                return new HipsResponse(HipsResponseIndicator.InvalidEpisode);
            }

            episode.ConsentWithdrawn = consentWithdrawn;
            ConsentAudit audit = new ConsentAudit();
            audit.EpisodeId = episode.EpisodeId.Value;
            audit.ConsentWithdrawn = consentWithdrawn;
            audit.AuditInformation = auditInformation;
            ConsentAuditDl auditDataAccess = new ConsentAuditDl(user);
            if (auditDataAccess.Insert(audit) && patientAccess.EpisodeDataAccess.Update(episode, transaction: null))
            {
                return new HipsResponse(HipsResponseIndicator.OK);
            }
            else
            {
                return new HipsResponse(HipsResponseIndicator.SystemError);
            }
        }
    }
}