﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Data.SqlTypes;
using System.Linq;
using HIPS.Base.Schemas.Enumerators;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.HL7.Common.Enumerators;
using HIPS.HL7.Common.Exceptions;
using HIPS.HL7.Common.DataStructure;
using HIPS.HL7.Common.Segment;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using hips = HIPS.PcehrDataStore.Schemas.Schemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;

namespace HIPS.CommonBusinessLogic.HL7
{
    public class PatientLoader
    {
        #region Private Static Fields

        private const int MINIMUM_DATE_OF_BIRTH_LENGTH = 8;

        private const int MINIMUM_DATE_OF_BIRTH_YEAR = 1753;

        // Defaults to the Home code for addresstype
        private const string UnknownAddressType = "H";

        // Placeholder values for situations when information is unknown or not supplied.
        private const int UnknownForeignKey = -1;

        private readonly static Dictionary<string, SexEnumerator> SexCodeMapping = new Dictionary<string, SexEnumerator>
        {
            { "M", SexEnumerator.Male },
            { "F", SexEnumerator.Female },
            { "A", SexEnumerator.IntersexOrIndeterminate },
            { "O", SexEnumerator.IntersexOrIndeterminate },
            { "U", SexEnumerator.NotStatedOrInadequatelyDescribed },
        };

        #endregion Private Static Fields

        #region Private Instance Fields

        private HospitalPatientDl hospitalPatientDataAccess;
        private PatientMasterDl patientMasterDataAccess;
        private PatientMasterIhiDl ihiDataAccess;
        private SqlTransaction transaction;
        private UserDetails user;

        #endregion Private Instance Fields

        public PatientLoader(UserDetails user, SqlTransaction transaction)
        {
            this.transaction = transaction;
            this.user = user;
            hospitalPatientDataAccess = new HospitalPatientDl(user);
            patientMasterDataAccess = new HIPS.PcehrDataStore.DataAccess.PatientMasterDl(user);
            ihiDataAccess = new PatientMasterIhiDl(user);
        }

        /// <summary>
        /// Gets the first identifier from the list that matches the given identifier.
        /// If there is no matching identifier, and a fallback item is given, then use the given fallback item.
        /// If an identifier is found and the type is a Medical Record Number ("MR") then apply the standard
        /// padding.
        /// </summary>
        /// <param name="identifierList">An array of identifiers such as PID-3 Patient Identifier List</param>
        /// <param name="type">The type of identifier that is required</param>
        /// <param name="fallback">The identifier to use if not found in the list</param>
        /// <returns>The matching identifier, or null if no matches</returns>
        public static CX IdentifierWithType(CX[] identifierList, IdentifierTypeCode type, CX fallback = null)
        {
            CX id = identifierList.Where(a => type.ToString() == a.identifiertypecode).FirstOrDefault();
            if (id == null)
            {
                id = fallback;
            }
            if (id != null && type == IdentifierTypeCode.MR)
            {
                id.ID = MrnPadding.Pad(id.ID);
            }
            return id;
        }

        /// <summary>
        /// Gets the first patient name from the list that matches the given name type code.
        /// If there is no matching name type, and a fallback item is given, then use the given fallback item.
        /// </summary>
        /// <param name="patientNameList">An array of patinet Names such as PID-5 Patient Names List</param>
        /// <param name="type">The type of name that is required</param>
        /// <param name="fallback">The name to use if not found in the list</param>
        /// <returns>The matching identifier, or null if no matches</returns>
        public static XPN PatientNameWithType(XPN[] patientNameList, NameTypeCode type, XPN fallback = null)
        {
            XPN name = patientNameList.Where(a => type.ToString() == a.nametypecode).FirstOrDefault();
            if (name == null)
            {
                name = fallback;
            }

            return name;
        }

        /// <summary>
        /// Populates the patient demographic information from the HL7 message into the patient record.
        /// </summary>
        /// <param name="pid">The PID segment of the HL7 message.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="oldPatientMaster">Returns a clone of the existing patient master, for checking what has changed.</param>
        public void Populate(PID pid, Hospital hospital, HospitalPatient hospitalPatient, PatientMaster patientMaster, out PatientMaster oldPatientMaster)
        {
            if (patientMaster == null)
            {
                patientMaster = new PatientMaster();
            }
            oldPatientMaster = CloneKeyIhiCriteria(patientMaster);

            // PID-3 Patient Identifier List: Medicare Card Number and DVA File Number
            CX medicareCard = IdentifierWithType(pid.PatientIdentifierList, IdentifierTypeCode.MC);
            string medicareCardNumber = medicareCard == null ? null : medicareCard.ID;
            string medicareIrn = null;
            if (medicareCardNumber != null && medicareCardNumber.Length == 11)
            {
                medicareIrn = medicareCardNumber.Substring(10, 1);
                medicareCardNumber = medicareCardNumber.Substring(0, 10);
            }
            CX dvaFile = IdentifierWithType(pid.PatientIdentifierList, IdentifierTypeCode.DVA);
            string dvaFileNumber = dvaFile == null ? null : dvaFile.ID;

            // PID-7 Patient Date of Birth
            DateTime? dateOfBirth = null;
            try
            {
                dateOfBirth = pid.DateTimeOfBirth.TimestampValue;
            }
            catch (Exception)
            {
                string errorMessage = string.Format(ConstantsResource.InvalidDateOfBirth, pid.DateTimeOfBirth.timeofanevent);
                throw new HL7MessageInfoException(errorMessage);
            }
            if (string.IsNullOrEmpty(pid.DateTimeOfBirth.timeofanevent))
            {
                throw new HL7MessageInfoException(ConstantsResource.NoDateOfBirthForPatient);
            }
            if (pid.DateTimeOfBirth.timeofanevent.Length < MINIMUM_DATE_OF_BIRTH_LENGTH)
            {
                throw new HL7MessageInfoException(ConstantsResource.InsufficientPrecisionInDateOfBirth);
            }
            if (dateOfBirth.HasValue && dateOfBirth.Value.Year < MINIMUM_DATE_OF_BIRTH_YEAR)
            {
                string errorMessage = string.Format(ConstantsResource.DateOfBirthOutOfRange, dateOfBirth);
                throw new HL7MessageInfoException(errorMessage);
            }

            // PID-8 Sex: Enterprise Standard Table Gender: M, F, (A, O), U transformed to M, F, I, U
            SexEnumerator sexEnum = SexEnumerator.NotStatedOrInadequatelyDescribed;
            int? sexId = null;
            if (SexCodeMapping.TryGetValue(pid.Sex.identifier, out sexEnum))
            {
                sexId = (int)sexEnum;
            }

            // Store the key demographic information into the patient master record
            patientMaster.MedicareNumber = medicareCardNumber;
            patientMaster.MedicareIrn = medicareIrn;

            if (oldPatientMaster == null || oldPatientMaster.MedicareNumber != patientMaster.MedicareNumber || !patientMaster.IsMedicareNumberValid.HasValue)
            {
                patientMaster.IsMedicareNumberValid = Medicare.Validate(patientMaster.MedicareNumber);
            }

            patientMaster.DvaNumber = dvaFileNumber;

            if (dateOfBirth.HasValue)
            {
                patientMaster.DateOfBirth = dateOfBirth.Value;
            }

            // If this is a new patient master, the Registered DOB must be set for the IHI logic.
            if (!patientMaster.PatientMasterId.HasValue)
            {
                patientMaster.RegisteredDateOfBirth = patientMaster.DateOfBirth;
            }

            patientMaster.CurrentSexId = sexId.HasValue ? sexId.Value : UnknownForeignKey;

            // Store the other information into the patient master record
            PopulateAddress(pid, patientMaster);
            PopulateSecondaryDemographics(pid, patientMaster, user);
            PopulateCommunicationMethods(pid, patientMaster);
            PopulateStatePatientIdentifier(pid, patientMaster, hospitalPatient, hospital, user);

            SavePatient(patientMaster, hospitalPatient);
        }

        /// <summary>
        /// Verifies that an existing patient has the same last name and date of birth as has been specified in
        /// the incoming message. Throws an HL7LoaderException if either are different.
        /// </summary>
        /// <param name="pid">PID segment of HL7 message</param>
        /// <param name="patientMaster">Existing patient master record</param>
        public void Verify(PID pid, PatientMaster patientMaster)
        {
            string message;
            string receivedLastName = pid.PatientName[0].familylastname.familyname;

            if (patientMaster.CurrentName.FamilyName.ToUpper() == receivedLastName.ToUpper())
            {
                // The last names match OK, check the dates of birth, ignoring any time component
                DateTime receivedDate = pid.DateTimeOfBirth.TimestampValue.Value.Date;
                DateTime currentDate = patientMaster.DateOfBirth.Date;

                if (receivedDate != currentDate)
                {
                    message = string.Format(ConstantsResource.DateOfBirthMismatchError, receivedDate, currentDate);
                    throw new HL7MessageErrorException(message);
                }
            }
            else
            {
                message = string.Format(ConstantsResource.LastNameMismatchError, receivedLastName, patientMaster.CurrentName.FamilyName);
                throw new HL7MessageErrorException(message);
            }
        }

        /// <summary>
        /// This method finds or creates a patient master, hospital 
        /// and hospital patient object using the primary identifier of the report upload request.
        /// If a ValidatedIhi is provided the PatientMasterIHI details will be updated and the PatientMaster reloaded using the IHI
        /// </summary>
        /// <param name="patientIdentification">The HL7 PID segment.</param>
        /// <param name="primaryId">The Mrn.</param>
        /// <param name="demographic">The demographic.</param>
        /// <param name="ihi">The ValidatedIhi.</param>
        /// <param name="patientMaster">Out. The patient master.</param>
        /// <param name="hospital">Out. The hospital.</param>
        /// <param name="hospitalPatient">Out.The hospital patient.</param>
        /// <param name="oldPatientMaster">Out.The existing PatientMaster record (null if a new record was created).</param>
        /// <param name="isExistingPatientMaster">Out. True if there is an existing PatientMaster record.</param>
        /// <param name="isExistingHospitalPatient">Out. True if there is an existing HospitalPatient record.</param>
        public void FindOrCreatePatient(PID patientIdentification, Mrn primaryId, Demographic demographics, ValidatedIhi validatedIhi, out PatientMaster patientMaster, out Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster oldPatientMaster, out bool isExistingPatientMaster, out bool isExistingHospitalPatient)
        {
            
            oldPatientMaster = null;
            isExistingPatientMaster = false;
            isExistingHospitalPatient = false;

            // Get the Hospital
            HospitalDl hospitalDataAccess = new HospitalDl();
            HospitalCodeDl hospitalCodeDl = new HospitalCodeDl();
            HospitalCode hospitalCode = hospitalCodeDl.GetAll(null).Where(i => i.Code == primaryId.HospitalCode).FirstOrDefault();
            if (hospitalCode == null)
            {
                throw new HL7MessageErrorException(ResponseStrings.HospitalNotFound);
            }
            hospital = hospitalDataAccess.Get(hospitalCode.CodeSystemId, hospitalCode.Code);
            if (hospital == null)
            {
                throw new HL7MessageErrorException(ResponseStrings.HospitalNotFound);
            }
            // Search the hips.HospitalPatient table to find a match for the primary identifier.
            hospitalPatient = hospitalPatientDataAccess.GetByMrn(primaryId.Value.ToString());

            // If a match was found set the patient master, hospital and hospital patient to the matched details.
            if (hospitalPatient != null && hospitalPatient.PatientId.HasValue)
            {
                isExistingPatientMaster = true;
                isExistingHospitalPatient = true;

                // Get the Patient Master
                patientMasterDataAccess.GetByHospitalIdMrn(hospitalPatient.HospitalId, hospitalPatient.Mrn, out patientMaster);
                // Set the old patientMaster object
                oldPatientMaster = CloneKeyIhiCriteria(patientMaster);
                
                // Create the new one with the demographics passed in (this will then check for a demographic update)
                patientMaster.SetNewCurrentName(-1, demographics.GivenName, demographics.FamilyName, -1);
                patientMaster.MedicareNumber = demographics.MedicareNumber;
                patientMaster.MedicareIrn = demographics.MedicareIrn;
                patientMaster.IsMedicareNumberValid = Medicare.Validate(patientMaster.MedicareNumber);
                patientMaster.DvaNumber = demographics.DvaNumber;
                patientMaster.DateOfBirth = demographics.DateOfBirth;
                patientMaster.CurrentSexId = (int)demographics.Sex;
                // Populate the Address and Communication Methods from the PID if one was passed in
                if (patientIdentification != null)
                {
                    PopulateAddress(patientIdentification, patientMaster);
                    PopulateCommunicationMethods(patientIdentification, patientMaster);
                }
                
            }
            // If no match was found:
            else
            {
                // Create a new patient master using the information provided in the demographics.
                patientMaster = new PatientMaster();
                patientMaster.SetNewCurrentName(-1, demographics.GivenName, demographics.FamilyName, -1);
                patientMaster.MedicareNumber = demographics.MedicareNumber;
                patientMaster.MedicareIrn = demographics.MedicareIrn;
                patientMaster.IsMedicareNumberValid = Medicare.Validate(patientMaster.MedicareNumber);
                patientMaster.DvaNumber = demographics.DvaNumber;
                patientMaster.DateOfBirth = demographics.DateOfBirth;
                patientMaster.RegisteredDateOfBirth = demographics.DateOfBirth;
                patientMaster.CurrentSexId = (int)demographics.Sex;
                // Populate the Address and Communication Methods from the PID if one was passed in
                if (patientIdentification != null)
                {
                    PopulateAddress(patientIdentification, patientMaster);
                    PopulateCommunicationMethods(patientIdentification, patientMaster);
                }

                // Create a new hospital patient using the details in the primary identifier.
                hospitalPatient = new HospitalPatient();
                hospitalPatient.Mrn = primaryId.Value;
                hospitalPatient.HospitalId = hospital.HospitalId.GetValueOrDefault();

            }
            // Insert the new patient or update the existing patient details
            SavePatient(patientMaster, hospitalPatient);

            // Update IHI and reload using updated IHI
            PatientMasterIhi patientMasterIhi = new PatientMasterIhi();
            // If a validated IHI was provided then add or replace the registered patient details with the information provided in the validated IHI for the current hospital organisation network.
            if (validatedIhi != null && !string.IsNullOrEmpty(validatedIhi.Ihi))
            {
                // Populate with the validatedIhi values ready for insert or update
                patientMasterIhi.Ihi = validatedIhi.Ihi;
                patientMasterIhi.DateLastValidated = validatedIhi.IhiLastValidated;
                patientMasterIhi.IhiStatusId = (int)validatedIhi.IhiStatus;
                patientMasterIhi.IhiRecordStatusId = (int)validatedIhi.IhiRecordStatus;
                patientMasterIhi.RegisteredFamilyName = validatedIhi.FamilyName;
                patientMasterIhi.RegisteredGivenName = validatedIhi.GivenName;
                patientMasterIhi.RegisteredSexId = (int)validatedIhi.Sex;
                patientMasterIhi.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;
                patientMasterIhi.DateOfBirth = validatedIhi.DateOfBirth;
                if (Settings.Instance.RegisteredDateOfBirthEnabled)
                {
                    patientMasterIhi.RegisteredDateOfBirth = validatedIhi.DateOfBirth;
                }
                // Get the new PatientMasterId to store PatientMasterIhi
                patientMasterIhi.PatientMasterId = patientMaster.PatientMasterId.Value;

                // Insert or Update the patient master ihi
                if (!ihiDataAccess.Update(patientMasterIhi, transaction))
                {
                    string message = string.Format(ResponseStrings.DatabaseError, ihiDataAccess.GetType().FullName);
                    throw new HL7MessageErrorException(message);
                }
                // Assign the IHI details to the Patient Master Record
                patientMaster.Ihi = patientMasterIhi.Ihi;
                patientMaster.IhiLastValidated = patientMasterIhi.DateLastValidated;
                patientMaster.IhiRecordStatusId = patientMasterIhi.IhiRecordStatusId;
                patientMaster.IhiStatusId = patientMasterIhi.IhiStatusId;
            }

        }
      
        /// <summary>
        /// Populates the name.
        /// </summary>
        /// <param name="pid">The pid.</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateAddress(PID pid, PatientMaster patientMaster)
        {
            ListSingleton lists = ListSingleton.Instance;
            List<hips.Address> updatedAddresses = new List<hips.Address>();

            // PID-11 Patient Address
            IEnumerable<XAD> xads = new XAD[0];
            if (pid.PatientAddress != null)
            {
                xads = xads.Concat(pid.PatientAddress);
            }

            //patientMaster.Addresses = new List<PcehrDataStore.Schemas.Schemas.Address>();
            foreach (XAD xad in xads)
            {
                string addressLine1 = xad.streetaddress;
                string addressLine2 = xad.otherdesignation;
                string addressLocality = xad.city;
                string addressStateCd = xad.stateorprovince;
                string addressPostcode = xad.ziporpostalcode;
                string addressTypeCode = xad.addresstype ?? UnknownAddressType;
                string addressCountry = xad.country ?? string.Empty;

                AddressType addressType = lists.AllAddressTypes.FirstOrDefault(a => a.Code == addressTypeCode);
                if (addressType == null)
                {
                    addressType = lists.AllAddressTypes.FirstOrDefault(c => c.Code == ConstantsResource.DefaultAddressTypeCode);
                }
                State addressState = lists.AllStates.FirstOrDefault(a => a.Code == addressStateCd);

                // Look for country by code.
                Country country = lists.AllCountries.FirstOrDefault(c => c.Code == addressCountry);
                if (country == null)
                {
                    // Look for country by name, case insensitive.
                    country = lists.AllCountries.FirstOrDefault(c => c.Description.ToUpper() == addressCountry.ToUpper());
                }
                if (country == null)
                {
                    // Default to Australia (code 1101).
                    country = lists.AllCountries.FirstOrDefault(c => c.Code == ConstantsResource.DefaultCountryCode);
                }

                hips.Address patientAddress = new hips.Address();
                patientAddress.AddressLine1 = addressLine1;
                patientAddress.AddressLine2 = addressLine2;
                patientAddress.PlaceName = addressLocality;
                patientAddress.PostCode = addressPostcode;
                patientAddress.AustralianStateId = (addressState != null) ? addressState.Id : UnknownForeignKey;
                patientAddress.CountryId = country.CountryId ?? UnknownForeignKey;
                patientAddress.AddressTypeId = addressType == null ? UnknownForeignKey : addressType.Id.Value;
                patientAddress.AddressTypeDescription = addressType == null ? null : addressType.Description;

                hips.Address existingAddress = patientMaster.Addresses.Find(result => result.CompareKey == patientAddress.CompareKey);
                if (existingAddress == null)
                {
                    updatedAddresses.Add(patientAddress);
                }
                else
                {
                    //Force address to be modified, so we can delete the missing addresses
                    updatedAddresses.Add(existingAddress);
                }
            }
            patientMaster.Addresses = updatedAddresses;
        }

        /// <summary>
        /// Populates the communication methods.
        /// </summary>
        /// <param name="pid">The pid.</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateCommunicationMethods(PID pid, PatientMaster patientMaster)
        {
            // PID-13 Home Phone Number and PID-14 Business Phone Number
            IEnumerable<XTN> xtns = new XTN[0];
            if (pid.PhoneNumberHome != null) xtns = xtns.Concat(pid.PhoneNumberHome);
            if (pid.PhoneNumberBusiness != null) xtns = xtns.Concat(pid.PhoneNumberBusiness);
            List<Contact> updatedContacts = new List<Contact>();
            bool haveHomePhone = false;
            foreach (XTN xtn in xtns)
            {
                // Use all of the available fields just in case source system populates them
                string phoneNumber = string.Format(ConstantsResource.ContactDetailsFormat, xtn.phonenumber1, xtn.CountryCode, xtn.Areacitycode, xtn.Phonenumber, xtn.Extension, xtn.anytext).Trim();
                Contact contact = null;
                if (TelecommunicationEquipmentType.CellularPhone == xtn.telecommunicationequipmentty)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.PersonalMobile, null);
                }
                else if (TelecommunicationUseCode.NetworkAddress == xtn.telecommunicationusecode)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.PersonalEmail, null);
                }
                else if (TelecommunicationUseCode.PrimaryResidenceNumber == xtn.telecommunicationusecode)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.HomePhone, null);
                    haveHomePhone = true;
                }
                else if (TelecommunicationUseCode.WorkNumber == xtn.telecommunicationusecode)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.WorkPhone, null);
                }
                else if (!haveHomePhone)
                {
                    contact = new Contact(phoneNumber, (int)ContactMethods.HomePhone, null);
                }
                if (contact == null)
                {
                    continue;
                }
                Contact existingContact = patientMaster.Contacts.Find(result => result.CompareKey == contact.CompareKey);
                if (existingContact == null)
                {
                    updatedContacts.Add(contact);
                }
                else
                {
                    updatedContacts.Add(existingContact);
                }
            }
            patientMaster.Contacts = updatedContacts;
        }

        /// <summary>
        /// Populates the secondary demographics.
        /// </summary>
        /// <param name="pid">The pid.</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateSecondaryDemographics(PID pid, PatientMaster patientMaster, UserDetails user)
        {
            ListSingleton lists = ListSingleton.Instance;
            bool current = true;
            foreach (XPN patientName in pid.PatientName)
            {
                PatientMasterName updatedName = new PatientMasterName();
                updatedName.FamilyName = patientName.familylastname.familyname;
                updatedName.GivenNames = string.Format("{0} {1}", patientName.givenname.Trim(), patientName.middleinitialorname.Trim()).Trim();

                string titleCode = patientName.prefix;
                Title title = null;
                if (!string.IsNullOrWhiteSpace(titleCode))
                {
                    title = lists.AllTitles.Where(a => a.Code.Equals(titleCode)).FirstOrDefault();
                    if (title == null)
                    {
                        TitleDl titleDl = new TitleDl();
                        title = new Title();
                        title.Code = titleCode;
                        title.Description = titleCode;
                        titleDl.Insert(title);
                    }
                }
                string suffixCode = patientName.suffix;
                Suffix suffix = null;
                if (!string.IsNullOrWhiteSpace(suffixCode))
                {
                    suffix = lists.AllSuffixes.Where(a => a.Code.Equals(suffixCode)).FirstOrDefault();
                    if (suffix == null)
                    {
                        SuffixDl suffixDl = new SuffixDl();
                        suffix = new Suffix();
                        suffix.Code = suffixCode;
                        suffix.Description = suffixCode;
                        suffixDl.Insert(suffix);
                    }
                }
                updatedName.TitleId = (title == null ? UnknownForeignKey : title.Id.Value);
                updatedName.SuffixId = (suffix == null ? UnknownForeignKey : suffix.Id.Value);

                if (current)
                {
                    patientMaster.SetNewCurrentName(updatedName.TitleId, updatedName.GivenNames, updatedName.FamilyName, updatedName.SuffixId);
                }
            }

            // PID-29 Patient Date of Death
            DateTime? dateOfDeath = pid.PatientDeathDateandTime == null ? null : pid.PatientDeathDateandTime.TimestampValue;

            // The SQL Server date is a datetime and thus can't accept dates before 1/1/1753.
            if (dateOfDeath != null)
            {
                // The SQL Server date is a datetime and thus can't accept dates before SqlDateTime.MinValue (1/1/1753). or after SqlDateTime.MinValue (31/12/9999)
                if (dateOfDeath < SqlDateTime.MinValue.Value || dateOfDeath > SqlDateTime.MaxValue.Value)
                {
                    //if date is too small then set date to minimum SQL Date and flag indicator as invalid date
                    dateOfDeath = SqlDateTime.MinValue.Value;
                    patientMaster.DeathIndicatorId = (int)DeathIndicator.InvalidDate;

                    //extract patient MRN and Facility code for INFO message
                    CX cx = PatientLoader.IdentifierWithType(pid.PatientIdentifierList, IdentifierTypeCode.MR);
                    string patientMRN;
                    string facilityCode;
                    if (cx != null)
                    {
                        patientMRN = cx.ID;
                        facilityCode = cx.assigningauthority.namespaceID;
                    }
                    else
                    {
                        patientMRN = "'MRN not found'";
                        facilityCode = "'Facility Code not found'";
                    }

                    //log as info message only
                    string description = string.Format(ConstantsResource.InfoMessageDateOfDeathBeforeSQLDateMin, patientMRN, facilityCode);
                    System.Exception exception = new Exception(string.Format(ConstantsResource.InfoMessageDateOfDeathBeforeSQLDateMinEx, pid.PatientDeathDateandTime.TimestampValue, SqlDateTime.MinValue.Value, DeathIndicator.InvalidDate));
                    EventLogger.WriteLog(description, exception, user, LogMessage.HIPS_MESSAGE_112);
                }
                else
                {
                    patientMaster.DeathIndicatorId = (int)DeathIndicator.ValidDate;
                }
            }
            else
            {
                // If date of death is null then so is the indicator
                patientMaster.DeathIndicatorId = null;
            }

            // Set date of death to calculated or received date of death
            patientMaster.DateOfDeath = dateOfDeath;
        }

        /// <summary>
        /// Populates the state patient identifier, if it was empty.
        /// Throws an error if the state patient identifier changes.
        /// Note, changes in state patient identifier should have been
        /// handled by the PatientLinker class before reaching this point.
        /// </summary>
        /// <param name="pid">The PID segment.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="user">The user.</param>
        private static void PopulateStatePatientIdentifier(PID pid, PatientMaster patientMaster, HospitalPatient hospitalPatient, Hospital hospital, UserDetails user)
        {
            if (pid.PatientID != null && pid.PatientID.identifiertypecode == IdentifierTypeCode.SAUHI.ToString())
            {
                string statePatientId = pid.PatientID.ID;
                if (string.IsNullOrEmpty(patientMaster.StatePatientId))
                {
                    patientMaster.StatePatientId = statePatientId;
                }
                else if (patientMaster.StatePatientId != statePatientId)
                {
                    string message = string.Format(ConstantsResource.StatePatientIdentifierChange, hospitalPatient.Mrn, hospital.Description, patientMaster.StatePatientId, statePatientId);
                    throw new HL7MessageErrorException(message);
                }
            }
        }

        /// <summary>
        /// Clones the key IHI criteria.
        /// </summary>
        /// <param name="old">The existing patient master.</param>
        /// <returns>A copy of the patient master which is independent of the given one that will be updated.</returns>
        private PatientMaster CloneKeyIhiCriteria(PatientMaster old)
        {
            if (!old.PatientMasterId.HasValue)
            {
                return null;
            }
            PatientMaster clone = new PatientMaster();
            clone.Names = (from name in old.Names
                           select new PatientMasterName()
                           {
                               FamilyName = name.FamilyName,
                               GivenNames = name.GivenNames,
                               TitleId = name.TitleId,
                               SuffixId = name.SuffixId,
                               NameTypeId = name.NameTypeId
                           }).ToList();
            clone.CurrentSexId = old.CurrentSexId;
            clone.RegisteredSexId = old.RegisteredSexId;
            clone.MedicareNumber = old.MedicareNumber;
            clone.MedicareIrn = old.MedicareIrn;
            clone.DvaNumber = old.DvaNumber;
            clone.DateOfBirth = old.DateOfBirth;
            clone.RegisteredFamilyName = old.RegisteredFamilyName;
            clone.RegisteredGivenName = old.RegisteredGivenName;
            return clone;
        }

        /// <summary>
        /// Saves the patient master and then the hospital patient, ensuring the hospital patient points to the patient master.
        /// </summary>
        /// <param name="patientMaster">Patient master object</param>
        /// <param name="hospitalPatient">Hospital patient object</param>
        /// <exception cref="HL7MessageErrorException">Unable to load the patient from the PID segment</exception>
        private void SavePatient(PatientMaster patientMaster, HospitalPatient hospitalPatient)
        {
            // Save the PatientMaster object.
            if (!patientMaster.PatientMasterId.HasValue)
            {
                if (!patientMasterDataAccess.Insert(patientMaster, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientMasterDataAccess.GetType().FullName));
                }
            }
            else
            {
                if (!patientMasterDataAccess.Update(patientMaster, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientMasterDataAccess.GetType().FullName));
                }
            }

            // We need to get the PatientMasterId and store it in the HospitalPatient before saving
            // the HospitalPatient.
            hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;

            // Save the HospitalPatient object.
            if (!hospitalPatient.PatientId.HasValue)
            {
                if (!hospitalPatientDataAccess.Insert(hospitalPatient, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, hospitalPatientDataAccess.GetType().FullName));
                }
            }
            else
            {
                if (!hospitalPatientDataAccess.Update(hospitalPatient, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, hospitalPatientDataAccess.GetType().FullName));
                }
            }
        }

        /// <summary>
        /// Creates the Patient Demographic and ValidatedIHI details based on the HL7 PID segment
        /// </summary>
        /// <param name="pid">The HL7 PID segment</param>
        /// <param name="demographics">Out. The newly created demographic details</param>
        /// <param name="validatedIhi">Out. The newly created validated IHI.</param>
        internal void CreatePatientDemographicValidatedIHI(HIPS.HL7.Common.Segment.PID pid, string hospitalCode, out Demographic demographics, out ValidatedIhi validatedIhi)
        {
            // PID-3 Patient Identifier List: Medicare Card Number and DVA File Number
            CX medicareCard = IdentifierWithType(pid.PatientIdentifierList, IdentifierTypeCode.MC);
            string medicareCardNumber = medicareCard == null ? null : medicareCard.ID;
            string medicareIrn = null;
            if (medicareCardNumber != null && medicareCardNumber.Length == 11)
            {
                medicareIrn = medicareCardNumber.Substring(10, 1);
                medicareCardNumber = medicareCardNumber.Substring(0, 10);
            }

            // Check the patient identifier list for a Department of Veterans Affairs number (DVA, DVG, DVO or DVW)
            var validDvaTypeCodes = new IdentifierTypeCode[] { IdentifierTypeCode.DVA, IdentifierTypeCode.DVG, IdentifierTypeCode.DVO, IdentifierTypeCode.DVW };
            CX dvaFile = validDvaTypeCodes.Select(type => IdentifierWithType(pid.PatientIdentifierList, type)).FirstOrDefault(result => result != null);

            string dvaFileNumber = dvaFile == null ? null : dvaFile.ID;

            // PID-7 Patient Date of Birth
            DateTime? dateOfBirth = null;
            try
            {
                dateOfBirth = pid.DateTimeOfBirth.TimestampValue;
            }
            catch (Exception)
            {
                string errorMessage = string.Format(ConstantsResource.InvalidDateOfBirth, pid.DateTimeOfBirth.timeofanevent);
                throw new HL7MessageInfoException(errorMessage);
            }
            if (string.IsNullOrEmpty(pid.DateTimeOfBirth.timeofanevent))
            {
                throw new HL7MessageInfoException(ConstantsResource.NoDateOfBirthForPatient);
            }
            if (pid.DateTimeOfBirth.timeofanevent.Length < MINIMUM_DATE_OF_BIRTH_LENGTH)
            {
                throw new HL7MessageInfoException(ConstantsResource.InsufficientPrecisionInDateOfBirth);
            }
            if (dateOfBirth.HasValue && dateOfBirth.Value.Year < MINIMUM_DATE_OF_BIRTH_YEAR)
            {
                string errorMessage = string.Format(ConstantsResource.DateOfBirthOutOfRange, dateOfBirth);
                throw new HL7MessageInfoException(errorMessage);
            }

            // PID-8 Sex: Enterprise Standard Table Gender: M, F, (A, O), U transformed to M, F, I, U
            SexEnumerator sexEnum = SexEnumerator.NotStatedOrInadequatelyDescribed;
            int? sexId = null;
            if (SexCodeMapping.TryGetValue(pid.Sex.identifier, out sexEnum))
            {
                sexId = (int)sexEnum;
            }

            string familyName;
            string givenNames;
            // PID-5 PatientName: Get patient's Legal Name
            XPN patientLegalName = PatientNameWithType(pid.PatientName, NameTypeCode.L);
            if (patientLegalName == null)
            {
                throw new HL7MessageInfoException(ConstantsResource.NoLegalNameForPatient);
            }
            familyName = patientLegalName.familylastname.familyname;
            givenNames = string.Format("{0} {1}", patientLegalName.givenname.Trim(), patientLegalName.middleinitialorname.Trim()).Trim();

            // Get patient's Ihi and validated date
            DateTime? ihiLastValidated = null;
            IhiRecordStatus ihiRecordStatus;
            IhiStatus ihiStatus;
            string ihiNumber = this.GetIhiNumber(pid.PatientIdentifierList, out ihiLastValidated, out ihiRecordStatus, out ihiStatus);

            // Create a Demographic and ValidatedIhi using the data from the HL7® message.
            demographics = this.CreateDemographic(familyName, givenNames, medicareCardNumber, medicareIrn, dvaFileNumber, dateOfBirth, sexEnum, hospitalCode, HL7.ConstantsResource.PatientAdministrationSystemHospitalCodeSystem);
            if (demographics == null)
            {
                throw new HL7MessageInfoException(ConstantsResource.NoMedicareOrDvaNumberForPatient);
            }
            validatedIhi = (!string.IsNullOrEmpty(ihiNumber) ? new ValidatedIhi(ihiNumber, ihiStatus, ihiRecordStatus, ihiLastValidated.GetValueOrDefault(), familyName, givenNames, dateOfBirth.Value, sexEnum, hospitalCode, HL7.ConstantsResource.PatientAdministrationSystemHospitalCodeSystem) : null);
           
        }

        /// <summary>
        /// Create a Patient demographic.
        /// </summary>
        /// <param name="familyName">Family Name</param>
        /// <param name="givenName">Given Name</param>
        /// <param name="medicareCardNumber">Medicare Card Number</param>
        /// <param name="medicareIrn">Medicare IRN</param>
        /// <param name="dvaNumber">DVA Number</param>
        /// <param name="dateOfBirth">Date of Birth</param>
        /// <param name="sex">Patient's Sex</param>
        /// <param name="hospitalCode">Hospital Code</param>
        /// <param name="hospitalCodeSystem">Hospital Code System</param>
        /// <returns>Patient's Demographic.</returns>
        private Demographic CreateDemographic(string familyName, string givenName, string medicareCardNumber, string medicareIrn, string dvaNumber, DateTime? dateOfBirth, SexEnumerator sex, string hospitalCode, string hospitalCodeSystem)
        {
            Demographic demographic = null;

            if (!string.IsNullOrEmpty(medicareCardNumber))
            {
                demographic = new Demographic(familyName, givenName, dateOfBirth.Value, sex, medicareCardNumber, medicareIrn, hospitalCode, hospitalCodeSystem);
            }
            else if (!string.IsNullOrEmpty(dvaNumber))
            {
                demographic = new Demographic(familyName, givenName, dateOfBirth.Value, sex, dvaNumber, hospitalCode, hospitalCodeSystem);
            }

            return demographic;
        }

        /// <summary>
        /// Gets Patient's IHI from the PID-3 segment
        /// </summary>
        /// <param name="patientIdentifierList">HL7 Patient Identifier List</param>
        /// <param name="ihiLastValidated">Out. Date of IHI's last validation.</param>
        /// <param name="ihiRecordStatus">Out. The returned IHI Record Status</param>
        /// <param name="ihiStatus">Out. The returned IHI Status</param>
        /// <returns>IHI number.</returns>
        private string GetIhiNumber(CX[] patientIdentifierList, out DateTime? ihiLastValidated, out IhiRecordStatus ihiRecordStatus, out IhiStatus ihiStatus)
        {
            ihiLastValidated = null;
            string result = string.Empty;
            ihiRecordStatus = IhiRecordStatus.Unknown;
            ihiStatus = IhiStatus.Unknown;
            DateTime ihiLastValidatedMinDate = DateTime.Parse(string.Format("1/1/{0}", MINIMUM_DATE_OF_BIRTH_YEAR.ToString()));

            // Get the IHI number where the Type Code is NI
            var ihi = PatientLoader.IdentifierWithType(patientIdentifierList, IdentifierTypeCode.NI);
            // If we have an IHI and an effective date then set this as the LastValidatedDate
            if (ihi != null)
            {
                result = ihi.ID;
                // If we have received an IHI from the HL7 then trust the IHI has been validated and verified already
                ihiRecordStatus = IhiRecordStatus.Verified;
                ihiStatus = IhiStatus.Active;
                ihiLastValidated = ihiLastValidatedMinDate;
                if (ihi.EffectiveDate != null && ihi.EffectiveDate.TimestampValue.HasValue)
                    ihiLastValidated = ihi.EffectiveDate.TimestampValue.Value;
            }

            // If the BypassHIService is set then check we have an IHI and assume it is valid
            if (Settings.Instance.BypassHIService)
            {
                // FR-PDI-7.2.1	In Mode 2 (No connection to HI Service) the message must contain the IHI of the patient and the HPI-I of the document author.
                if (ihi == null || string.IsNullOrEmpty(ihi.ID))
                    throw new HL7MessageInfoException(ConstantsResource.NoIhiForPatient);

                // FR-PDI-7.2.2	In Mode 2 (No connection to HI Service) the eHISC service will trust that all IHI and HPI-I identifiers in the message have already been validated by another system.
                if (ihiLastValidated.Value == ihiLastValidatedMinDate)
                    ihiLastValidated = DateTime.Now;
            }

            return result;
        }
    }
}