﻿using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Mapping;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.HpiiSchemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace HIPS.CommonBusinessLogic.Hpii
{
    public class HpiiLocalManagement
    {
        #region Methods

        #region Public

        /// <summary>
        /// Lists all Health Provider Individuals from local data store
        /// </summary>
        /// <param name="hospitalIdentifier">The Hospital Identifier used for the hospital code system.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="localProviders">List of Health Provider Individual objects</param>
        /// <param name="localProviderIdentifiers">List of Hospital Health Provider Individual objects</param>
        /// <returns>Hips Response</returns>
        public HipsResponse ListLocalProviders(HospitalIdentifier hospitalIdentifier, UserDetails user, out List<HealthProviderIndividual> localProviders, out List<HospitalHealthProviderIndividual> localProviderIdentifiers)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            localProviders = null;
            localProviderIdentifiers = null;

            try
            {
                // Check if the provided hospital identifier is valid.
                var lists = ListSingleton.Instance;
                var hospitalCode = lists.AllHospitalCodes
                    .Where(c => c.Code == hospitalIdentifier.HospitalCode && c.CodeSystemCode == hospitalIdentifier.HospitalCodeSystem)
                    .FirstOrDefault();

                if (hospitalCode == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, hospitalIdentifier.HospitalCode, hospitalIdentifier.HospitalCodeSystem);
                }

                // 1. Return a list of local providers from the hips.HealthProviderIndividual table.
                // 2. Include HPI-I data from the hips.HealthProviderIndividualHpii if it exists.
                HealthProviderIndividualDl healthProviderIndividualDataAccess = new HealthProviderIndividualDl(user);
                localProviders = healthProviderIndividualDataAccess.GetAll();

                // 3. Include a list of LocalProviderIdentifiers from the hips.HospitalHealthProviderIndividual table 
                //    returning the hospital code for the “pasFacCd”, the hospital name and both the CIS and PAS identifiers.
                HospitalHealthProviderIndividualDl hospitalHealthProviderIndividualDataAccess = new HospitalHealthProviderIndividualDl(user);
                localProviderIdentifiers = hospitalHealthProviderIndividualDataAccess.GetAll(hospitalIdentifier.HospitalCodeSystem);

            }
            catch (ItemNotFoundException ex)
            {
                response.Status = HipsResponseIndicator.InvalidHospital;
                response.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            catch (Exception ex)
            {
                response.Status = HipsResponseIndicator.DatabaseError;
                response.HipsErrorMessage = ResponseStrings.UnableToListLocalProviders;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            
            return response;
           
        }

        /// <summary>
        /// Add or Edit a Health Provider Individual and Hpii in the Local Data Store
        /// If the Hpii has changed, or Health Provider Demographics have changed then Hpii is revalidated
        /// </summary>
        /// <param name="hospitalIdentifier">The Hospital Identifier used to look up the Hospital details</param>
        /// <param name="localProviderCode">The Health Provider Individuals LocalProviderCode</param>
        /// <param name="iamProviderIdentifier">The Health Provider Individuals LocalProviderCode</param>
        /// <param name="hpii">The Health Provider Individuals HPI-I</param>
        /// <param name="familyName">The Health Provider Individuals Family Name</param>
        /// <param name="givenNames">The Health Provider Individuals Given Names</param>
        /// <param name="suffix">The Health Provider Individuals Suffix</param>
        /// <param name="title">The Health Provider Individuals Title (prefix)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>Hips Response object</returns>
        public HipsResponse AddEditLocalProviders(HospitalIdentifier hospitalIdentifier, string localProviderCode, string iamProviderIdentifier, string hpii, 
            string familyName, string givenNames, string suffix, string title, UserDetails user)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            
            try
            {

                var lists = ListSingleton.Instance;
                var hospitalCode = lists.AllHospitalCodes.Where(c => c.Code == hospitalIdentifier.HospitalCode
                                                                  && c.CodeSystemCode == hospitalIdentifier.HospitalCodeSystem)
                                                          .FirstOrDefault();

                if (hospitalCode == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, hospitalIdentifier.HospitalCode, hospitalIdentifier.HospitalCodeSystem);
                }

                var hospitalDataAccess = new HospitalDl();
                var hospital = hospitalDataAccess.Get(hospitalCode.HospitalId);

                if (hospital == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, hospitalIdentifier.HospitalCode, hospitalCode.HospitalId.ToString());
                }

                HealthProviderIndividual healthProviderIndividual = null;
                var healthProviderIndividualDataAccess = new HealthProviderIndividualDl(user);

                if (!string.IsNullOrEmpty(localProviderCode) || !string.IsNullOrEmpty(iamProviderIdentifier))
                {
                    healthProviderIndividual = healthProviderIndividualDataAccess.Get(hospital.HospitalId.Value, localProviderCode, iamProviderIdentifier);
                }
                else if (!string.IsNullOrEmpty(hpii))
                {
                    // Check if the HPI-I number in the request already exists in the database
                    var hpiiDataAccess = new HealthProviderIndividualHpiiDl(user);
                    if (hpiiDataAccess.Get(hpii, hospital.HealthProviderOrganisationNetworkId) != null)
                    {
                        response.Status = HipsResponseIndicator.ValidationError;
                        response.HipsErrorMessage = ResponseStrings.ExistingProviderWithHPIIFound;
                    }
                }

                bool needsValidation = true;
                if (healthProviderIndividual != null)
                {
                    // Check if we can skip validation.
                    if (healthProviderIndividual.HpiI == hpii &&
                        healthProviderIndividual.FamilyName == familyName &&
                        healthProviderIndividual.GivenNames == givenNames)
                    {
                        needsValidation = false;
                    }
                }
                else
                {
                    // Create a new health provider individual and set any fields that are needed for new health providers
                    healthProviderIndividual = new HealthProviderIndividual();
                    healthProviderIndividual.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;
                    healthProviderIndividual.LocalProviderCode = localProviderCode;

                    if (string.IsNullOrEmpty(localProviderCode))
                    {
                        healthProviderIndividual.LocalProviderCode = Guid.NewGuid().ToString();
                    }
                }

                // Update the health provider individual with the data from the request
                healthProviderIndividual.FamilyName = familyName;
                healthProviderIndividual.GivenNames = givenNames;
                healthProviderIndividual.Suffix = suffix;
                healthProviderIndividual.Title = title;
                healthProviderIndividual.HpiI = hpii;
                healthProviderIndividual.IamProviderIdentifier = iamProviderIdentifier;
                this.PopulateSuffixAndTitle(healthProviderIndividual);

                if (hpii == null)
                {
                    // Clear any existing HPI-I validation and skip validation for requests without a HPI-I number
                    healthProviderIndividual.HpiiLastValidated = null;
                    healthProviderIndividual.HpiiStatusId = -1;
                    needsValidation = false;
                }

                if (needsValidation)
                {
                    response = this.ValidateHpiiAndSave(healthProviderIndividual, hospital, user);
                }
                else
                {
                    // If the request does not need HPI-I validation then reset HPII status and last validated date, and save the data directly
                    healthProviderIndividualDataAccess.Save(healthProviderIndividual);
                }
            }
            catch (ItemNotFoundException ex)
            {
                response.Status = HipsResponseIndicator.InvalidHospital;
                response.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            catch (Exception ex)
            {
                response.Status = HipsResponseIndicator.SystemError;
                response.HipsErrorMessage = ResponseStrings.ErrorUnableToAddOrEditLocalProvider;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            
            return response;
        }

        /// <summary>
        /// Add or Replace a Local Provider Identifier which maps the Hospital to the Health Provider Individual object
        /// </summary>
        /// <param name="hospitalIdentifier">The Health Facilities Identifier information</param>
        /// <param name="localProviderCode">The Health Provider Individual's Unique Local Provider Code</param>
        /// <param name="pasProviderIdentifier">The Local PAS Code used to link the Hospital to the Health Provider Individual</param>
        /// <param name="cisProviderIdentifier">The Local CIS Code used to link the Hospital to the Health Provider Individual</param>
        /// <param name="replace">True if this is to replace the Identifier or False otherwise</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>Hips Response object</returns>
        public HipsResponse AddReplaceLocalProviderIdentifier(HospitalIdentifier hospitalIdentifier, string localProviderCode, string pasProviderIdentifier, string cisProviderIdentifier, bool replace, UserDetails user)
        {

            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                var lists = ListSingleton.Instance;
                var hospitalCode = lists.AllHospitalCodes.Where(c => c.Code == hospitalIdentifier.HospitalCode
                                                                  && c.CodeSystemCode == hospitalIdentifier.HospitalCodeSystem)
                                                          .FirstOrDefault();

                if (hospitalCode == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, hospitalIdentifier.HospitalCode, hospitalIdentifier.HospitalCodeSystem);
                }

                var hospitalDataAccess = new HospitalDl();
                var hospital = hospitalDataAccess.Get(hospitalCode.HospitalId);

                if (hospital == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, hospitalIdentifier.HospitalCode, hospitalCode.HospitalId.ToString());
                }

                // Find the health provider individual corresponding to the local provider code
                // Note: The hospital id is not required because we might be inserting a new local provider which is not in any hospitals already.
                var healthProviderIndividualDataAccess = new HealthProviderIndividualDl(user);
                var healthProviderIndividual = healthProviderIndividualDataAccess.Get(null, localProviderCode, null);

                if (healthProviderIndividual == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, localProviderCode, null);
                }

                // Find any existing entries for this local provider in the current hospital.
                // This is used so we can determine if we are adding or replacing a local provider identifier.
                var hospitalHealthDataAccess = new HospitalHealthProviderIndividualDl(user);
                var hospitalHealthProviderIndividual = hospitalHealthDataAccess.Get(hospital.HospitalId.Value, healthProviderIndividual.HealthProviderIndividualId.Value);

                if (hospitalHealthProviderIndividual != null)
                {
                    if (!replace)
                    {
                        throw new LocalProviderIdentifierAlreadyExistsException();
                    }

                    hospitalHealthProviderIndividual.PasProviderIdentifier = pasProviderIdentifier;
                    hospitalHealthProviderIndividual.CisProviderIdentifier = cisProviderIdentifier;

                    hospitalHealthDataAccess.Update(hospitalHealthProviderIndividual);
                }
                else
                {
                    hospitalHealthProviderIndividual = new HospitalHealthProviderIndividual();
                    hospitalHealthProviderIndividual.PasProviderIdentifier = pasProviderIdentifier;
                    hospitalHealthProviderIndividual.CisProviderIdentifier = cisProviderIdentifier;
                    hospitalHealthProviderIndividual.HospitalId = hospital.HospitalId;
                    hospitalHealthProviderIndividual.HealthProviderIndividualId = healthProviderIndividual.HealthProviderIndividualId;

                    hospitalHealthDataAccess.Insert(hospitalHealthProviderIndividual);
                }
            }
            catch (ItemNotFoundException ex)
            {
                response.Status = HipsResponseIndicator.InvalidHospital;
                response.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            catch (Exception ex)
            {
                response.Status = HipsResponseIndicator.SystemError;
                response.HipsErrorMessage = ResponseStrings.ErrorUnableToAddOrEditLocalProvider;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            
            return response;
        }

        /// <summary>
        /// Sets the deactivated for the Health Provider Individual in the Local Data Store
        /// This does not call the HI Service 
        /// </summary>
        /// <param name="hospitalIdentifier">The Hospital Identifie=ying information</param>
        /// <param name="localProviderCode">The Unique Local Provider Code</param>
        /// <param name="deactivatedDate">The Date the deactivation should start</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>Hips Response Object</returns>
        public HipsResponse DeactivateLocalProvider(HospitalIdentifier hospitalIdentifier, string localProviderCode, DateTime? deactivatedDate, UserDetails user)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                var lists = ListSingleton.Instance;
                var hospitalCode = lists.AllHospitalCodes.Where(c => c.Code == hospitalIdentifier.HospitalCode
                                                                  && c.CodeSystemCode == hospitalIdentifier.HospitalCodeSystem)
                                                          .FirstOrDefault();

                if (hospitalCode == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, hospitalIdentifier.HospitalCode, hospitalIdentifier.HospitalCodeSystem);
                }

                var hospitalDataAccess = new HospitalDl();
                var hospital = hospitalDataAccess.Get(hospitalCode.HospitalId);

                if (hospital == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, hospitalIdentifier.HospitalCode, hospitalCode.HospitalId.ToString());
                }
                var healthProviderIndividualDataAccess = new HealthProviderIndividualDl(user);

                var healthProviderIndividual = healthProviderIndividualDataAccess.Get(hospital.HospitalId.Value, localProviderCode, null);
                
                if (healthProviderIndividual == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital, localProviderCode, null);
                }
                healthProviderIndividual.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;
                healthProviderIndividual.DeactivatedDate = deactivatedDate;
                if (!healthProviderIndividualDataAccess.Save(healthProviderIndividual))
                {
                    response.Status = HipsResponseIndicator.DatabaseError;
                    response.HipsErrorMessage = ResponseStrings.ErrorUnableToDeactivateLocalProvider;
                }
            
            }
            catch (ItemNotFoundException ex)
            {
                response.Status = HipsResponseIndicator.InvalidHospital;
                response.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            catch (Exception ex)
            {
                response.Status = HipsResponseIndicator.SystemError;
                response.HipsErrorMessage = ResponseStrings.ErrorUnableToAddOrEditLocalProvider;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            
            return response;
        }
       
        #endregion

        #region Private
        /// <summary>
        /// Validate Hpii. If the validation is successful, this will update the health provider individual.
        /// </summary>
        /// <param name="healthProviderIndividual">Health Provider Individual.</param>
        /// <param name="facility">A Facility.</param>
        /// <param name="user">User Context.</param>
        /// <returns>Hips Response object</returns>
        private HipsResponse ValidateHpiiAndSave(HealthProviderIndividual healthProviderIndividual, Hospital facility, UserDetails user)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            
            var patientAccess = new PatientAccess(user);
            var query = new HpiiIdentifierQuery()
            {
                HpiiNumber = healthProviderIndividual.HpiI,
                FamilyName = healthProviderIndividual.FamilyName,
                GivenName = healthProviderIndividual.GivenNames != null ? healthProviderIndividual.GivenNames.Split(' ') : null
            };

            HpiiQueryResponse result = patientAccess.ValidateHpii(query, healthProviderIndividual, facility);

            // Return response
            return result.HipsResponse;
        }

        /// <summary>
        /// Populate the Suffix and title of Health Provider Individual
        /// </summary>
        /// <param name="healthProviderIndividual">The Health Provider Individual.</param>
        private void PopulateSuffixAndTitle(HealthProviderIndividual healthProviderIndividual)
        {
            var lists = ListSingleton.Instance;
            var suffix = lists.AllSuffixes.Where(s => s.Code.Equals(healthProviderIndividual.Suffix)).FirstOrDefault();
            var title = lists.AllTitles.Where(s => s.Code.Equals(healthProviderIndividual.Title)).FirstOrDefault();

            healthProviderIndividual.SuffixId = suffix == null ? -1 : suffix.Id.Value;
            healthProviderIndividual.Suffix = null;

            healthProviderIndividual.TitleId = title == null ? -1 : title.Id.Value;
            healthProviderIndividual.Title = null;
        }

        #endregion

        #endregion
    }
}
