﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Xml;
using HIPS.Configuration;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using HIPS.PcehrSchemas.Exceptions;
using Nehta.VendorLibrary.CDAPackage;
using Nehta.VendorLibrary.Common;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class CdaPackaging
    {
        #region Constants

        private const int PCEHR_LOGO_HEIGHT = 100;

        /// <summary>
        /// CDA Rendering Specification makes it a requirement to validate the logo size does not exceed 400 pixels in width or 100 pixels in height.
        /// </summary>
        private const int PCEHR_LOGO_WIDTH = 400;

        #endregion Constants

        #region Private Fields

        private XmlDocument cdaDocument;
        private byte[] cdaDocumentBytes;
        private XmlNamespaceManager cdaNamespaces;

        #endregion Private Fields

        #region Constructors

        public CdaPackaging(byte[] cdaDocumentBytes)
        {
            this.cdaDocumentBytes = cdaDocumentBytes;
            this.cdaDocument = new XmlDocument();
            using (MemoryStream ms = new MemoryStream(cdaDocumentBytes))
            {
                this.cdaDocument.Load(ms);
            }
            this.cdaNamespaces = new XmlNamespaceManager(new NameTable());
            this.cdaNamespaces.AddNamespace(XmlStringMap.HL7V3NamespaceAbbreviation, XmlStringMap.HL7V3Namespace);
            this.cdaNamespaces.AddNamespace(XmlStringMap.AustralianCdaExtensionsV3NamespaceAbbreviation, XmlStringMap.AustralianCdaExtensionsV3Namespace);
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Generates the CDA Package in memory.
        /// </summary>
        /// <param name="hospital">The hospital.</param>
        /// <param name="attachments">The attachments.</param>
        /// <returns>The CDA package.</returns>
        internal byte[] GenerateCdaPackageInMemory(Hospital hospital, Attachment[] attachmentArray)
        {
            X509Certificate2 signingCert = Helpers.GetSigningCertificate(hospital);
            Approver approver = GetApprover(GetAuthor());
            var package = new CDAPackage(approver);
            List<Attachment> attachments = (attachmentArray ?? new Attachment[0]).ToList();
            ProcessLogo(hospital, attachments);

            // If attachments were supplied by the calling application, add them to the package.
            foreach (Attachment a in attachments)
            {
                package.AddDocumentAttachment(
                    a.FileName,
                    a.Contents
                );
            }

            // Create the CDA root document and create the CDA package.
            package.CreateRootDocument(this.cdaDocumentBytes);
            return CDAPackageUtility.Create(package, signingCert);
        }

        /// <summary>
        /// Gets the author of the CDA document.
        /// </summary>
        /// <returns>The author of the CDA Document</returns>
        internal Participant GetAuthor()
        {
            Participant author = new Participant();
            author.GivenNames = new List<string>();
            author.Titles = new List<string>();
            author.Suffixes = new List<string>();
            author.Identifiers = new List<EntityIdentifier>();
            try
            {
                XmlNode node = cdaDocument.SelectSingleNode(XmlStringMap.AuthorPersonNameXPath, cdaNamespaces);
                for (int i = 0; i < node.ChildNodes.Count; i++)
                {
                    if (node.ChildNodes[i].Name.ToUpper() == XmlStringMap.CdaFamilyName)
                    {
                        author.FamilyName = node.ChildNodes[i].InnerText;
                    }
                    else if (node.ChildNodes[i].Name.ToUpper() == XmlStringMap.CdaGivenName)
                    {
                        author.GivenNames.Add(node.ChildNodes[i].InnerText);
                    }
                    else if (node.ChildNodes[i].Name.ToUpper() == XmlStringMap.CdaNamePrefix)
                    {
                        author.Titles.Add(node.ChildNodes[i].InnerText);
                    }
                    else if (node.ChildNodes[i].Name.ToUpper() == XmlStringMap.CdaNameSuffix)
                    {
                        author.Suffixes.Add(node.ChildNodes[i].InnerText);
                    }
                }

                //get HPI-I
                XmlNodeList authorIdNodes = cdaDocument.SelectNodes(XmlStringMap.AuthorPersonIdentifierXPath, cdaNamespaces);
                foreach (XmlNode authorIdNode in authorIdNodes)
                {
                    EntityIdentifier id = new EntityIdentifier();
                    author.Identifiers.Add(id);
                    XmlAttribute assigningAuthorityName = authorIdNode.Attributes[XmlStringMap.AssigningAuthorityNameAttributeName];
                    if (assigningAuthorityName != null)
                    {
                        id.AssigningAuthorityName = assigningAuthorityName.Value;
                    }
                    id.Root = authorIdNode.Attributes[XmlStringMap.RootAttributeName].Value;
                    if (id.Root.StartsWith(XmlStringMap.NationalHealthcareIdentifierOidPrefix))
                    {
                        // The root contains an HPII, which should be qualified by "http://ns.electronichealth.net.au/id/hi/hpii/1.0/".
                        id.QualifiedIdentifier = new Uri(string.Format("{0}{1}",
                            HIQualifiers.HPIIQualifier,
                            id.Root.Substring(XmlStringMap.NationalHealthcareIdentifierOidPrefix.Length)));
                    }
                    else
                    {
                        // The extension contains the local ID of the author within the organisation, and should be qualified using the domain namespace of the organisation where the ID came from.
                        // See FAQ_Implementation_clarification_for_relaxation_of_HPII_in_DS_rev001.pdf
                        // For example http://ns.health.domain.au/id/cda/userid/1.0/jsmith01
                        string extension = authorIdNode.Attributes[XmlStringMap.ExtensionAttributeName].Value;
                        string format = Settings.Instance.CdaUserIdQualifierFormat;
                        id.QualifiedIdentifier = new Uri(string.Format(format, extension));
                        id.Extension = extension;
                    }
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }

            return author;
        }

        /// <summary>
        /// Gets the approver, which will go into the CDA_SIGN.XML file in the CDA package.
        /// </summary>
        /// <param name="participant">The person who should be the approver.</param>
        /// <returns>The CDA signature approver.</returns>
        private static Approver GetApprover(Participant participant)
        {
            Approver approver = new Approver();
            approver.PersonGivenNames = participant.GivenNames;
            approver.PersonFamilyName = participant.FamilyName;
            approver.PersonTitles = participant.Titles;
            approver.PersonNameSuffixes = participant.Suffixes;
            approver.PersonId = participant.Identifiers[0].QualifiedIdentifier;
            return approver;
        }

        /// <summary>
        /// Gets the referenced file name for the organisational logo in the document.
        /// </summary>
        /// <returns>The referenced file name for the organisational logo, or null if there is no logo.</returns>
        private string GetLogoFileName()
        {
            XmlNode logoReferenceElement = cdaDocument.SelectSingleNode(XmlStringMap.LogoReferenceXPath, cdaNamespaces);
            if (logoReferenceElement != null)
            {
                XmlAttribute referenceValueAttribute = logoReferenceElement.Attributes[XmlStringMap.ValueAttributeName];
                if (referenceValueAttribute != null)
                {
                    return referenceValueAttribute.Value;
                }
            }
            return null;
        }

        /// <summary>
        /// If the document references an organisational logo, checks that the
        /// logo meets the PCEHR requirements (no more than 400 by 100 pixels)
        /// and adds the integrity check attributes to the logo reference. The
        /// logo can either be supplied as an attachment or pre-configured in
        /// the Logo column of the Hospital table.
        ///
        /// If using a pre-configured logo, then this method will add it to the
        /// list of attachments.
        /// </summary>
        /// <param name="hospital">The hospital that the document came from.</param>
        /// <param name="attachments">The supplied attachments.</param>
        /// <exception cref="CdaPackagingException">Thrown when logo not found </exception>
        private void ProcessLogo(Hospital hospital, List<Attachment> attachments)
        {
            string logoFileName = GetLogoFileName();
            if (logoFileName != null)
            {
                byte[] logoBytes;
                Attachment logoAttachment = attachments.FirstOrDefault(a => a.FileName == logoFileName);
                if (logoAttachment != null)
                {
                    logoBytes = logoAttachment.Contents;
                }
                else if (hospital.Logo != null)
                {
                    logoBytes = hospital.Logo;
                    attachments.Add(new Attachment() { Contents = logoBytes, FileName = logoFileName });
                }
                else
                {
                    string message = string.Format(ResponseStrings.LogoNotSupplied, logoFileName);
                    throw new CdaPackagingException(message);
                }
                SetLogoIntegrityCheck(logoBytes);
                using (MemoryStream ms = new MemoryStream(logoBytes))
                {
                    Image image = Bitmap.FromStream(ms);
                    if (image.Height > PCEHR_LOGO_HEIGHT || image.Width > PCEHR_LOGO_WIDTH)
                    {
                        string message = string.Format(ResponseStrings.LogoTooLarge, hospital.Description, image.Width, image.Height);
                        throw new CdaPackagingException(message);
                    }
                }
            }
        }

        /// <summary>
        /// Adds the integrity check attributes to the logo reference in the CDA document.
        /// Stores the modified document in the cdaDocumentBytes field, encoded as UTF-8
        /// without a Byte Order Mark (BOM) - this is imperative for PCEHR.
        /// </summary>
        /// <param name="logoFile">The logo file</param>
        private void SetLogoIntegrityCheck(byte[] logoFile)
        {
            XmlNode logoValueElement = cdaDocument.SelectSingleNode(XmlStringMap.LogoValueXPath, cdaNamespaces);
            if (logoValueElement != null)
            {
                XmlAttribute integrityCheck = cdaDocument.CreateAttribute(XmlStringMap.IntegrityCheckAttributeName);
                integrityCheck.Value = Convert.ToBase64String(SHA1.Create().ComputeHash(logoFile));
                logoValueElement.Attributes.Append(integrityCheck);

                XmlAttribute integrityCheckAlgorithm = cdaDocument.CreateAttribute(XmlStringMap.IntegrityCheckAlgorithmAttributeName);
                integrityCheckAlgorithm.Value = XmlStringMap.IntegrityCheckAlgorithmAttributeValue;
                logoValueElement.Attributes.Append(integrityCheckAlgorithm);
            }
            using (MemoryStream stream = new MemoryStream())
            {
                XmlWriterSettings settings = new XmlWriterSettings();
                settings.Encoding = new UTF8Encoding(encoderShouldEmitUTF8Identifier: false);
                using (XmlWriter writer = XmlWriter.Create(stream, settings))
                {
                    cdaDocument.Save(writer);
                }
                this.cdaDocumentBytes = stream.ToArray();
            }
        }

        #endregion Methods
    }
}