﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.Configuration.Tracing;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.DocumentRegistry;

using Registry = Nehta.VendorLibrary.PCEHR.DocumentRegistry;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class DocumentList
    {
        /// <summary>
        /// This function is given the WS name of GetDocumentList.
        /// It returns all documents within the list without filtering the returned list
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>The list of documents with any error details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            DocumentQuery docQuery = new DocumentQuery();
            return _GetDocumentList(patientIdentifier, user, docQuery);
        }

        /// <summary>
        /// This function is given the WS name of GetDocumentListFilterDates.
        /// It returns all documents within the list with the ability to filter by start and end datetimes for creation and the service
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="creationTimeStart">Include documents created after this time.</param>
        /// <param name="creationTimeEnd">Include documents created before this time.</param>
        /// <param name="serviceTimeStart">Include documents for healthcare services after this time.</param>
        /// <param name="serviceTimeEnd">Include documents for healthcare services before this time.</param>
        /// <returns>List of documents.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd)
        {
            DocumentQuery docQuery = new DocumentQuery();
            docQuery.CreationTimeFrom = creationTimeStart;
            docQuery.CreationTimeTo = creationTimeEnd;
            docQuery.ServiceStartTimeFrom = serviceTimeStart;
            docQuery.ServiceStartTimeTo = serviceTimeEnd;

            return _GetDocumentList(patientIdentifier, user, docQuery);
        }

        /// <summary>
        /// Gets the list of documents with the ability to filter by multiple statuses, start and end dates and times for document creation and the healthcare service.
        /// This function is given the WS name of GetDocumentListFilterStatusAndDates.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="documentStatus">Include documents with this status.</param>
        /// <param name="creationTimeStart">Include documents created after this time.</param>
        /// <param name="creationTimeEnd">Include documents created before this time.</param>
        /// <param name="serviceTimeStart">Include documents for healthcare services after this time.</param>
        /// <param name="serviceTimeEnd">Include documents for healthcare services before this time.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, IList<DocumentStatus> documentStatus, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd)
        {
            DocumentQuery docQuery = new DocumentQuery();
            docQuery.DocumentStatus = documentStatus;
            docQuery.CreationTimeFrom = creationTimeStart;
            docQuery.CreationTimeTo = creationTimeEnd;
            docQuery.ServiceStartTimeFrom = serviceTimeStart;
            docQuery.ServiceStartTimeTo = serviceTimeEnd;

            return _GetDocumentList(patientIdentifier, user, docQuery);
        }

        /// <summary>
        /// Get the list of documents corresponding to the specified document query.
        /// This function is given the WS name of GetDocumentListWithQuery.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="docQuery">The query parameters to be sent to the document registry for retrieval.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, DocumentQuery docQuery)
        {
            return _GetDocumentList(patientIdentifier, user, docQuery);
        }

        /// <summary>
        /// Gets the list of documents that have the 'Approved' status.
        /// This function is given the WS name of GetDocumentListActive.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentListActive(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            DocumentQuery docQuery = new DocumentQuery();
            docQuery.DocumentStatus = new List<DocumentStatus> { DocumentStatus.Approved };

            return _GetDocumentList(patientIdentifier, user, docQuery);
        }

        /// <summary>
        /// Gets the list of documents that have the 'Approved' status with the ability to filter by start and end date for document creation and the healthcare service.
        /// This function is given the WS name of GetDocumentListActiveFilterDates.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="creationTimeStart">Include documents created after this time.</param>
        /// <param name="creationTimeEnd">Include documents created before this time.</param>
        /// <param name="serviceTimeStart">Include documents for services after this time.</param>
        /// <param name="serviceTimeEnd">Include documents for services before this time.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentListActive(PatientIdentifierBase patientIdentifier, UserDetails user, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd)
        {
            DocumentQuery docQuery = new DocumentQuery();
            docQuery.DocumentStatus = new List<DocumentStatus> { DocumentStatus.Approved };
            docQuery.CreationTimeFrom = creationTimeStart;
            docQuery.CreationTimeTo = creationTimeEnd;
            docQuery.ServiceStartTimeFrom = serviceTimeStart;
            docQuery.ServiceStartTimeTo = serviceTimeEnd;

            return _GetDocumentList(patientIdentifier, user, docQuery);
        }

        /// <summary>
        /// Internal private function that is used by all calling methods to generate the request and retrieves the response
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="docQuery">The document query.</param>
        /// <returns>DocumentListResponse will pass back success and error messages</returns>
        private DocumentListResponse<PatientIdentifierBase> _GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, DocumentQuery docQuery)
        {
            using (HIPSTraceWriter trace = new HIPSTraceWriter())
            {
                DocumentListResponse<PatientIdentifierBase> docList = new DocumentListResponse<PatientIdentifierBase>();
                docList.PatientIdentifier = patientIdentifier;

                PatientAccess patientAccess = new PatientAccess(user);
                Hospital hospital;
                PatientMaster patientMaster;

                //get hospital
                HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);

                if (status.Status != HipsResponseIndicator.OK)
                {
                    docList.HipsResponse = status;
                    return docList;
                }

                HospitalPatient hospitalPatient;

                // Get the Patient
                status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
                if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi)
                {
                    docList.HipsResponse = status;
                    return docList;
                }

                // Validate the IHI if stale
                IhiSearchResponse ihiResponse = new PatientIhiValidation().GetValidatedIhi(patientIdentifier, hospital, user, patientMaster);
                if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    docList.HipsResponse = ihiResponse.HipsResponse;
                    return docList;
                }

                // Check that a valid IHI was found
                patientAccess.ValidateLocalIhiInformation(patientMaster, docList.HipsResponse);
                if (docList.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return docList;
                }

                //set IHI found
                docList.IhiNumber = patientMaster.Ihi;

                //populate user
                if (!User.PopulateAndValidateUser(hospital, user))
                {
                    status.Status = HipsResponseIndicator.InvalidUser;
                    docList.HipsResponse = status;
                    return docList;
                }

                X509Certificate2 certificate = Helpers.GetConnectionCertificate(hospital);
                Uri uri = Helpers.GetDocumentListUrl();

                // Create PCEHR header
                CommonPcehrHeader header = Helpers.GetHeader(patientIdentifier, docList.IhiNumber, user, hospital);

                // Instantiate the client
                GetDocumentListClient documentListClient = new GetDocumentListClient(uri, certificate, certificate);

                try
                {
                    System.Reflection.FieldInfo clientfield = documentListClient.GetType().GetField("documentRegistryClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                    object ppc = clientfield.GetValue(documentListClient);
                    System.Reflection.FieldInfo clientField2 = ppc.GetType().GetField("documentRegistryClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                    DocumentRegistry_PortTypeClient ptc = clientField2.GetValue(ppc) as DocumentRegistry_PortTypeClient;
                    CustomBinding binding = ptc.Endpoint.Binding as CustomBinding;
                    HttpsTransportBindingElement https = binding.Elements.First(a => a.GetType() == typeof(HttpsTransportBindingElement)) as HttpsTransportBindingElement;

                    //Set the connection timeout for the GetDocumentList service
                    binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                    binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                    binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);

                    // Avoid the proxy
                    if (Settings.Instance.AvoidProxy)
                    {
                        https.UseDefaultWebProxy = false;
                    }

                    // Add server certificate validation callback
                    ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

                    // Create a query object and extract set items from DocumentQuery Object to set into the Adhoc Query Builder
                    AdhocQueryBuilder adhocQueryBuilder = BuildAdhocQuery(docList.IhiNumber, docQuery);

                    // Create the request using the query
                    AdhocQueryRequest queryRequest = adhocQueryBuilder.BuildRequest();

                    // Invoke the service
                    AdhocQueryResponse queryResponse = documentListClient.GetDocumentList(header, queryRequest);

                    //check if there are no errors returned from the service
                    if (queryResponse.RegistryErrorList == null)
                    {
                        //only loop if we have a returned resultset
                        if (queryResponse.RegistryObjectList.ExtrinsicObject != null)
                        {
                            int docCount = queryResponse.RegistryObjectList.ExtrinsicObject.Count();

                            //loop over all Extrinsic Objects and extract data to create DocumentMetaDataItems
                            foreach (Registry.ExtrinsicObjectType extrinsicObj in queryResponse.RegistryObjectList.ExtrinsicObject)
                            {
                                string IhiNumber = "";
                                docList.DocumentList.Add(DocumentHelper.ExtractMetaData(extrinsicObj, out IhiNumber));
                                docList.IhiNumber = IhiNumber;
                            }
                        }

                        //all has succeeded so add in the HIPSResponseObject
                        HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
                        docList.HipsResponse = response;
                    }
                    else //parse for the errors
                    {
                        //fail has occurred so add in the HIPSResponseObject
                        HipsResponse response = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                        String errorCodeContext = queryResponse.RegistryErrorList.RegistryError[0].codeContext;
                        if (!errorCodeContext.IsNullOrEmptyWhitespace())
                        {
                            // Check if this is in the format PCEHR_ERROR_XXXX - Description
                            if (errorCodeContext.IndexOf(" - ") > 0)
                            {
                                string[] errorCodeDelimiter = new string[] { " - " };
                                string[] errors = errorCodeContext.Split(errorCodeDelimiter, StringSplitOptions.None);
                                response.ResponseCode = errors[0];
                                response.ResponseCodeDescription = errors[0]; 
                            }
                            else // No it's another error format, let's just output the string
                            {
                                response.ResponseCode = "Unknown PCEHR Error Code";
                                response.ResponseCodeDescription = errorCodeContext; 
                            }
                        }
                        else
                        {
                            response.ResponseCode = "Unknown PCEHR Error Code";
                            response.ResponseCodeDescription = queryResponse.RegistryErrorList.RegistryError[0].errorCode;
                        }
                        docList.HipsResponse = response;
                    }
                }
                catch (FaultException fe)
                {
                    HipsResponse response = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                    response.ResponseCode = fe.Reason.ToString();
                    response.ResponseCodeDescription = fe.Message;
                    response.ResponseCodeDetails = fe.StackTrace;
                    docList.HipsResponse = response;
                }
                catch (Exception e)
                {
                    HipsResponse response = new HipsResponse(HipsResponseIndicator.SystemError);
                    response.HipsErrorMessage = e.Message;
                    //grab the inner exception if there is one
                    if (e.InnerException != null)
                    {
                        response.ResponseCodeDescription = e.InnerException.Message;
                    }
                    response.ResponseCodeDetails = e.StackTrace;
                    docList.HipsResponse = response;
                }
                finally
                {
                    documentListClient.Close();
                }

                Helpers.InsertAudit(patientMaster, user, hospital, AuditOperationNames.GetDocumentList,
                    docList.HipsResponse,
                    documentListClient.SoapMessages);

                return docList;
            }
        }

        /// <summary>
        /// Builds the AdhocQueryBuilder from the variables set in the DocumentQuery Object
        /// </summary>
        /// <param name="ihi">The IHI to include in the query.</param>
        /// <param name="docQuery">The document query</param>
        /// <returns>The AdhocQueryBuilder</returns>
        private AdhocQueryBuilder BuildAdhocQuery(string ihi, DocumentQuery docQuery)
        {
            AdhocQueryBuilder adhocQueryBuilder;

            if (docQuery.DocumentStatus != null)
            {
                adhocQueryBuilder = new AdhocQueryBuilder(ihi, docQuery.DocumentStatus);
            }
            else
            {
                adhocQueryBuilder = new AdhocQueryBuilder(ihi);
            }

            if (docQuery.CreationTimeFrom.HasValue)
            {
                adhocQueryBuilder.CreationTimeFrom = new ISO8601DateTime((DateTime)docQuery.CreationTimeFrom);
            }

            if (docQuery.CreationTimeTo.HasValue)
            {
                adhocQueryBuilder.CreationTimeTo = new ISO8601DateTime((DateTime)docQuery.CreationTimeTo);
            }

            if (docQuery.DocumentClassCode != null)
            {
                adhocQueryBuilder.ClassCode = docQuery.DocumentClassCode;
            }

            if (docQuery.DocumentUniqueId != null)
            {
                // Convert UUID to OID for caller's convenience. The PCEHR system requires an OID for this parameter.
                adhocQueryBuilder.DocumentUniqueId = XdsMetadataHelper.UuidToOid(docQuery.DocumentUniqueId);
            }

            if (docQuery.FormatCodes != null)
            {
                var allFormats = ListSingleton.Instance.AllDocumentFormats;
                adhocQueryBuilder.FormatCode = new List<AdhocQueryBuilder.PcehrFormatCode>();
                foreach (string oid in docQuery.FormatCodes)
                {
                    DocumentFormat format = allFormats.FirstOrDefault(a => a.Code == oid);
                    if (format != null)
                    {
                        AdhocQueryBuilder.PcehrFormatCode formatCode = new AdhocQueryBuilder.PcehrFormatCode();
                        formatCode.ConceptCode = format.Code;
                        formatCode.ConceptName = format.Description;
                        adhocQueryBuilder.FormatCode.Add(formatCode);
                    }
                    else
                    {
                        string message = string.Format(ResponseStrings.DocumentFormatNotFound, oid);
                        throw new Exception(message);
                    }
                }
            }

            if (docQuery.HealthCareFacilityType != null)
            {
                adhocQueryBuilder.HealthcareFacilityTypeCode = docQuery.HealthCareFacilityType;
            }

            if (docQuery.PracticeSettingTypes != null)
            {
                adhocQueryBuilder.HealthcareFacilityTypeCode = docQuery.HealthCareFacilityType;
            }

            if (docQuery.ServiceStartTimeFrom != null)
            {
                adhocQueryBuilder.ServiceStartTimeFrom = new ISO8601DateTime((DateTime)docQuery.ServiceStartTimeFrom);
            }

            if (docQuery.ServiceStartTimeTo != null)
            {
                adhocQueryBuilder.ServiceStartTimeTo = new ISO8601DateTime((DateTime)docQuery.ServiceStartTimeTo);
            }

            if (docQuery.ServiceStopTimeFrom != null)
            {
                adhocQueryBuilder.ServiceStopTimeFrom = new ISO8601DateTime((DateTime)docQuery.ServiceStopTimeFrom);
            }

            if (docQuery.ServiceStopTimeTo != null)
            {
                adhocQueryBuilder.ServiceStopTimeTo = new ISO8601DateTime((DateTime)docQuery.ServiceStopTimeTo);
            }

            return adhocQueryBuilder;
        }

        /// <summary>
        /// Validates the service certificate.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="certificate">The certificate.</param>
        /// <param name="chain">The chain.</param>
        /// <param name="sslPolicyErrors">The SSL policy errors.</param>
        /// <returns>True if the service certificate was valid.</returns>
        private bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return DocumentHelper.ValidateServiceCertificate(sender, certificate, chain, sslPolicyErrors);
        }
    }
}