﻿using System;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.Text;
using System.Xml;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Cda;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrQueueLogic;
using HIPS.PcehrSchemas;
using HIPS.PcehrSchemas.Exceptions;
using System.Text.RegularExpressions;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    /// <summary>
    /// This class contains the business logic for uploading documents to PCEHR that runs before the operation is placed on the queue.
    /// </summary>
    public class DocumentUploadBeforeQueue
    {
        #region Private Fields

        /// <summary>
        /// Specifies the upper limit on the size of an attachment to a document being uploaded to the PCEHR system.
        /// The PCEHR system limits attachments to ten (10) megabytes.
        /// </summary>
        private const long PCEHR_ATTACHMENT_SIZE_LIMIT = 10 * 1000 * 1000;

        /// <summary>
        /// Lists the acceptable filename extensions for attachments to documents being uploaded to the PCEHR system.
        /// It is a CCA requirement that authoring systems disallow the send to PCEHR if an attachment of unsupported type
        /// is included. The PCEHR system supports the following attachment types (MIME):
        ///     - image/gif
        ///     - image/jpeg
        ///     - application/pdf
        ///     - image/png
        ///     - image/tiff
        /// </summary>
        private readonly string[] PCEHR_ATTACHMENT_EXTENSIONS = { ".gif", ".jpg", ".jpeg", ".pdf", ".png", ".tif", ".tiff" };

        private DateTime admissionDate;
        private Attachment[] attachments;
        private byte[] cdaDocument;
        private string documentFormatCode;
        private string documentIhi;
        private bool inputValidated;
        private PatientAccess patientAccess;
        private QueuedUploadOperation queuedUploadOperation;

        #endregion Private Fields

        #region Properties

        /// <summary>
        /// Gets the hospital
        /// </summary>
        private Hospital Hospital
        {
            get
            {
                return queuedUploadOperation.Hospital;
            }
        }

        /// <summary>
        /// Gets the patient master
        /// </summary>
        private PatientMaster PatientMaster
        {
            get
            {
                return queuedUploadOperation.PatientMaster;
            }
        }

        /// <summary>
        /// Gets a value indicating whether the patient consents to the upload. Under the
        /// PCEHR consent model, if the patient has not withdrawn consent, the patient is
        /// assumed to have consented.
        /// </summary>
        /// <value>
        ///   <c>true</c> if the patient consents; otherwise, <c>false</c>.
        /// </value>
        private bool UploadConsent
        {
            get
            {
                if (queuedUploadOperation.Episode == null)
                {
                    throw new InvalidOperationException();
                }
                return !queuedUploadOperation.Episode.ConsentWithdrawn;
            }
        }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Creates an instance of the business logic for the upload of a document to the PCEHR system.
        /// </summary>
        /// <param name="user">Identifies the person responsible for sending the document.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The date and time when the episode of care started.</param>
        /// <param name="cdaDocument">The CDA document, which is an XML document, typically encoded as UTF-8.</param>
        /// <param name="attachments">Any other files that form part of the document.</param>
        /// <param name="documentFormatCode">A code that identifies the validation rules to be applied by the PCEHR system.</param>
        public DocumentUploadBeforeQueue(UserDetails user, PatientIdentifierBase patientIdentifier, DateTime admissionDate, byte[] cdaDocument, Attachment[] attachments, string documentFormatCode)
        {
            this.queuedUploadOperation = new QueuedUploadOperation(user);
            this.patientAccess = new PatientAccess(user);
            this.queuedUploadOperation.PatientIdentifier = patientIdentifier;
            this.admissionDate = admissionDate;
            this.cdaDocument = cdaDocument;
            this.attachments = attachments;
            this.inputValidated = false;
            this.documentFormatCode = documentFormatCode;
        }

        /// <summary>
        /// Creates an instance of the business logic for the upload of a document to the PCEHR system.
        /// </summary>
        /// <param name="user">Identifies the person responsible for sending the document.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The date and time when the episode of care started.</param>
        /// <param name="cdaDocument">The CDA document, which is an XML document, typically encoded as UTF-8.</param>
        /// <param name="attachments">Any other files that form part of the document.</param>
        /// <param name="documentFormatCode">A code that identifies the validation rules to be applied by the PCEHR system.</param>
        /// <param name="pendingItem">An existing Pending Item (PCEHRMessageQueue).</param>
        public DocumentUploadBeforeQueue(UserDetails user, PatientIdentifierBase patientIdentifier, DateTime admissionDate, byte[] cdaDocument, Attachment[] attachments, string documentFormatCode, PcehrMessageQueue pendingItem)
        {
            this.queuedUploadOperation = new QueuedUploadOperation(user);
            this.patientAccess = new PatientAccess(user);
            this.queuedUploadOperation.PatientIdentifier = patientIdentifier;
            this.admissionDate = admissionDate;
            this.cdaDocument = cdaDocument;
            this.attachments = attachments;
            this.inputValidated = false;
            this.documentFormatCode = documentFormatCode;
            this.queuedUploadOperation.PendingItem = pendingItem;
        }
        #endregion Constructors

        #region Public Methods

        /// <summary>
        /// Sets the HL7 Message Log ID of the original message for eventual acknowledgement.
        /// </summary>
        /// <param name="hl7MessageLogId">The HL7 message log ID.</param>
        public void SetHL7MessageLogId(int hl7MessageLogId)
        {
            this.queuedUploadOperation.HL7MessageLogId = hl7MessageLogId;
        }

        /// <summary>
        /// Uploads the document to the PCEHR National Repository.
        /// </summary>
        /// <returns>Status information</returns>
        public HipsResponse UploadOrSupersedeDocument()
        {
            HipsResponse response = ValidateInput();
            if (response.Status != HipsResponseIndicator.OK)
            {
                return response;
            }

            PatientIhiValidation patientIhiValidation = new PatientIhiValidation();
            IhiSearchResponse ihiResponse;
            ihiResponse = patientIhiValidation.GetValidatedIhi(queuedUploadOperation.PatientIdentifier, Hospital, queuedUploadOperation.User, PatientMaster);
            if (ihiResponse.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
            {
                // If the IHI cannot be validated because the HI service is unavailable, HIPS will
                // allow the upload operation to go onto the queue. For the purpose of the comparison
                // between the local IHI and the document IHI below, the current IHI on the patient
                // master will be used.

                EventLogger.WriteLog(ResponseStrings.InfoUploadQueuedWithStaleIhi, new Exception(ihiResponse.HipsResponse.HipsErrorMessage), queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_092);
                ihiResponse = new IhiSearchResponse();
                ihiResponse.Ihi = PatientMaster.Ihi;
            }
            else
            {
                if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return ihiResponse.HipsResponse;
                }

                this.patientAccess.ValidateLocalIhiInformation(PatientMaster, response);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    return response;
                }
            }

            response = PackageDocument();
            if (response.Status != HipsResponseIndicator.OK)
            {
                return response;
            }
            if (!UploadConsent)
            {
                // Log Message:       (INFO) Document not uploaded because patient withdrew consent
                // Exception Message: Document ID {extension} was blocked because patient {identifier} at {hospital} had withdrawn consent.
                // Response Message:  The patient has withdrawn consent to upload this document.
                string shortDocumentId = queuedUploadOperation.SourceSystemDocumentId.Split('^').Last();
                string exceptionMessage = string.Format(ExceptionMessages.ConsentWithdrawn, shortDocumentId, queuedUploadOperation.PatientIdentifier, Hospital.Description);
                EventLogger.WriteLog(LogMessages.InfoConsentWithdrawn, new Exception(exceptionMessage), queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_093);
                return new HipsResponse(HipsResponseIndicator.ConsentWithdrawn, ResponseStrings.UploadDeniedByPatient);
            }
            if (Hospital.UploadDocumentMinimumAge != 0)
            {
                DateTime cutOff = PatientMaster.RegisteredDateOfBirth.AddYears(Hospital.UploadDocumentMinimumAge);
                if (admissionDate < cutOff)
                {
                    // Response message:  Discharge Summary has not been sent to National eHealth Record for child under {age} due to local policy.
                    // Log message:       (INFO) Document not uploaded because patient under age
                    // Exception Message: Document ID {extension} was blocked because patient {identifier} at {hospital} was under {age} at admission.
                    string responseMessage = string.Format(ResponseStrings.PatientUnderAge, Hospital.UploadDocumentMinimumAge);
                    string shortDocumentId = queuedUploadOperation.SourceSystemDocumentId.Split('^').Last();
                    string exceptionMessage = string.Format(ExceptionMessages.PatientUnderAge, shortDocumentId, queuedUploadOperation.PatientIdentifier, Hospital.Description, Hospital.UploadDocumentMinimumAge);
                    EventLogger.WriteLog(LogMessages.InfoPatientUnderAge, new Exception(exceptionMessage), queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_094);
                    return new HipsResponse(HipsResponseIndicator.PatientUnderAge, responseMessage);
                }
            }
            if (this.documentIhi != ihiResponse.Ihi)
            {
                IhiLookupAlert item = new IhiLookupAlert();
                item.PatientMasterId = PatientMaster.PatientMasterId.Value;
                item.Message = string.Format(ResponseStrings.IhiMismatchDetail, ihiResponse.Ihi, this.documentIhi);
                new IhiLookupAlertDl().Insert(item);
                return new HipsResponse(HipsResponseIndicator.InvalidIhi, ResponseStrings.IhiMismatch);
            }
            response = PlaceOperationOnQueue();
            return response;
        }

        #endregion Public Methods

        #region Private Methods

        /// <summary>
        /// Gets the CDA information.
        /// </summary>
        /// <exception>When an error occurs while extracting information from the CDA document.</exception>
        private void GetDocumentInformationFromCda()
        {
            using (MemoryStream stream = new MemoryStream(cdaDocument))
            {
                XmlDocument xmlDoc = new XmlDocument();
                xmlDoc.Load(stream);
                XmlNamespaceManager xnm = new XmlNamespaceManager(new NameTable());
                xnm.AddNamespace("x", "urn:hl7-org:v3");
                xnm.AddNamespace("ext", "http://ns.electronichealth.net.au/Ci/Cda/Extensions/3.0");
                XmlNode node = xmlDoc.SelectSingleNode("//x:code", xnm);

                string documentTypeCode = node.Attributes["code"].Value;
                queuedUploadOperation.DocumentType = ListSingleton.Instance.AllDocumentTypes.Where(a => a.Code == documentTypeCode).FirstOrDefault();
                if (queuedUploadOperation.DocumentType == null)
                {
                    string message = string.Format(ResponseStrings.UnknownDocumentTypeCode, documentTypeCode);
                    throw new Exception(message);
                }

                node = xmlDoc.SelectSingleNode("//x:id", xnm);
                StringBuilder docId = new StringBuilder(node.Attributes.GetNamedItem("root").Value);

                bool isGuidFormat = Regex.IsMatch(docId.ToString(), @"\b[A-F0-9]{8}(?:-[A-F0-9]{4}){3}-[A-F0-9]{12}\b", RegexOptions.IgnoreCase);
                bool isOidFormat = Regex.IsMatch(docId.ToString(), @"^[\d]+(.[\d]+)*$", RegexOptions.IgnoreCase);

                if (!isGuidFormat && !isOidFormat)
                {
                    string message = string.Format(ResponseStrings.UnknownDocumentTypeCode, documentTypeCode);
                    throw new Exception(message);
                }
                
                if (node.Attributes["extension"] != null)
                {
                    docId.Append("^");
                    docId.Append(node.Attributes.GetNamedItem("extension").Value);
                }
                queuedUploadOperation.SourceSystemDocumentId = docId.ToString();

                node = xmlDoc.SelectSingleNode("//x:setId", xnm);
                StringBuilder setId = new StringBuilder(node.Attributes.GetNamedItem("root").Value);
                if (node.Attributes["extension"] != null)
                {
                    setId.Append("^");
                    setId.Append(node.Attributes.GetNamedItem("extension").Value);
                }
                queuedUploadOperation.SourceSystemSetId = setId.ToString();

                node = xmlDoc.SelectSingleNode("//x:recordTarget/x:patientRole/x:patient/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='IHI']", xnm);
                string ihi = node.Attributes["root"].Value;
                string oidQualifier = "1.2.36.1.2001.1003.0.";
                if (ihi.StartsWith(oidQualifier))
                {
                    ihi = ihi.Substring(oidQualifier.Length);
                }
                this.documentIhi = ihi;

                CdaValidation validation = new CdaValidation();
                validation.Validate(xmlDoc, xnm, queuedUploadOperation.DocumentType, queuedUploadOperation.Hospital, queuedUploadOperation.PatientMaster);
            }
        }

        /// <summary>
        /// Packages the document
        /// </summary>
        /// <returns>Success (OK) or reason for failure </returns>
        private HipsResponse PackageDocument()
        {
            if (!this.inputValidated)
            {
                return new HipsResponse(HipsResponseIndicator.SystemError, "Input has not been validated");
            }

            try
            {
                CdaPackaging cdaPackaging = new CdaPackaging(cdaDocument);
                queuedUploadOperation.Author = cdaPackaging.GetAuthor();
                queuedUploadOperation.Package = cdaPackaging.GenerateCdaPackageInMemory(queuedUploadOperation.Hospital, attachments);
            }
            catch (CdaPackagingException ex)
            {
                HipsResponse response = new HipsResponse(HipsResponseIndicator.InvalidDocument, ResponseStrings.InvalidCda);
                response.ResponseCode = ex.Message;
                EventLogger.WriteLog(ResponseStrings.InvalidCda, ex, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_100);
                return response;
            }
            catch (Exception ex)
            {
                HipsResponse response = new HipsResponse(HipsResponseIndicator.InvalidDocument, ResponseStrings.InvalidCda);
                response.ResponseCode = ex.Message;
                EventLogger.WriteLog(ResponseStrings.InvalidCda, ex, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_101);
                return response;
            }

            return new HipsResponse(HipsResponseIndicator.OK, ResponseStrings.Success);
        }

        /// <summary>
        /// Populates the pending item, saves the operation in the PCEHR Data Store and adds it to the MSMQ queue.
        /// </summary>
        /// <returns>Response explaining any failure to store in database or MSMQ.</returns>
        private HipsResponse PlaceOperationOnQueue()
        {
            // Ensure that the other methods have been executed successfully first.
            if (!this.inputValidated || this.queuedUploadOperation.Package == null)
            {
                return new HipsResponse(HipsResponseIndicator.SystemError);
            }
            // Create a new PendingItem if one hasn't already been created
            if (this.queuedUploadOperation.PendingItem == null) 
                this.queuedUploadOperation.PendingItem = new PcehrMessageQueue();
            this.queuedUploadOperation.PendingItem.QueueOperationId = (int)QueueOperation.UploadOrSupersede;
            this.queuedUploadOperation.PendingItem.QueueStatusId = (int)QueueStatus.Pending;
            this.queuedUploadOperation.PendingItem.EpisodeId = queuedUploadOperation.Episode.EpisodeId.Value;
            this.queuedUploadOperation.PendingItem.SourceSystemSetId = queuedUploadOperation.SourceSystemSetId;
            this.queuedUploadOperation.PendingItem.SourceSystemDocumentId = queuedUploadOperation.SourceSystemDocumentId;

            using (MemoryStream stream = new MemoryStream())
            {
                new DataContractSerializer(typeof(QueuedUploadOperation)).WriteObject(stream, this.queuedUploadOperation);
                this.queuedUploadOperation.PendingItem.SerialisedObject = stream.ToArray();
            }

            DocumentQueueTransactionHandler handler = new DocumentQueueTransactionHandler(this.queuedUploadOperation.User);
            return handler.SaveUploadOperationAndPlaceOnQueue(this.queuedUploadOperation);
        }

        /// <summary>
        /// Validates the size and type of the attachments for upload to PCEHR.
        /// </summary>
        /// <param name="attachments">The array of attachments</param>
        /// <param name="user">The user who is attempting to upload the document</param>
        /// <returns>A response object with status OK or else InvalidDocument and an error message.</returns>
        private HipsResponse ValidateAttachments()
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            if (this.attachments == null)
            {
                //Initialise so we don't break anything downstream
                this.attachments = new Attachment[0];
                return response;
            }
            foreach (Attachment attachment in this.attachments)
            {
                string fileName = attachment.FileName;
                long size = attachment.Contents.LongLength;

                if (size > PCEHR_ATTACHMENT_SIZE_LIMIT)
                {
                    string message = string.Format(ResponseStrings.AttachmentTooLarge, fileName, size);
                    response.Status = HipsResponseIndicator.InvalidDocument;
                    response.HipsErrorMessage = message;
                    EventLogger.WriteLog(message, null, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_102);
                    break;
                }
                string extension;
                try
                {
                    if (fileName.Contains('/') || fileName.Contains('\\'))
                    {
                        throw new ArgumentException("Filename contains path separator characters.");
                    }
                    extension = new FileInfo(fileName).Extension;
                }
                catch (ArgumentException ex)
                {
                    response.Status = HipsResponseIndicator.InvalidDocument;
                    response.HipsErrorMessage = string.Format(ResponseStrings.AttachmentFilenameInvalid, fileName);
                    EventLogger.WriteLog(response.HipsErrorMessage, ex, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_103);
                    break;
                }
                if (!PCEHR_ATTACHMENT_EXTENSIONS.Contains(extension, StringComparer.CurrentCultureIgnoreCase))
                {
                    string message = string.Format(ResponseStrings.AttachmentTypeDisallowed, fileName);
                    response.Status = HipsResponseIndicator.InvalidDocument;
                    response.HipsErrorMessage = message;
                    EventLogger.WriteLog(message, null, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_104);
                    break;
                }
            }

            return response;
        }

        /// <summary>
        /// Checks the attachment types and sizes, CDA document length not zero,
        /// populates the user details (authorised employee name and user ID),
        /// and retrieves the hospital, patient and episode from the database.
        ///
        /// If the patient identifier is a ValidatedIhi and the patient or episode do
        /// not exist, then stub records will be created for the patient and episode.
        /// </summary>
        /// <returns>Information about success or failure</returns>
        private HipsResponse ValidateInput()
        {
            HipsResponse response = ValidateAttachments();

            if (response.Status != HipsResponseIndicator.OK)
            {
                return response;
            }

            if (cdaDocument.Length == 0)
            {
                EventLogger.WriteLog(ResponseStrings.NoCdaDocument, null, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_095);
                response.Status = HipsResponseIndicator.InvalidDocument;
                response.HipsErrorMessage = ResponseStrings.NoCdaDocument;
                return response;
            }
            Hospital hospital;
            HipsResponse errorIndicator = patientAccess.GetHospital(queuedUploadOperation.PatientIdentifier, out hospital);
            if (errorIndicator.Status != HipsResponseIndicator.OK)
            {
                EventLogger.WriteLog(ResponseStrings.RetrieveHospitalFailed, null, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_096);
                return errorIndicator;
            }
            queuedUploadOperation.Hospital = hospital;
            if (!User.PopulateAndValidateUser(hospital, queuedUploadOperation.User))
            {
                return new HipsResponse(HipsResponseIndicator.InvalidUser, ResponseStrings.RetrievePatientFailed);
            }

            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            errorIndicator = patientAccess.GetPatient(queuedUploadOperation.PatientIdentifier, queuedUploadOperation.Hospital, out hospitalPatient, out patientMaster);
            if (errorIndicator.Status != HipsResponseIndicator.OK && errorIndicator.Status != HipsResponseIndicator.InvalidIhi)
            {
                EventLogger.WriteLog(ResponseStrings.RetrievePatientFailed, null, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_097);
                return errorIndicator;
            }
            queuedUploadOperation.PatientMaster = patientMaster;
            queuedUploadOperation.HospitalPatient = hospitalPatient;

            try
            {
                GetDocumentInformationFromCda();
            }
            catch (Exception ex)
            {
                response = new HipsResponse(HipsResponseIndicator.InvalidDocument, ResponseStrings.InvalidCda);
                response.ResponseCode = ex.Message;
                EventLogger.WriteLog(ResponseStrings.InvalidCda, ex, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_134);
                return response;
            }

            // Check if we have a Pending Item object and an EpisodeId in that
            if (queuedUploadOperation.PendingItem != null && queuedUploadOperation.PendingItem.EpisodeId > 0)
            {
                queuedUploadOperation.Episode = patientAccess.EpisodeDataAccess.GetAll(queuedUploadOperation.HospitalPatient.PatientId, null).FirstOrDefault(e => e.EpisodeId == queuedUploadOperation.PendingItem.EpisodeId);
            }
            else
            {
                //A Specific Episode that has not been cancelled must be found in this case
                queuedUploadOperation.Episode = patientAccess.GetEpisodeWithoutCancelled(queuedUploadOperation.PatientIdentifier, admissionDate, hospitalPatient, queuedUploadOperation.SourceSystemSetId);
            }
            
            //Could not find the episode for the patient to upload the document against
            if (queuedUploadOperation.Episode == null)
            {
                // Response message:  Discharge Summary has not been sent to the National eHealth Record as the admission at {admission} for the patient {patient identifier} at {hospital} could not be found.  Check that this is the correct Patient Identifier..
                // Log message:       (INFO) Document not uploaded because episode missing
                // Exception Message: HIPS had not received (or could not find) the admission at {admission} for patient {patient identifier} at {hospital} and so blocked the upload of document {extension}.
                string responseMessage = string.Format(ResponseStrings.CouldNotRetrieveEpisode, admissionDate, queuedUploadOperation.PatientIdentifier, Hospital.Description);
                string shortDocumentId = queuedUploadOperation.SourceSystemDocumentId.Split('^').Last();
                string exceptionMessage = string.Format(ExceptionMessages.CouldNotRetrieveEpisode, admissionDate, queuedUploadOperation.PatientIdentifier, Hospital.Description, shortDocumentId);
                EventLogger.WriteLog(LogMessages.InfoCouldNotRetrieveEpisode, new Exception(exceptionMessage), queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_098);
                return new HipsResponse(HipsResponseIndicator.InvalidEpisode, responseMessage);
            }

            if (string.IsNullOrEmpty(documentFormatCode))
            {
                documentFormatCode = Settings.Instance.DefaultDocumentFormatCode;
            }

            queuedUploadOperation.DocumentFormat = ListSingleton.Instance.AllDocumentFormats.Where(a => a.Code == documentFormatCode).FirstOrDefault();
            if (queuedUploadOperation.DocumentFormat == null)
            {
                string message = string.Format(ResponseStrings.DocumentFormatNotFound, documentFormatCode);
                EventLogger.WriteLog(message, null, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_099);
                return new HipsResponse(HipsResponseIndicator.InvalidDocument, message);
            }

            this.inputValidated = true;
            return new HipsResponse(HipsResponseIndicator.OK);
        }

        #endregion Private Methods
    }
}