﻿using System;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.CDAPackage;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.GetView;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class GetViewInvoker
    {
        /// <summary>
        /// Returns the Document and Attachments from the passed in request
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="request">The view request.</param>
        /// <returns>ViewResponse will also pass back success and error messages</returns>
        public ViewResponse GetView(PatientIdentifierBase patientIdentifier, UserDetails user, ViewRequestBase request)
        {
            ViewResponse response = new ViewResponse();
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            PatientMaster patientMaster;

            //get hospital
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = status;
                return response;
            }

            HospitalPatient hospitalPatient;

            // Get patient ensures that we have an IHI with no outstanding alerts. It is alright if the IHI is invalid for now.
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi)
            {
                response.HipsResponse = status;
                return response;
            }

            // If the IHI was last validated outside the configured period, attempt validation.
            IhiSearchResponse ihiResponse = new PatientIhiValidation().GetValidatedIhi(patientIdentifier, hospital, user, patientMaster);
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = ihiResponse.HipsResponse;
                return response;
            }

            // If the IHI is still invalid then HIPS must prevent access to the PCEHR.
            patientAccess.ValidateLocalIhiInformation(patientMaster, status);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = status;
                return response;
            }

            // Populate user
            if (!User.PopulateAndValidateUser(hospital, user))
            {
                status.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse = status;
                return response;
            }

            X509Certificate2 certificate = Helpers.GetConnectionCertificate(hospital);
            Uri uri = Helpers.GetViewUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(patientIdentifier, patientMaster.Ihi, user, hospital);

            // Instantiate the client
            GetViewClient getViewClient = new GetViewClient(uri, certificate, certificate);

            try
            {
                System.Reflection.FieldInfo clientField = getViewClient.GetType().GetField("getViewClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                GetViewPortTypeClient ptc = clientField.GetValue(getViewClient) as GetViewPortTypeClient;
                WSHttpBinding wsbinding = ptc.Endpoint.Binding as WSHttpBinding;

                //Set the connection timeout for the GetView service
                wsbinding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                wsbinding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                wsbinding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);

                // Avoid the proxy
                if (Settings.Instance.AvoidProxy)
                {
                    wsbinding.UseDefaultWebProxy = false;
                }

                // Add server certificate validation call-back
                ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

                // Create a request
                Nehta.VendorLibrary.PCEHR.GetView.getView getViewRequest = new Nehta.VendorLibrary.PCEHR.GetView.getView();
                getViewRequest.view = request.ToNehtaLibraryObject();

                // Invoke the service
                getViewResponse getViewResponse = getViewClient.GetView(header, getViewRequest);

                //check if there are no errors returned from the service
                if (getViewResponse != null)
                {
                    if (getViewResponse.view != null)
                    {
                        if (getViewResponse.view.data != null && getViewResponse.view.data.Length > 0)
                        {
                            // Special workaround for CCA testing of CIS_301_018634_A because we can't actually
                            // get an invalid package from a view, but only inject it right at the point before
                            // the validation happens.
                            /*
                                getViewResponse.view.data = System.IO.File.ReadAllBytes(
                                    @"..\..\..\Test.CommonCcaNoc\SampleDocuments\PCEHR_CIS_ConformanceTestDataID_51.zip");
                            */
                            //unpack the root document
                            CDAPackage package = CDAPackageUtility.Extract(getViewResponse.view.data, VerifyCertificate);

                            //add the unique document id and repository id
                            response.TemplateId = getViewResponse.view.templateID;

                            //add root document details
                            response.Document = package.CDADocumentRoot.FileContent;
                            response.FileName = package.CDADocumentRoot.FileName;
                            response.MimeType = System.Net.Mime.MediaTypeNames.Text.Xml;

                            //add attachments from the package
                            if (package.CDADocumentAttachments != null)
                            {
                                foreach (CDAPackageFile attachment in package.CDADocumentAttachments)
                                {
                                    Attachment responseAttachment = new Attachment();
                                    responseAttachment.FileName = attachment.FileName;
                                    responseAttachment.Contents = attachment.FileContent;
                                    response.Attachments.Add(responseAttachment);
                                }
                            }

                            //all has succeeded so add in the HIPSResponseObject
                            response.HipsResponse = new HipsResponse(HipsResponseIndicator.OK);

                            // Now add warnings to the HipsResponse if the demographics do not match.
                            DemographicCheck demographicCheck = new DemographicCheck(patientMaster, user);
                            demographicCheck.ValidateDemographics(response.Document, response.HipsResponse);
                        }
                        else
                        {
                            //no documents can be found in the registry response
                            HipsResponse noDocumentResponse = new HipsResponse(HipsResponseIndicator.InvalidDocument);
                            noDocumentResponse.ResponseCodeDescription = "No Documents were returned with the view response";
                            response.HipsResponse = noDocumentResponse;
                            return response;
                        }
                    }
                    else //parse for the errors
                    {
                        // A failure has occurred so add the details in the HipsResponse object
                        HipsResponse errorResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                        string errorCodeContext =
                        errorResponse.ResponseCode = getViewResponse.responseStatus.code;
                        errorResponse.ResponseCodeDescription = getViewResponse.responseStatus.description;
                        errorResponse.ResponseCodeDetails = getViewResponse.responseStatus.details;
                        response.HipsResponse = errorResponse;
                    }
                }
                else
                {
                    // No response object found.
                    HipsResponse noViewResponse = new HipsResponse(HipsResponseIndicator.InvalidDocument);
                    noViewResponse.ResponseCodeDescription = "No response was returned from GetView";
                    response.HipsResponse = noViewResponse;
                }
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.GetView.StandardErrorType> fe)
            {
                HipsResponse faultResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                string[] messageParts = fe.Detail.message.Split(new string[] { " - " }, StringSplitOptions.RemoveEmptyEntries);
                faultResponse.ResponseCode = messageParts[0];
                if (messageParts.Length > 1)
                {
                    faultResponse.ResponseCodeDescription = messageParts[1];
                }
                faultResponse.ResponseCodeDetails = fe.Detail.errorCode.ToString();
                response.HipsResponse = faultResponse;
            }
            catch (FaultException fe)
            {
                HipsResponse faultResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                faultResponse.ResponseCode = fe.Reason.ToString();
                faultResponse.ResponseCodeDescription = fe.Message;
                response.HipsResponse = faultResponse;
            }
            catch (SignatureVerificationException sve)
            {
                HipsResponse invalidDocumentResponse = new HipsResponse(HipsResponseIndicator.InvalidDocument);
                invalidDocumentResponse.HipsErrorMessage = sve.Message;
                response.HipsResponse = invalidDocumentResponse;
            }
            catch (Exception e)
            {
                HipsResponse exceptionResponse = new HipsResponse(HipsResponseIndicator.SystemError);
                exceptionResponse.HipsErrorMessage = e.Message;

                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    exceptionResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                exceptionResponse.ResponseCodeDetails = e.StackTrace;
                response.HipsResponse = exceptionResponse;
            }
            finally
            {
                getViewClient.Close();
            }

            Helpers.InsertAudit(patientMaster, user, hospital, AuditOperationNames.GetView, response.HipsResponse, getViewClient.SoapMessages);

            return response;
        }

        /// <summary>
        /// Validates the service certificate.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="certificate">The certificate.</param>
        /// <param name="chain">The chain.</param>
        /// <param name="sslPolicyErrors">The SSL policy errors.</param>
        /// <returns>True if the service certificate was valid.</returns>
        private bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return DocumentHelper.ValidateServiceCertificate(sender, certificate, chain, sslPolicyErrors);
        }

        private void VerifyCertificate(X509Certificate2 certificate)
        {
            // This is an sample certificate check, which does an online revocation check.
            // In the future, there may be CDA packages which are signed with certificates
            // which are valid at signing time, but have since been revoked or expired.
            // In this case, the certificate check should be relaxed. One such way is to
            // change the revocation mode to "no check". Eg:
            // chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;

            // Setup the chain
            var chain = new X509Chain();
            chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;
            chain.ChainPolicy.RevocationFlag = X509RevocationFlag.EndCertificateOnly;

            // Perform the validation
            chain.Build(certificate);

            // Check the results
            if (chain.ChainStatus.Length == 0)
            {
                // No errors found
            }
            else
            {
                // Errors found
            }
        }
    }
}