﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.RegisterPCEHR;
using HIPS.PcehrSchemas.Exceptions;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    /// <summary>
    /// Contains methods to assist in the registration of a patient with the PCEHR
    /// </summary>
    public static class PcehrRegistration
    {
        #region Events

        /// <summary>
        /// Validates the service certificate.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="certificate">The certificate.</param>
        /// <param name="chain">The chain.</param>
        /// <param name="sslPolicyErrors">The SSL policy errors.</param>
        /// <returns></returns>
        private static bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return DocumentHelper.ValidateServiceCertificate(sender, certificate, chain, sslPolicyErrors);
        }

        #endregion Events

        #region Private Fields

        /// <summary>
        /// Specifies the upper limit on the size of a consent form being uploaded to the PCEHR system.
        /// The PCEHR system limits attachments to two hundred (200) kilobytes.
        /// </summary>
        private const long PCEHR_CONSENT_FORM_SIZE_LIMIT = 200 * 1000;

        /// <summary>
        /// Lists the acceptable filename extensions for consent forms being uploaded to the PCEHR system.
        /// It is a CCA requirement that authoring systems disallow the send to PCEHR if a consent form of an unsupported type
        /// is included. The PCEHR system supports the following attachment types (MIME):
        ///     - image/gif
        ///     - image/jpeg
        ///     - application/pdf
        ///     - image/png
        ///     - image/tiff
        ///     - text/xml
        /// </summary>
        private static readonly string[] PCEHR_CONSENT_FORM_EXTENSIONS = { ".gif", ".jpg", ".jpeg", ".pdf", ".png", ".tif", ".tiff", ".xml" };

        #endregion Private Fields

        #region Methods

        /// <summary>
        /// Lists the patients without a PCEHR.
        /// </summary>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <returns>List of patients without a PCEHR</returns>
        /// <exception cref="System.Exception">Hospital code or code system invalid.</exception>
        public static PatientWithoutPcehrResponse ListPatientsWithoutPcehr(string hospitalCodeSystem, string hospitalCode)
        {
            PatientWithoutPcehrResponse response = new PatientWithoutPcehrResponse();

            int? hospitalId = null;

            // If the user has provide a Hospital Code and Code System then retrieve the Hospital Id.
            if (hospitalCode != null)
            {
                HospitalSingleton hospitals = HospitalSingleton.Value;
                Hospital hospital = hospitals.Find(hospitalCode, hospitalCodeSystem);

                if (hospital == null)
                {
                    HipsResponse hipsResponse = new HipsResponse(HipsResponseIndicator.InvalidHospital, ResponseStrings.HospitalNotFound);

                    response.HipsResponse = hipsResponse;

                    return response;
                }
                else
                {
                    hospitalId = hospital.Id;
                }
            }

            try
            {
                AssistedRegistrationDl assistedReg = new AssistedRegistrationDl();

                response.PatientWithoutPcehrList = assistedReg.ListPatientsWithoutPcehr(hospitalCodeSystem, hospitalId);

                response.HipsResponse = new HipsResponse(HipsResponseIndicator.OK);
            }
            catch (Exception ex)
            {
                HipsResponse exceptionResponse = new HipsResponse(HipsResponseIndicator.SystemError);
                exceptionResponse.HipsErrorMessage = ex.Message;

                //grab the inner exception if there is one
                if (ex.InnerException != null)
                {
                    exceptionResponse.ResponseCodeDescription = ex.InnerException.Message;
                }
                exceptionResponse.ResponseCodeDetails = ex.StackTrace;
                response.HipsResponse = exceptionResponse;
            }

            return response;
        }

        /// <summary>
        /// Registers the patient.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="representativeDemographics">The representative demographics.</param>
        /// <param name="identityType">Type of the identity.</param>
        /// <param name="indigenousStatus">The indigenous status.</param>
        /// <param name="documentConsent">The document consent.</param>
        /// <param name="correspondenceChannel">The correspondence channel.</param>
        /// <param name="acceptedTermsAndConditions">if set to <c>true</c> [accepted terms and conditions].</param>
        /// <param name="user">Information to identify the person responsible for registering the patient</param>
        /// <param name="consentForm">The consent form.</param>
        /// <param name="patientMasterId">The patient master identifier.</param>
        /// <returns>The PCEHR registration response.</returns>
        public static PcehrRegistrationResponse RegisterPatient(PatientIdentifierBase patientIdentifier, Demographic representativeDemographics, EvidenceOfIdentityType identityType, IndigenousStatus indigenousStatus, List<RegistrationDocumentConsent> documentConsent, RegistrationCorrespondenceChannel correspondenceChannel, bool acceptedTermsAndConditions, UserDetails user, Attachment consentForm, int patientMasterId = 0, bool dependantRegistration = false, bool representativeDeclaration = false)
        {
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            registerPCEHR request;

            PatientAccess patientAccess = new PatientAccess(user);

            PcehrRegistrationResponse result = new PcehrRegistrationResponse();
            result.HipsResponse = new HipsResponse(HipsResponseIndicator.SystemError);

            //patient identifier must be set
            if (patientIdentifier == null)
            {
                result.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                result.HipsResponse.HipsErrorMessage = ResponseStrings.IndividualNotProvided;
                return result;
            }

            //indigenous Status must be set
            if (indigenousStatus == 0)
            {
                result.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                result.HipsResponse.HipsErrorMessage = ResponseStrings.IndigenousStatusNotProvided;
                return result;
            }

            //Evidence Of IdentityType must be set
            if (identityType == 0)
            {
                result.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                result.HipsResponse.HipsErrorMessage = ResponseStrings.EvidenceOfIdentityTypeNotProvided;
                return result;
            }

            if (dependantRegistration)
            {
                if (representativeDemographics != null)
                {
                    // Check that the representative contains valid demographics
                    result.HipsResponse = ValidateDemographics(representativeDemographics, ResponseStrings.Representative);
                    if (result.HipsResponse.Status != HipsResponseIndicator.OK)
                    {
                        return result;
                    }
                }
                else
                {
                    result.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                    result.HipsResponse.HipsErrorMessage = ResponseStrings.RepresentativeNotProvided;
                    return result;
                }
            }

            // Validate the assertions
            result.HipsResponse = ValidateConsentForm(consentForm);
            if (result.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return result;
            }

            result.HipsResponse = ValidateDocumentConsent(documentConsent);
            if (result.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return result;
            }

            result.HipsResponse = ValidateTermsAndConditions(acceptedTermsAndConditions);
            if (result.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return result;
            }

            result.HipsResponse = ValidateIvcCorrespondence(correspondenceChannel);
            if (result.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return result;
            }

            result.HipsResponse = ValidateRepresentativeDeclaration(representativeDemographics, representativeDeclaration);
            if (result.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return result;
            }

            // If we have patient demographics
            if (patientIdentifier is Demographic)
            {
                // Check that the patient contains valid demographics
                result.HipsResponse = ValidateDemographics(patientIdentifier as Demographic, ResponseStrings.Patient);

                if (result.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    // An error occurred with the validation so return the error to the calling program.
                    return result;
                }

                // Get the hospital for the selected patient
                result.HipsResponse = patientAccess.GetVisitorHospital(patientIdentifier, out hospital);
                if (result.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return result;
                }

                // Get the patient master and hospital patient records
                if (patientMasterId == 0)
                {
                    result.HipsResponse = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster, allowDemographics: true);
                }
                else
                {
                    result.HipsResponse = GetAndUpdatePatient(patientAccess, patientIdentifier as Demographic, patientMasterId, hospital, user, out hospitalPatient, out patientMaster);
                }

                // Check if the result.HipsResponse is not equal to Ok and Invalid Ihi
                if (result.HipsResponse.Status != HipsResponseIndicator.OK && result.HipsResponse.Status != HipsResponseIndicator.InvalidIhi)
                {
                    return result;
                }

                if (patientMaster == null || hospitalPatient == null)
                {
                    return result;
                }

                // Now that we have the patient data validate it against the PCEHR rules
                HipsResponse validationResponse = ValidatePatientAndRepresentative(patientMaster, representativeDemographics);

                if (validationResponse.Status != HipsResponseIndicator.OK)
                {
                    result.HipsResponse = validationResponse;
                    return result;
                }

                // If we don't have a validated IHI for the patient then search for one via Medicare
                if (result.HipsResponse.Status == HipsResponseIndicator.InvalidIhi && Settings.Instance.BypassHIService == false)
                {
                    PatientIhiValidation validation = new PatientIhiValidation();

                    IhiSearchResponse searchResponse = validation.GetValidatedIhi(patientIdentifier, hospital, user, patientMaster, patientMaster.DateOfBirth);

                    if (searchResponse.IhiStatus != IhiStatus.Active)
                    {
                        result.HipsResponse.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                        result.HipsResponse.HipsErrorMessage = searchResponse.HipsResponse.HipsErrorMessage;
                        if (patientMaster.PatientMasterId.HasValue)
                        {
                            result.PatientMasterId = patientMaster.PatientMasterId.Value;
                        }
                        return result;
                    }

                    result.HipsResponse.Status = HipsResponseIndicator.OK;
                }
                else if (result.HipsResponse.Status == HipsResponseIndicator.InvalidIhi && Settings.Instance.BypassHIService == true)
                {
                    // We cannot check with the HI service so send the demographics to the PCEHR service
                    result.HipsResponse.Status = HipsResponseIndicator.OK;
                }

                if (!string.IsNullOrEmpty(patientMaster.Ihi))
                {
                    // CR 20209: If an IHI is to be used in registering an eHealth record,
                    // the clinical information system shall validate the IHI against the
                    // HI Service at the time of registration.
                    PatientIhiValidation validation = new PatientIhiValidation();
                    validation.RevalidateIhi(patientMaster, hospital, user);
                    patientAccess.ValidateLocalIhiInformation(patientMaster, result.HipsResponse);
                }
            }
            else
            {
                // Get the hospital for the selected patient
                result.HipsResponse = patientAccess.GetHospital(patientIdentifier, out hospital);
                if (result.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return result;
                }

                // Get the patient information for the selected patient
                result.HipsResponse = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);

                if (result.HipsResponse.Status == HipsResponseIndicator.InvalidIhi || result.HipsResponse.Status == HipsResponseIndicator.OK)
                {
                    // CR 20209: If an IHI is to be used in registering an eHealth record,
                    // the clinical information system shall validate the IHI against the
                    // HI Service at the time of registration.
                    PatientIhiValidation validation = new PatientIhiValidation();
                    try
                    {
                        validation.RevalidateIhi(patientMaster, hospital, user);
                    }
                    catch (PcehrServiceUnavailableException ex)
                    {
                        result.HipsResponse = ex.Detail;
                        return result;
                    }
                    catch (Exception ex)
                    {
                        result.HipsResponse.Status = HipsResponseIndicator.SystemError;
                        result.HipsResponse.HipsErrorMessage = ex.Message;
                        return result;
                    }
                    patientAccess.ValidateLocalIhiInformation(patientMaster, result.HipsResponse);
                }

                // Now that we have the patient data validate it against the PCEHR rules
                HipsResponse validationResponse = ValidatePatientAndRepresentative(patientMaster, representativeDemographics);

                if (validationResponse.Status != HipsResponseIndicator.OK)
                {
                    result.HipsResponse = validationResponse;
                    return result;
                }
            }

            // Check if we successfully got the patient information
            if (result.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return result;
            }
            else if (patientMaster.Ihi != null)
            {
                // Check for the existence of a PCEHR
                DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
                DoesPcehrExistResponse doesPcehrExistResponse = doesPcehrExist.PcehrExists(null, hospital, patientMaster, user);

                // If the patient already has a PCEHR return with an error
                if (doesPcehrExistResponse.PcehrExists.HasValue && doesPcehrExistResponse.PcehrExists.Value)
                {
                    result.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                    result.HipsResponse.HipsErrorMessage = ResponseStrings.PcehrExists;

                    return result;
                }
            }

            // Create the request for the registration call
            if (representativeDemographics != null)
            {
                result.HipsResponse = CreateRequest(patientMaster, representativeDemographics, identityType, indigenousStatus, documentConsent, correspondenceChannel, acceptedTermsAndConditions, representativeDeclaration, consentForm, out request);
            }
            else
            {
                result.HipsResponse = CreateRequest(patientMaster, identityType, indigenousStatus, documentConsent, correspondenceChannel, acceptedTermsAndConditions, consentForm, out request);
            }

            // Make the call to register the patient
            if (result.HipsResponse.Status == HipsResponseIndicator.OK)
            {
                result = _CallRegisterPcehr(patientIdentifier, patientMaster, hospital, request, user);
            }

            return result;
        }

        #endregion Methods

        #region Private Methods

        /// <summary>
        /// Makes the call to the PCEHR register function.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="request">The request.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>The PCEHR registration response.</returns>
        private static PcehrRegistrationResponse _CallRegisterPcehr(PatientIdentifierBase patientIdentifier, PatientMaster patientMaster, Hospital hospital, registerPCEHR request, UserDetails user)
        {
            CommonPcehrHeader pcehrHeader;

            PcehrRegistrationResponse result = new PcehrRegistrationResponse();
            result.HipsResponse = new HipsResponse(HipsResponseIndicator.SystemError);

            // Set up the headers for the SOAP call
            pcehrHeader = Helpers.GetHeader(patientIdentifier, patientMaster.Ihi, user, hospital);

            X509Certificate2 signingCert = Helpers.GetConnectionCertificate(hospital);
            Uri url = Helpers.GetRegisterPcehrUrl();

            RegisterPCEHRClient registerPcehrClient = new RegisterPCEHRClient(url, signingCert, signingCert);
            ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

            if (Settings.Instance.AvoidProxy)
            {
                System.Reflection.FieldInfo field = registerPcehrClient.GetType().GetField("registerPcehrClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                RegisterPCEHRPortTypeClient ppptc = field.GetValue(registerPcehrClient) as RegisterPCEHRPortTypeClient;
                System.ServiceModel.WSHttpBinding binding = ppptc.Endpoint.Binding as System.ServiceModel.WSHttpBinding;
                binding.UseDefaultWebProxy = false;
            }
            StringBuilder errorMessages = new StringBuilder();

            try
            {
                // Invoke the service
                registerPCEHRResponse response = registerPcehrClient.RegisterPCEHR(pcehrHeader, request);

                // Check the response
                if (response.responseStatus.code == "PCEHR_SUCCESS")
                {
                    result.HipsResponse.Status = HipsResponseIndicator.OK;
                    // If we request an IVC response then return the data.
                    // Note: If the patient has been previously subscribed
                    // and then unsubscribe an IVC code and expiry date
                    // will not be returned.
                    if (request.assertions.ivcCorrespondence.channel == registerPCEHRAssertionsIvcCorrespondenceChannel.response
                        && response.ivcDetails != null)
                    {
                        result.IvcCode = response.ivcDetails.code;
                        result.IvcExpiryDate = response.ivcDetails.expiryDate;
                    }
                }
                else
                {
                    // If an error was received then return it to the calling application.
                    result.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                    result.HipsResponse.ResponseCode = response.responseStatus.code;
                    result.HipsResponse.ResponseCodeDescription = response.responseStatus.description;
                    result.HipsResponse.ResponseCodeDetails = response.responseStatus.details;
                }
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.RegisterPCEHR.StandardErrorType> fe)
            {
                errorMessages.Append(string.Format(ResponseStrings.ErrorCodeAndDetailsFormat, fe.Detail.errorCode, fe.Detail.message));
                HipsResponse faultResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                string[] messageParts = fe.Detail.message.Split(new string[] { " - " }, StringSplitOptions.RemoveEmptyEntries);
                faultResponse.ResponseCode = messageParts[0];
                if (messageParts.Length > 1)
                {
                    faultResponse.ResponseCodeDescription = messageParts[1];
                }
                faultResponse.ResponseCodeDetails = fe.Detail.errorCode.ToString();
                result.HipsResponse = faultResponse;
                string logMessage = string.Format(LogMessages.ErrorUnableToConnectRegisterPatient);
                EventLogger.WriteLog(logMessage, new Exception(errorMessages.ToString()), user, LogMessage.HIPS_MESSAGE_140);
            }
            catch (FaultException fe)
            {
                if (fe.InnerException != null)
                {
                    errorMessages.AppendLine(fe.InnerException.Message);
                    result.HipsResponse.ResponseCodeDetails = fe.InnerException.Message;
                }
                HipsResponse faultResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                faultResponse.ResponseCode = fe.Reason.ToString();
                faultResponse.ResponseCodeDescription = fe.Message;
                result.HipsResponse = faultResponse;
                string logMessage = string.Format(LogMessages.ErrorUnableToConnectRegisterPatient);
                EventLogger.WriteLog(logMessage, fe, user, LogMessage.HIPS_MESSAGE_140);
            }
            //catch (FaultException<StandardErrorType> ex)
            //{
            //    errorMessages.Append(string.Format(ResponseStrings.ErrorCodeAndDetailsFormat, ex.Detail.errorCode, ex.Detail.message));
            //    result.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
            //    result.HipsResponse.HipsErrorMessage = ResponseStrings.UnableToConnect;
            //    result.HipsResponse.ResponseCode = ex.Detail.errorCode.ToString();
            //    result.HipsResponse.ResponseCodeDescription = ex.Detail.message;
            //    string logMessage = string.Format(LogMessages.ErrorUnableToConnectRegisterPatient);

            //    EventLogger.WriteLog(logMessage, new Exception(errorMessages.ToString()), user, LogMessage.HIPS_MESSAGE_140);
            //}
            catch (Exception ex)
            {
                if (ex.InnerException != null)
                {
                    errorMessages.AppendLine(ex.InnerException.Message);
                    result.HipsResponse.ResponseCodeDetails = ex.InnerException.Message;
                }
                errorMessages.Append(ex.Message);
                result.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                result.HipsResponse.HipsErrorMessage = ResponseStrings.UnableToConnect;
                result.HipsResponse.ResponseCodeDescription = ex.Message;
                string logMessage = string.Format(LogMessages.ErrorUnableToConnectRegisterPatient);

                EventLogger.WriteLog(logMessage, ex, user, LogMessage.HIPS_MESSAGE_141);
            }
            finally
            {
                registerPcehrClient.Close();
            }

            InsertRegistrationAudit(patientMaster, user, hospital, AuditOperationNames.RegisterPcehr, result.HipsResponse, registerPcehrClient);

            if (result.HipsResponse.Status == HipsResponseIndicator.OK && patientMaster.Ihi != null)
            {
                // Confirm that the PCEHR exists and record its existence
                DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
                DoesPcehrExistResponse doesPcehrExistResponse = doesPcehrExist.PcehrExists(null, hospital, patientMaster, user);
            }

            return result;
        }

        /// <summary>
        /// Creates the request.
        /// </summary>
        /// <param name="identityType">Type of the identity.</param>
        /// <param name="indigenousStatus">The indigenous status.</param>
        /// <param name="documentConsent">The document consent.</param>
        /// <param name="correspondenceChannel">The correspondence channel.</param>
        /// <param name="acceptedTermsAndConditions">if set to <c>true</c> [accepted terms and conditions].</param>
        /// <param name="consentForm">The consent form.</param>
        /// <param name="request">The request.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        private static HipsResponse CreateRequest(PatientMaster patientMaster, EvidenceOfIdentityType identityType, IndigenousStatus indigenousStatus, List<RegistrationDocumentConsent> documentConsent, RegistrationCorrespondenceChannel correspondenceChannel, bool acceptedTermsAndConditions, Attachment consentForm, out registerPCEHR request)
        {
            return CreateRequest(patientMaster, null, identityType, indigenousStatus, documentConsent, correspondenceChannel, acceptedTermsAndConditions, false, consentForm, out request);
        }

        /// <summary>
        /// Creates the request.
        /// </summary>
        /// <param name="representativeDemographics">The representative demographics.</param>
        /// <param name="identityType">Type of the identity.</param>
        /// <param name="indigenousStatus">The indigenous status.</param>
        /// <param name="documentConsent">The document consent.</param>
        /// <param name="correspondenceChannel">The correspondence channel.</param>
        /// <param name="acceptedTermsAndConditions">if set to <c>true</c> [accepted terms and conditions].</param>
        /// <param name="consentForm">The consent form.</param>
        /// <param name="request">The request.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        private static HipsResponse CreateRequest(PatientMaster patientMaster, Demographic representativeDemographics, EvidenceOfIdentityType identityType, IndigenousStatus indigenousStatus, List<RegistrationDocumentConsent> documentConsent, RegistrationCorrespondenceChannel correspondenceChannel, bool acceptedTermsAndConditions, bool representativeDeclaration, Attachment consentForm, out registerPCEHR request)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            request = new registerPCEHR();

            // If we don't have a IHI then need to set the demographics
            if (patientMaster.Ihi == null)
            {
                request.individual = SetIndividual(patientMaster);
            }

            // Create the registration assertions
            request.assertions = new registerPCEHRAssertions();

            if (representativeDemographics != null)
            {
                request.representative = SetRepresentative(representativeDemographics);
                request.assertions.representativeDeclaration = representativeDeclaration;
                request.assertions.representativeDeclarationSpecified = true;
            }

            request.assertions.ivcCorrespondence = SetIvcCorrespondence(correspondenceChannel);

            request.assertions.documentConsent = SetDocumentConsent(documentConsent);

            request.assertions.identity = new registerPCEHRAssertionsIdentity();
            request.assertions.identity.evidenceOfIdentity = SetEvidenceOfIdentity(identityType);

            request.assertions.identity.indigenousStatus = SetIndigenousStatus(indigenousStatus);

            // Set the consent form
            if (consentForm != null)
            {
                request.assertions.identity.signedConsentForm = consentForm.Contents;
            }

            // Set the terms and conditions
            request.assertions.acceptedTermsAndConditions = acceptedTermsAndConditions;

            return response;
        }

        /// <summary>
        /// Gets and update the patient master record.
        /// </summary>
        /// <param name="patientAccess">The patient access.</param>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <param name="patientMasterId">The patient master identifier.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        private static HipsResponse GetAndUpdatePatient(PatientAccess patientAccess, Demographic patientIdentifier, int patientMasterId, Hospital hospital, UserDetails user, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response;

            PatientMasterId patientMasterIdentifier = new PatientMasterId(patientMasterId, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);

            response = patientAccess.GetPatient(patientMasterIdentifier, hospital, out hospitalPatient, out patientMaster);

            if (response.Status == HipsResponseIndicator.OK)
            {
                patientMaster.SetNewCurrentName(-1, patientIdentifier.GivenName, patientIdentifier.FamilyName, -1);
                patientMaster.DateOfBirth = patientIdentifier.DateOfBirth;
                patientMaster.MedicareNumber = patientIdentifier.MedicareNumber;
                patientMaster.MedicareIrn = patientIdentifier.MedicareIrn;
                patientMaster.DvaNumber = patientIdentifier.DvaNumber;

                SqlTransaction transaction = null;

                PatientMasterDl patientMasterDl = new PatientMasterDl(user);

                if (!patientMasterDl.Update(patientMaster, transaction))
                {
                    response.Status = HipsResponseIndicator.SystemError;
                    response.HipsErrorMessage = "Failed to update patient details";

                    return response;
                }

                patientAccess.ValidateLocalIhiInformation(patientMaster, response);
            }

            return response;
        }

        /// <summary>
        /// Inserts the registration audit.
        /// </summary>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="auditOperationName">Name of the audit operation.</param>
        /// <param name="response">The response.</param>
        /// <param name="pcehrClient">The PCEHR client.</param>
        /// <returns>Whether the registration audit was inserted.</returns>
        private static bool InsertRegistrationAudit(PatientMaster patientMaster, UserDetails user, Hospital hospital, string auditOperationName, HipsResponse response, RegisterPCEHRClient pcehrClient)
        {
            PatientMaster patientMasterClone = null;

            if (auditOperationName == AuditOperationNames.RegisterPcehr && !string.IsNullOrEmpty(pcehrClient.SoapMessages.SoapResponse))
            {
                XmlDocument doc = new XmlDocument();
                doc.Load(new StringReader(pcehrClient.SoapMessages.SoapResponse));
                XmlNamespaceManager xnm = new XmlNamespaceManager(new NameTable());
                xnm.AddNamespace("s", XmlStringMap.SoapEnvelopeNamespace);
                xnm.AddNamespace("p", XmlStringMap.PcehrRegistrationNamespace);
                XmlNode accessCodeNode = doc.SelectSingleNode(XmlStringMap.IvcCodeXPath, xnm);
                if (accessCodeNode != null)
                {
                    accessCodeNode.InnerText = XmlStringMap.IvcCodeRedacted;
                }
                StringWriter writer = new StringWriter();
                doc.Save(writer);
                pcehrClient.SoapMessages.SoapResponse = writer.ToString();
            }

            // We need to store a value in the IHI field of the Audit table.
            // As we don't have an IHI we will add Visitor to the log.
            if (patientMaster.Ihi == null)
            {
                patientMasterClone = patientMaster.Clone();

                patientMasterClone.Ihi = "Visitor";

                return Helpers.InsertAudit(patientMasterClone, user, hospital, AuditOperationNames.RegisterPcehr, response, pcehrClient.SoapMessages);
            }
            else
            {
                return Helpers.InsertAudit(patientMaster, user, hospital, AuditOperationNames.RegisterPcehr, response, pcehrClient.SoapMessages);
            }
        }

        /// <summary>
        /// Sets the document consent.
        /// </summary>
        /// <param name="documentConsent">The document consent.</param>
        /// <returns></returns>
        private static registerPCEHRAssertionsDocument[] SetDocumentConsent(List<RegistrationDocumentConsent> documentConsent)
        {
            // Select consented documents
            registerPCEHRAssertionsDocument[] documents = new registerPCEHRAssertionsDocument[documentConsent.Count];

            for (int i = 0; i < documentConsent.Count; i++)
            {
                registerPCEHRAssertionsDocument assertionDocument = new registerPCEHRAssertionsDocument();
                switch (documentConsent[i].DocumentConsentType)
                {
                    case RegistrationDocumentConsentType.ACIR:
                        assertionDocument.type = registerPCEHRAssertionsDocumentType.ACIR;
                        break;

                    case RegistrationDocumentConsentType.AODR:
                        assertionDocument.type = registerPCEHRAssertionsDocumentType.AODR;
                        break;

                    case RegistrationDocumentConsentType.MBS:
                        assertionDocument.type = registerPCEHRAssertionsDocumentType.MBS;
                        break;

                    case RegistrationDocumentConsentType.MBSPastAssimilation:
                        assertionDocument.type = registerPCEHRAssertionsDocumentType.MBSPastAssimilation;
                        break;

                    case RegistrationDocumentConsentType.PBS:
                        assertionDocument.type = registerPCEHRAssertionsDocumentType.PBS;
                        break;

                    case RegistrationDocumentConsentType.PBSPastAssimilation:
                        assertionDocument.type = registerPCEHRAssertionsDocumentType.PBSPastAssimilation;
                        break;
                }

                switch (documentConsent[i].DocumentConsentStatus)
                {
                    case RegistrationDocumentConsentStatus.ConsentGiven:
                        assertionDocument.status = registerPCEHRAssertionsDocumentStatus.ConsentGiven;
                        break;

                    case RegistrationDocumentConsentStatus.ConsentNotGiven:
                        assertionDocument.status = registerPCEHRAssertionsDocumentStatus.ConsentNotGiven;
                        break;
                }

                documents[i] = assertionDocument;
            }

            return documents;
        }

        /// <summary>
        /// Sets the evidence of identity.
        /// </summary>
        /// <param name="identityType">Type of the identity.</param>
        /// <returns></returns>
        private static registerPCEHRAssertionsIdentityEvidenceOfIdentity SetEvidenceOfIdentity(EvidenceOfIdentityType identityType)
        {
            // Set the identity verification
            registerPCEHRAssertionsIdentityEvidenceOfIdentity identity = new registerPCEHRAssertionsIdentityEvidenceOfIdentity();

            switch (identityType)
            {
                case EvidenceOfIdentityType.IdentityVerificationMethod1:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod1;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod2:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod2;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod3:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod3;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod4:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod4;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod5:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod5;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod6:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod6;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod7:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod7;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod8:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod8;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod9:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod9;
                    break;

                case EvidenceOfIdentityType.IdentityVerificationMethod10:
                    identity.type = registerPCEHRAssertionsIdentityEvidenceOfIdentityType.IdentityVerificationMethod10;
                    break;
            }

            return identity;
        }

        /// <summary>
        /// Sets the indigenous status.
        /// </summary>
        /// <param name="indigenousStatus">The indigenous status.</param>
        /// <returns></returns>
        private static registerPCEHRAssertionsIdentityIndigenousStatus SetIndigenousStatus(IndigenousStatus indigenousStatus)
        {
            // Set the indigenous status
            registerPCEHRAssertionsIdentityIndigenousStatus identityIndigenousStatus = new registerPCEHRAssertionsIdentityIndigenousStatus();

            switch (indigenousStatus)
            {
                case IndigenousStatus.AboriginalNotTorresStraitIslander:
                    identityIndigenousStatus = registerPCEHRAssertionsIdentityIndigenousStatus.Item1;
                    break;

                case IndigenousStatus.TorresStraitIslanderNotAboriginal:
                    identityIndigenousStatus = registerPCEHRAssertionsIdentityIndigenousStatus.Item2;
                    break;

                case IndigenousStatus.BothAboriginalAndTorresStraitIslander:
                    identityIndigenousStatus = registerPCEHRAssertionsIdentityIndigenousStatus.Item3;
                    break;

                case IndigenousStatus.NeitherAboriginalAndTorresStraitIslander:
                    identityIndigenousStatus = registerPCEHRAssertionsIdentityIndigenousStatus.Item4;
                    break;

                case IndigenousStatus.NotStatedInadequatelyDescribed:
                    identityIndigenousStatus = registerPCEHRAssertionsIdentityIndigenousStatus.Item9;
                    break;
            }

            return identityIndigenousStatus;
        }

        /// <summary>
        /// Sets the PCEHR individual from the Demographic data.
        /// </summary>
        /// <param name="patientMaster">The patients demographics.</param>
        /// <returns></returns>
        private static registerPCEHRIndividual SetIndividual(PatientMaster patientMaster)
        {
            registerPCEHRIndividual individual = new registerPCEHRIndividual();
            registerPCEHRIndividualDemographics demographics = new registerPCEHRIndividualDemographics();

            nameTypeSupp name = new nameTypeSupp();

            name.givenName = patientMaster.Names[0].GivenNames.Split(' ');
            name.familyName = patientMaster.Names[0].FamilyName;

            demographics.name = name;
            demographics.dateOfBirth = patientMaster.DateOfBirth;
            demographics.sex = SetSex(patientMaster.CurrentSexId);
            if (!String.IsNullOrEmpty(patientMaster.MedicareNumber))
            {
                demographics.medicareCardNumber = patientMaster.MedicareNumber;
                demographics.medicareIRN = patientMaster.MedicareIrn;
            }
            demographics.dvaFileNumber = patientMaster.DvaNumber;

            individual.demographics = demographics;

            return individual;
        }

        /// <summary>
        /// Sets the IVC correspondence.
        /// </summary>
        /// <param name="correspondenceChannel">The correspondence channel.</param>
        /// <returns></returns>
        private static registerPCEHRAssertionsIvcCorrespondence SetIvcCorrespondence(RegistrationCorrespondenceChannel correspondenceChannel)
        {
            // Select the IVC correspondence channel
            registerPCEHRAssertionsIvcCorrespondence ivcCorrespondence = new registerPCEHRAssertionsIvcCorrespondence();
            contactDetailsType contactDetails = new contactDetailsType();

            switch (correspondenceChannel.Channel)
            {
                case IvcCorrespondenceChannel.email:
                    ivcCorrespondence.channel = registerPCEHRAssertionsIvcCorrespondenceChannel.email;
                    contactDetails.emailAddress = correspondenceChannel.email;
                    break;

                case IvcCorrespondenceChannel.mail:
                    ivcCorrespondence.channel = registerPCEHRAssertionsIvcCorrespondenceChannel.mail;
                    break;

                case IvcCorrespondenceChannel.none:
                    ivcCorrespondence.channel = registerPCEHRAssertionsIvcCorrespondenceChannel.none;
                    break;

                case IvcCorrespondenceChannel.response:
                    ivcCorrespondence.channel = registerPCEHRAssertionsIvcCorrespondenceChannel.response;
                    break;

                case IvcCorrespondenceChannel.sms:
                    ivcCorrespondence.channel = registerPCEHRAssertionsIvcCorrespondenceChannel.sms;
                    contactDetails.mobilePhoneNumber = correspondenceChannel.phoneNumber;
                    break;
            }

            //contactDetails must not be in the request if the correspondenceChannel is empty
            if (!(String.IsNullOrEmpty(contactDetails.emailAddress) && String.IsNullOrEmpty(contactDetails.mobilePhoneNumber)))
            {
                ivcCorrespondence.contactDetails = contactDetails;
            }

            return ivcCorrespondence;
        }

        /// <summary>
        /// Sets the PCEHR representative from the Demographic data.
        /// </summary>
        /// <param name="representativeDemographics">The representative demographics.</param>
        /// <returns></returns>
        private static registerPCEHRRepresentative SetRepresentative(Demographic representativeDemographics)
        {
            registerPCEHRRepresentative representative = new registerPCEHRRepresentative();
            registerPCEHRRepresentativeDemographics demographics = new registerPCEHRRepresentativeDemographics();

            nameTypeSupp name = new nameTypeSupp();

            name.givenName = new string[] { representativeDemographics.GivenName };
            name.familyName = representativeDemographics.FamilyName;

            demographics.name = name;
            demographics.dateOfBirth = representativeDemographics.DateOfBirth;
            demographics.sex = SetSex(representativeDemographics.Sex);
            demographics.medicareCardNumber = representativeDemographics.MedicareNumber;
            demographics.medicareIRN = representativeDemographics.MedicareIrn;
            demographics.dvaFileNumber = representativeDemographics.DvaNumber;

            representative.demographics = demographics;

            return representative;
        }

        /// <summary>
        /// Converts the HIPS sex enumerator to a PCEHR one.
        /// </summary>
        /// <param name="demographicSex">The demographic sex.</param>
        /// <returns></returns>
        private static sex SetSex(SexEnumerator demographicSex)
        {
            sex pcehrSex;

            switch (demographicSex)
            {
                case SexEnumerator.Female:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.F;
                    break;

                case SexEnumerator.IntersexOrIndeterminate:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.I;
                    break;

                case SexEnumerator.Male:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.M;
                    break;

                default:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.N;
                    break;
            }

            return pcehrSex;
        }

        /// <summary>
        /// Converts the HIPS sex id to a PCEHR one.
        /// </summary>
        /// <param name="demographicSex">The demographic sex.</param>
        /// <returns></returns>
        private static sex SetSex(int sex)
        {
            sex pcehrSex;

            switch (sex)
            {
                case 2:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.F;
                    break;

                case 3:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.I;
                    break;

                case 1:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.M;
                    break;

                default:
                    pcehrSex = Nehta.VendorLibrary.PCEHR.RegisterPCEHR.sex.N;
                    break;
            }

            return pcehrSex;
        }

        #endregion Private Methods

        #region Validation

        /// <summary>
        /// Validates the consent form.
        /// </summary>
        /// <param name="consentForm">The consent form.</param>
        /// <returns></returns>
        private static HipsResponse ValidateConsentForm(Attachment consentForm)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            if (consentForm != null)
            {
                string fileName = consentForm.FileName;
                long size = consentForm.Contents.LongLength;

                if (size > PCEHR_CONSENT_FORM_SIZE_LIMIT)
                {
                    string message = string.Format(ResponseStrings.ConsentFormTooLarge, fileName, String.Format("{0:#,###0}", size));
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = message;
                    return response;
                }
                string extension;
                try
                {
                    if (fileName.Contains('/') || fileName.Contains('\\'))
                    {
                        response.Status = HipsResponseIndicator.ValidationError;
                        response.HipsErrorMessage = ResponseStrings.FilenameContainsPathSeperators;
                        return response;
                    }
                    extension = new FileInfo(fileName).Extension;
                }
                catch (ArgumentException)
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = string.Format(ResponseStrings.ConsentFormFilenameInvalid, fileName);
                    return response;
                }
                if (!PCEHR_CONSENT_FORM_EXTENSIONS.Contains(extension, StringComparer.CurrentCultureIgnoreCase))
                {
                    string message = string.Format(ResponseStrings.ConsentFormTypeDisallowed, fileName);
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = message;
                    return response;
                }
            }

            return response;
        }

        /// <summary>
        /// Validates the demographics of a patient.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="demographicName">The name to show in error messages.</param>
        /// <returns>Success (OK) or reason for failure (Validation Error)</returns>
        private static HipsResponse ValidateDemographics(Demographic patientIdentifier, string demographicName)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            if (string.IsNullOrEmpty(patientIdentifier.GivenName))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = String.Format("{0}: No given name provided.", demographicName);
                return response;
            }

            if (string.IsNullOrEmpty(patientIdentifier.FamilyName))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = String.Format("{0}: No family name provided.", demographicName);
                return response;
            }

            if (patientIdentifier.DateOfBirth > DateTime.Now)
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = String.Format("{0}: Cannot have future dated date of birth.", demographicName);
                return response;
            }

            if (patientIdentifier.DateOfBirth < DateTime.Now.AddYears(-140))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = String.Format("{0}: Date of birth cannot be more than 140 years ago.", demographicName);
                return response;
            }

            if (string.IsNullOrEmpty(patientIdentifier.MedicareNumber) && string.IsNullOrEmpty(patientIdentifier.DvaNumber))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = String.Format("{0}: Must include a Medicare or DVA number.", demographicName);
                return response;
            }

            if (!string.IsNullOrEmpty(patientIdentifier.MedicareNumber))
            {
                if (!Medicare.Validate(patientIdentifier.MedicareNumber).Value)
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = String.Format("{0}: Medicare number is incorrect.", demographicName);
                    return response;
                }

                if (int.Parse(patientIdentifier.MedicareIrn) < 1 || int.Parse(patientIdentifier.MedicareIrn) > 9)
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = String.Format("{0}: Medicare IRN must be a number between 1 and 9.", demographicName);
                    return response;
                }
            }

            return response;
        }

        /// <summary>
        /// Validates that all documents have a consent record.
        /// </summary>
        /// <param name="documentConsent">The document consent.</param>
        /// <returns>Success (OK) or reason for failure (Validation Error)</returns>
        private static HipsResponse ValidateDocumentConsent(List<RegistrationDocumentConsent> documentConsent)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            // Check if an entry for ACIR is present
            if (!documentConsent.Exists(d => d.DocumentConsentType == RegistrationDocumentConsentType.ACIR))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = "Must provide a response for ACIR documents consent.";
                return response;
            }

            // Check if an entry for AODR is present
            if (!documentConsent.Exists(d => d.DocumentConsentType == RegistrationDocumentConsentType.AODR))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = "Must provide a response for AODR documents consent.";
                return response;
            }

            // Check if an entry for MBS is present
            if (!documentConsent.Exists(d => d.DocumentConsentType == RegistrationDocumentConsentType.MBS))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = "Must provide a response for MBS documents consent.";
                return response;
            }
            else
            {
                // Check if an entry for MBS Past is present
                if (!documentConsent.Exists(d => d.DocumentConsentType == RegistrationDocumentConsentType.MBSPastAssimilation))
                {
                    if (documentConsent.Find(d => d.DocumentConsentType == RegistrationDocumentConsentType.MBS).DocumentConsentStatus == RegistrationDocumentConsentStatus.ConsentNotGiven)
                    {
                        // If the MBS is consent not given and we don't have a MBS Past record we can add one.
                        RegistrationDocumentConsent document = new RegistrationDocumentConsent();
                        document.DocumentConsentType = RegistrationDocumentConsentType.MBSPastAssimilation;
                        document.DocumentConsentStatus = RegistrationDocumentConsentStatus.ConsentNotGiven;

                        documentConsent.Add(document);
                    }
                    else
                    {
                        response.Status = HipsResponseIndicator.ValidationError;
                        response.HipsErrorMessage = "Must provide a response for MBS Past Assimilation documents consent.";
                        return response;
                    }
                }
            }

            // Check if an entry for PBS is present
            if (!documentConsent.Exists(d => d.DocumentConsentType == RegistrationDocumentConsentType.PBS))
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = "Must provide a response for PBS documents consent.";
                return response;
            }
            else
            {
                // Check if an entry for PBS Past is present
                if (!documentConsent.Exists(d => d.DocumentConsentType == RegistrationDocumentConsentType.PBSPastAssimilation))
                {
                    if (documentConsent.Find(d => d.DocumentConsentType == RegistrationDocumentConsentType.PBS).DocumentConsentStatus == RegistrationDocumentConsentStatus.ConsentNotGiven)
                    {
                        // If the PBS is consent not given and we don't have a PBS Past record we can add one.
                        RegistrationDocumentConsent document = new RegistrationDocumentConsent();
                        document.DocumentConsentType = RegistrationDocumentConsentType.PBSPastAssimilation;
                        document.DocumentConsentStatus = RegistrationDocumentConsentStatus.ConsentNotGiven;

                        documentConsent.Add(document);
                    }
                    else
                    {
                        response.Status = HipsResponseIndicator.ValidationError;
                        response.HipsErrorMessage = "Must provide a response for PBS Past Assimilation documents consent.";
                        return response;
                    }
                }
            }

            return response;
        }

        /// <summary>
        /// Validates the IVC correspondence.
        /// </summary>
        /// <param name="correspondenceChannel">The correspondence channel.</param>
        /// <returns>Success (OK) or reason for failure (Validation Error)</returns>
        private static HipsResponse ValidateIvcCorrespondence(RegistrationCorrespondenceChannel correspondenceChannel)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            if (correspondenceChannel == null || correspondenceChannel.Channel == IvcCorrespondenceChannel.notselected)
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = "IVC Correspondence Channel has not been specified.";

                return response;
            }

            if (correspondenceChannel.Channel == IvcCorrespondenceChannel.sms)
            {
                if (correspondenceChannel.phoneNumber == null || correspondenceChannel.phoneNumber == string.Empty)
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = "Mobile phone number is required for IVC SMS correspondence.";
                }
                else if (!Regex.IsMatch(correspondenceChannel.phoneNumber, @"^(\+?)(61)(4[0-9]{8})$|^([0+])(4[0-9]{8})$"))
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = "Invalid mobile phone number.";
                }
            }

            if (correspondenceChannel.Channel == IvcCorrespondenceChannel.email)
            {
                if (correspondenceChannel.email == null || correspondenceChannel.email == string.Empty)
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = "Email address is required for IVC email correspondence.";
                }
                else if (!Regex.IsMatch(correspondenceChannel.email, @"[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?"))
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = "Invalid email address.";
                }
            }

            return response;
        }

        /// <summary>
        /// Validates the patient and representative.
        /// </summary>
        /// <param name="patient">The patient.</param>
        /// <param name="representative">The representative.</param>
        /// <returns>Success (OK) or reason for failure (Validation Error)</returns>
        private static HipsResponse ValidatePatientAndRepresentative(PatientMaster patient, Demographic representative)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            if (representative != null)
            {
                //
                if (patient.RegisteredDateOfBirth < DateTime.Now.AddYears(-18))
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = String.Format("Dependant cannot be older than 18 years.");
                    return response;
                }

                if (patient.RegisteredDateOfBirth < representative.DateOfBirth.AddYears(14))
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = String.Format("There cannot be less than a 14 year age gap between the dependant and the representative.");
                    return response;
                }

                if (string.IsNullOrEmpty(representative.MedicareNumber))
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = String.Format("The representatives Medicare number must be provided.");
                    return response;
                }

                // If the patient data is from a validated IHI then we don't have their Medicare Number
                if (!string.IsNullOrEmpty(patient.MedicareNumber) && patient.MedicareNumber != representative.MedicareNumber)
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = String.Format("The dependant and the representative must be on the same Medicare card.");
                    return response;
                }
            }
            else
            {
                if (patient.RegisteredDateOfBirth > DateTime.Now.AddYears(-14))
                {
                    response.Status = HipsResponseIndicator.ValidationError;
                    response.HipsErrorMessage = String.Format("An individual cannot be less than 14 years old.");
                    return response;
                }
            }

            return response;
        }

        /// <summary>
        /// Validates the representative declaration.
        /// </summary>
        /// <param name="representativeDemographics">The representative demographics.</param>
        /// <param name="representativeDeclaration">if set to <c>true</c> [representative declaration].</param>
        /// <returns>Success (OK) or reason for failure (Validation Error)</returns>
        private static HipsResponse ValidateRepresentativeDeclaration(Demographic representativeDemographics, bool representativeDeclaration)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            if (representativeDemographics != null && representativeDeclaration == false)
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = "Representative declaration is required for assisted registration.";
            }

            return response;
        }

        /// <summary>
        /// Validates the terms and conditions.
        /// </summary>
        /// <param name="acceptedTermsAndConditions">if set to <c>true</c> [accepted terms and conditions].</param>
        /// <returns>Success (OK) or reason for failure (Validation Error)</returns>
        private static HipsResponse ValidateTermsAndConditions(bool acceptedTermsAndConditions)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

            if (!acceptedTermsAndConditions)
            {
                response.Status = HipsResponseIndicator.ValidationError;
                response.HipsErrorMessage = "The latest terms and conditions have not been accepted.";
            }

            return response;
        }

        #endregion Validation
    }
}