﻿using System.Linq;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.CommonBusinessLogic
{
    public static class User
    {
        /// <summary>
        /// Adds the HPI-O and Authorised Employee Name and User ID from the Hospital.
        /// </summary>
        /// <param name="hospital">the hospital</param>
        /// <param name="user">the user details</param>
        /// <returns>whether the user details are complete</returns>
        static public bool PopulateAndValidateUser(Hospital hospital, UserDetails user)
        {
            bool isValid = true;
            switch (user.Role)
            {
                case UserRole.AuthorisedEmployee:
                    if (string.IsNullOrEmpty(user.AuthorisedEmployeeUserId))
                    {
                        user.AuthorisedEmployeeUserId = hospital.AuthorisedEmployeeUserId;
                        user.Name = hospital.AuthorisedEmployeeName;
                    }
                    break;

                case UserRole.InteractiveUser:
                    isValid &= !string.IsNullOrWhiteSpace(user.Login);
                    isValid &= !string.IsNullOrWhiteSpace(user.Domain);
                    break;

                case UserRole.ProviderIndividual:
                    isValid &= !string.IsNullOrWhiteSpace(user.HpiI);
                    break;
            }
            isValid &= !string.IsNullOrWhiteSpace(user.Name);
            return isValid;
        }

        /// <summary>
        /// Adds the HPI-O and Authorised Employee Name and User ID from the HPIO in the HealthProviderOrganisation table.
        /// </summary>
        /// <param name="hpo">The Health Provider Organisation object</param>
        /// <param name="user">the user details</param>
        /// <returns>whether the user details are complete</returns>
        static public bool PopulateAndValidateUserFromHpio(HealthProviderOrganisation hpo, UserDetails user)
        {
            bool isValid = true;
            switch (user.Role)
            {
                case UserRole.AuthorisedEmployee:
                    if (string.IsNullOrEmpty(user.AuthorisedEmployeeUserId))
                    {
                        user.AuthorisedEmployeeUserId = hpo.AuthorisedEmployeeUserId;
                        user.Name = hpo.AuthorisedEmployeeName;
                    }
                    break;

                case UserRole.InteractiveUser:
                    isValid &= !string.IsNullOrWhiteSpace(user.Login);
                    isValid &= !string.IsNullOrWhiteSpace(user.Domain);
                    break;

                case UserRole.ProviderIndividual:
                    isValid &= !string.IsNullOrWhiteSpace(user.HpiI);
                    break;
            }
            isValid &= !string.IsNullOrWhiteSpace(user.Name);
            return isValid;
        }

        /// <summary>
        /// Validates the user has all the required fields populated.
        /// </summary>
        /// <param name="user">The user details</param>
        /// <returns>whether valid</returns>
        static public bool ValidateUser(UserDetails user)
        {
            if (string.IsNullOrEmpty(user.Name))
            {
                return false;
            }
            switch (user.Role)
            {
                case UserRole.AuthorisedEmployee:
                    if (string.IsNullOrEmpty(user.AuthorisedEmployeeUserId))
                    {
                        return false;
                    }
                    break;

                case UserRole.InteractiveUser:
                    if (string.IsNullOrEmpty(user.Domain) || string.IsNullOrEmpty(user.Login))
                    {
                        return false;
                    }
                    break;

                case UserRole.ProviderIndividual:
                    if (string.IsNullOrEmpty(user.HpiI))
                    {
                        return false;
                    }
                    break;
            }
            return true;
        }
    }
}