﻿using System;
using System.Runtime.Serialization;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.CommonSchemas.PatientIdentifier
{
    /// <summary>
    /// The Demographic identifier is used for Assisted Registration of a person using manually-entered demographic information.
    /// Services other than Assisted Registration will reject this Patient Identifier.
    /// </summary>
    [Serializable]
    [DataContract]
    public class Demographic : PatientIdentifierBase
    {
        #region Private Members

        private DateTime dateOfBirth;
        private string dvaNumber;
        private string familyName;
        private string givenName;
        private string medicareIrn;
        private string medicareNumber;
        private SexEnumerator sex;

        #endregion Private Members

        #region Properties

        /// <summary>
        /// Gets or sets the person's date of birth.
        /// </summary>
        [DataMember]
        public DateTime DateOfBirth
        {
            get
            {
                return dateOfBirth;
            }
            set
            {
                dateOfBirth = value;
            }
        }

        /// <summary>
        /// Gets or sets the DVA file number on the person's repatriation health card.
        /// </summary>
        [DataMember]
        public string DvaNumber
        {
            get
            {
                return dvaNumber;
            }
            set
            {
                dvaNumber = value;
            }
        }

        /// <summary>
        /// Gets or sets the family name that appears on the person's Medicare or DVA card.
        /// </summary>
        [DataMember]
        public string FamilyName
        {
            get
            {
                return familyName;
            }
            set
            {
                familyName = value;
            }
        }

        /// <summary>
        /// Gets or sets the given name that appears on the person's Medicare or DVA card.
        /// </summary>
        [DataMember]
        public string GivenName
        {
            get
            {
                return givenName;
            }
            set
            {
                givenName = value;
            }
        }

        /// <summary>
        /// Gets or sets the 1-digit IRN for this person on the Medicare card.
        /// </summary>
        [DataMember]
        public string MedicareIrn
        {
            get
            {
                return medicareIrn;
            }
            set
            {
                medicareIrn = value;
            }
        }

        /// <summary>
        /// Gets or sets the 10-digit card number on the person's Medicare card.
        /// </summary>
        [DataMember]
        public string MedicareNumber
        {
            get
            {
                return medicareNumber;
            }
            set
            {
                medicareNumber = value;
            }
        }

        /// <summary>
        /// Gets or sets the person's sex.
        /// </summary>
        [DataMember]
        public SexEnumerator Sex
        {
            get
            {
                return sex;
            }
            set
            {
                sex = value;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Initialises a new instance of the <see cref="Demographic" /> class. This basic constructor is used by AutoMapper.
        /// </summary>
        public Demographic()
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="Demographic" /> class. This constructor is used for people with a Medicare card number and Individual Reference Number (IRN).
        /// </summary>
        /// <param name="familyName">The family name from the person's Medicare card.</param>
        /// <param name="givenName">The given name from the person's Medicare card.</param>
        /// <param name="dateOfBirth">The date of birth reported by the person.</param>
        /// <param name="sex">The sex reported by the person.</param>
        /// <param name="medicareNumber">The 10-digit card number on the person's Medicare card.</param>
        /// <param name="medicareIrn">The one-digit individual reference number on the person's Medicare card.</param>
        /// <param name="hospitalCode">A code identifying the hospital in which this person completed Assisted Registration.</param>
        /// <param name="hospitalCodeSystem">A code identifying the system under which the hospital code was allocated</param>
        public Demographic(string familyName, string givenName, DateTime dateOfBirth, SexEnumerator sex, string medicareNumber, string medicareIrn, string hospitalCode, string hospitalCodeSystem)
            : base(hospitalCode, hospitalCodeSystem)
        {
            this.familyName = familyName;
            this.givenName = givenName;
            this.dateOfBirth = dateOfBirth;
            this.sex = sex;
            this.medicareNumber = medicareNumber;
            this.medicareIrn = medicareIrn;
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="Demographic" /> class. This constructor is used for people with a DVA file number.
        /// </summary>
        /// <param name="familyName">The family name on the person's DVA card.</param>
        /// <param name="givenName">The given name on the person's DVA card.</param>
        /// <param name="dateOfBirth">The date of birth reported by the person.</param>
        /// <param name="sex">The sex reported by the person.</param>
        /// <param name="dvaNumber">The DVA file number on the person's DVA card.</param>
        /// <param name="hospitalCode">A code identifying the hospital in which this person completed Assisted Registration.</param>
        /// <param name="hospitalCodeSystem">A code identifying the system under which the hospital code was allocated</param>
        public Demographic(string familyName, string givenName, DateTime dateOfBirth, SexEnumerator sex, string dvaNumber, string hospitalCode, string hospitalCodeSystem)
            : base(hospitalCode, hospitalCodeSystem)
        {
            this.familyName = familyName;
            this.givenName = givenName;
            this.dateOfBirth = dateOfBirth;
            this.sex = sex;
            this.dvaNumber = dvaNumber;
        }

        /// <summary>
        /// Represents the Demographic patient identifier as a string for logging and debugging purposes.
        /// </summary>
        /// <returns>A string representation of the object.</returns>
        public override string ToString()
        {
            return string.Format("Family Name {0} Given Name {1}", familyName, givenName);
        }

        #endregion Methods
    }
}