﻿using System.Runtime.Serialization;

namespace HIPS.CommonSchemas.PatientIdentifier
{
    /// <summary>
    /// This Patient Identifier is used for patients who are registered at the enterprise level.
    /// Their registration can be automatically extended from one facility to another.
    /// </summary>
    [DataContract]
    public class RegisteredEnterprisePatient : PatientIdentifierBase
    {
        #region Private Members

        /// <summary>
        /// The facility-level local identifier for the patient.
        /// </summary>
        private string mrn;

        /// <summary>
        /// The enterprise-wide local identifier for the patient.
        /// </summary>
        private string statePatientId;

        #endregion Private Members

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="RegisteredEnterprisePatient" /> class.
        /// This basic constructor is used by AutoMapper.
        /// </summary>
        public RegisteredEnterprisePatient()
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="RegisteredEnterprisePatient" /> class.
        /// This is a specific constructor for the Registered Enterprise Patient.
        /// </summary>
        /// <param name="mrn">The facility-level local identifier for the patient.</param>
        /// <param name="statePatientId">The enterprise-wide local identifier for the patient.</param>
        /// <param name="hospitalCode">A code identifying the hospital in which this patient received healthcare.</param>
        /// <param name="hospitalCodeSystem">A code identifying the system under which the hospital code was allocated.</param>
        public RegisteredEnterprisePatient(string mrn, string statePatientId, string hospitalCode, string hospitalCodeSystem)
            : base(hospitalCode, hospitalCodeSystem)
        {
            this.mrn = mrn;
            this.statePatientId = statePatientId;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the facility-level ID for the patient.
        /// </summary>
        [DataMember]
        public string Mrn
        {
            get
            {
                return this.mrn;
            }

            set
            {
                this.mrn = value;
            }
        }

        /// <summary>
        /// Gets or sets the enterprise-level ID for the patient.
        /// </summary>
        [DataMember]
        public string StatePatientId
        {
            get
            {
                return this.statePatientId;
            }

            set
            {
                this.statePatientId = value;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Represents the patient identifier as a string for logging and debugging purposes.
        /// </summary>
        /// <returns>A string representation of the object.</returns>
        public override string ToString()
        {
            return string.Format("MRN: {0}, SPI: {1}", this.mrn, this.statePatientId);
        }

        #endregion Methods
    }
}