﻿using System;
using System.Configuration;
using System.Linq;

namespace HIPS.Configuration
{
    /// <summary>
    /// This is a Singleton Object that will hold all the application settings
    /// From the App config file or even from the database
    /// </summary>
    public sealed class Settings
    {
        #region Private Properties

        private static volatile Settings instance;
        private static object syncRoot = new Object();

        private bool pumaEnabled;
        private bool patientMasterOverride;

        private int lookupRefreshSeconds;
        private bool avoidProxy = false;
        private bool deleteQueuedItemOnSuccess;
        private string cdaUserIdQualifierFormat;
        private string defaultDocumentFormatCode;
        private int ihiCleanupProcessMinutes;
        private int doesPCEHRExistTimeoutSeconds;
        private int documentProductionTimeoutSeconds;
        private int documentConsumptionTimeoutSeconds;
        private int ihiSearchTimeoutSeconds;
        private int hpiiSearchTimeoutSeconds;
        private bool bypassHIService;
        private bool registeredDateOfBirthEnabled;
        private string pathologyReportDocumentFormatCode;
        private string diagnosticImagingReportDocumentFormatCode;
        private string configuredDirectory;

        private string mllpEndpoint;
        private string fileLocation;
        private bool useHL7MessageDateTime;

        // All of the Mock times should be used in TEST and DEV only. This replaces an actual call to the PCEHR or HI service with the wait time in seconds
        private int mockPcehrServiceDocumentUploadWaitSeconds;
        private int mockPcehrServiceDocumentRemoveWaitSeconds;
        private int mockPcehrServicePCEHRExistsWaitSeconds;
        private int mockHiServiceIHISearchWaitSeconds;
        private int mockHiServiceHPIISearchWaitSeconds;

        #endregion Private Properties

        #region Constructors

        /// <summary>
        /// Prevents a default instance of the <see cref="Settings"/> class from being created.
        /// </summary>
        private Settings()
        {
            RefreshAppConfigSettings();
        }

        /// <summary>
        /// Gets the instance of the settings class.
        /// </summary>
        public static Settings Instance
        {
            get
            {
                if (instance == null)
                {
                    lock (syncRoot)
                    {
                        if (instance == null)
                            instance = new Settings();
                    }
                }
                return instance;
            }
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets a value indicating whether [puma enabled].
        /// </summary>
        /// <value>
        ///   <c>true</c> if [puma enabled]; otherwise, <c>false</c>.
        /// </value>
        public bool PumaEnabled
        {
            get
            {
                return this.pumaEnabled;
            }
        }

        /// <summary>
        /// Gets a value indicating whether [patientMasterOverride enabled].
        /// </summary>
        /// <value>
        ///   <c>true</c> if [patientMasterOverride enabled]; otherwise, <c>false</c>.
        /// </value>
        public bool PatientMasterOverride
        {
            get
            {
                return this.patientMasterOverride;
            }
        }

        /// <summary>
        /// Gets a value indicating whether to avoid using a proxy server.
        /// </summary>
        /// <value>
        ///   <c>true</c> if avoid proxy; otherwise, <c>false</c>.
        /// </value>
        public bool AvoidProxy
        {
            get
            {
                return this.avoidProxy;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds the list of static data can be cached for.
        /// </summary>
        /// <value>
        /// 	the number of seconds the list of static data can be cached for.
        /// </value>
        public int LookupRefreshSeconds
        {
            get
            {
                return this.lookupRefreshSeconds;
            }
        }

        /// <summary>
        /// Gets a value indicating whether to delete the PcehrMessageQueue record in the PCEHR Data Store after a successful queued operation.
        /// </summary>
        /// <value>
        /// Whether to delete the PcehrMessageQueue record.
        /// </value>
        public bool DeleteQueuedItemOnSuccess
        {
            get
            {
                return this.deleteQueuedItemOnSuccess;
            }
        }

        /// <summary>
        /// Gets a value indicating the format for the approver identifier in the CDA signature file.
        /// </summary>
        /// <value>
        /// The format for the approver identifier in the CDA signature file.
        /// </value>
        public string CdaUserIdQualifierFormat
        {
            get
            {
                return this.cdaUserIdQualifierFormat;
            }
        }

        /// <summary>
        /// Gets a value indicating the default document format code.
        /// </summary>
        /// <value>
        /// The default document format code.
        /// </value>
        public string DefaultDocumentFormatCode
        {
            get
            {
                return this.defaultDocumentFormatCode;
            }
        }

        /// <summary>
        /// Gets a value indicating the MLLP Endpoint.
        /// </summary>
        /// <value>
        /// The default MLLP Endpoint.
        /// </value>
        public string MLLPEndpoint
        {
            get
            {
                return this.mllpEndpoint;
            }
        }

        /// <summary>
        /// Gets a value indicating the File Location where the acknowledgement will be stored.
        /// </summary>
        /// <value>
        /// The default File Location where the acknowledgement will be stored.
        /// </value>
        public string FileLocation
        {
            get
            {
                return this.fileLocation;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of minutes to wait between runs of the IHI cleanup process.
        /// </summary>
        /// <value>
        /// The number of minutes to wait between runs of the IHI cleanup process.
        /// </value>
        public int IhiCleanupProcessMinutes
        {
            get
            {
                return this.ihiCleanupProcessMinutes;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the DoesPCEHRExist Service.
        /// </summary>
        /// <value>
        /// The number of seconds before the operation times out
        /// </value>
        public int DoesPCEHRExistTimeoutSeconds
        {
            get
            {
                return this.doesPCEHRExistTimeoutSeconds;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the Document Production Services
        /// - UploadDocument, RemoveDocument.
        /// </summary>
        /// <value>
        /// The number of seconds before the operation times out
        /// </value>
        public int DocumentProductionTimeoutSeconds
        {
            get
            {
                return this.documentProductionTimeoutSeconds;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the Document Consumption Services
        /// - GetDocument, GetDocumentList, GetChangeHistoryView, GainPCEHRAccess
        /// </summary>
        /// <value>
        /// The number of seconds before the operation times out
        /// </value>
        public int DocumentConsumptionTimeoutSeconds
        {
            get
            {
                return this.documentConsumptionTimeoutSeconds;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the IHI Search
        /// </summary>
        /// <value>
        /// The number of seconds before the operation times out
        /// </value>
        public int IhiSearchTimeoutSeconds
        {
            get
            {
                return this.ihiSearchTimeoutSeconds;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the HPII Search
        /// </summary>
        /// <value>
        /// The number of seconds before the operation times out
        /// </value>
        public int HpiiSearchTimeoutSeconds
        {
            get
            {
                return this.hpiiSearchTimeoutSeconds;
            }
        }

        /// <summary>
        /// Gets a value indicating whether the system should bypass the HI service.
        /// </summary>
        /// <value>
        ///   <c>true</c> if [bypass HI service]; otherwise, <c>false</c>.
        /// </value>
        public bool BypassHIService
        {
            get
            {
                return this.bypassHIService;
            }
        }

        /// <summary>
        /// Gets a value indicating the value Registered Date of Birth
        /// </summary>
        /// <value>
        ///  <c>true</c> if [RegisteredDateOfBirthEnabled enabled]; otherwise, <c>false</c>.
        /// </value>
        public bool RegisteredDateOfBirthEnabled
        {
            get
            {
                return this.registeredDateOfBirthEnabled;
            }
        }
        
        /// <summary>
        /// Gets a value indicating the number of days to wait until the validation expires for the HPII.
        /// </summary>
        public int HpiiValidationPeriodDays { get; private set; }

        /// <summary>
        /// Gets a value indicating the default document format code for Diagnostic Imaging Report.
        /// </summary>
        /// <value>
        /// The default document format code  for Diagnostic Imaging Report.
        /// </value>
        public string DiagnosticImagingReportDocumentFormatCode
        {
            get
            {
                return this.diagnosticImagingReportDocumentFormatCode;
            }
        }

        /// <summary>
        /// Gets a value indicating the default Configured Directory for the PDF Report.
        /// </summary>
        /// <value>
        /// The default Configured Direcotry for the PDF Report.
        /// </value>
        public string ConfiguredDirectory
        {
            get
            {
                return this.configuredDirectory;
            }
        }

        /// <summary>
        /// Gets a value indicating the default document format code for Pathology Report.
        /// </summary>
        /// <value>
        /// The default document format code  for Pathology Report.
        /// </value>
        public string PathologyReportDocumentFormatCode
        {
            get
            {
                return this.pathologyReportDocumentFormatCode;
            }
        }

        /// <summary>
        /// Flag to indicate if the HL7MessageDateTime should be used for comparison when checking if earlier pending messages
        /// </summary>
        public bool UseHL7MessageDateTime
        {
            get
            {
                return this.useHL7MessageDateTime;
            }
        }

        /// <summary>
        /// Used in DEV or TEST only to mock the Pcehr Document Upload PCEHR call
        /// </summary>
        public int MockPcehrServiceDocumentUploadWaitSeconds
        {
            get
            {
                return this.mockPcehrServiceDocumentUploadWaitSeconds;
            }
        }
        /// <summary>
        /// Used in DEV or TEST only to mock the Pcehr Document Remove PCEHR call
        /// </summary>
        public int MockPcehrServiceDocumentRemoveWaitSeconds
        {
            get
            {
                return this.mockPcehrServiceDocumentRemoveWaitSeconds;
            }
        }
        /// <summary>
        /// Used in DEV or TEST only to mock the Pcehr PCEHR Exists call
        /// </summary>
        public int MockPcehrServicePCEHRExistsWaitSeconds
        {
            get
            {
                return this.mockPcehrServicePCEHRExistsWaitSeconds;
            }
        }
        /// <summary>
        /// Used in DEV or TEST only to mock the HI IHI Search call
        /// </summary>
        public int MockHiServiceIHISearchWaitSeconds
        {
            get
            {
                return this.mockHiServiceIHISearchWaitSeconds;
            }
        }
        /// <summary>
        /// Used in DEV or TEST only to mock the Pcehr HII Search call
        /// </summary>
        public int MockHiServiceHpiiSearchWaitSeconds
        {
            get
            {
                return this.mockHiServiceHPIISearchWaitSeconds;
            }
        }
        #endregion Properties

        #region Methods

        /// <summary>
        /// Gets the latest settings from the configuration file.
        /// </summary>
        public void RefreshAppConfigSettings()
        {
            GetPumaEnabled();
            GetPatientMasterOverride();

            GetAvoidProxy();
            GetLookupRefreshSeconds();
            GetDeleteQueuedItemOnSuccess();
            GetCdaUserIdQualifierFormat();
            GetDefaultDocumentFormatCode();
            GetIhiCleanupProcessMinutes();
            GetDoesPCEHRExistTimeoutSeconds();
            GetDocumentConsumptionTimeoutSeconds();
            GetDocumentProductionTimeoutSeconds();
            GetIhiSearchTimeoutSeconds();
            GetHpiiSearchTimeoutSeconds();
            GetBypassHIService();
            GetRegisteredDateOfBirth();
            GetPathologyReportCode();
            GetDiagnosticImagingReportCode();
            GetConfiguredDirectory();
            this.GetHpiiValidationPeriodDays();
            this.GetUseHL7MessageDateTime();
            GetMLLPEndpoint();
            GetFileLocation();

            GetMockPcehrServiceDocumentUploadWaitSeconds();
            GetMockPcehrServiceDocumentRemoveWaitSeconds();
            GetMockPcehrServicePCEHRExistsWaitSeconds();
            GetMockHiServiceIHISearchWaitSeconds();
            GetMockHiServiceHPIISearchWaitSeconds();

        }

        /// <summary>
        /// Gets the puma enabled app config setting.
        /// The pumaEnabled configuration was made redundant and replaced with the patientMasterOverride, however they can be used interchangeably.
        /// It has been left only for backward compatibility with installations before 4.1.2

        /// </summary>
        private void GetPumaEnabled()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.PumaEnabled))
            {
                bool converted = bool.TryParse(ConfigurationManager.AppSettings[AppConfigResource.PumaEnabled].ToString(), out pumaEnabled);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToBool, AppConfigResource.PumaEnabled));
                }
            }
            else
            {
                pumaEnabled = false;
            }
        }

        /// <summary>
        /// Gets the patientMasterOverride enabled app config setting.
        /// </summary>
        private void GetPatientMasterOverride()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.PatientMasterOverride))
            {
                bool converted = bool.TryParse(ConfigurationManager.AppSettings[AppConfigResource.PatientMasterOverride].ToString(), out patientMasterOverride);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToBool, AppConfigResource.PatientMasterOverride));
                }
            }
            else
            {
                patientMasterOverride = false;
            }
            //patientMasterOverride and pumaEnabled are the same setting
            pumaEnabled = patientMasterOverride;
        }

        /// <summary>
        /// Gets the avoid proxy app config setting.
        /// </summary>
        private void GetAvoidProxy()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.AvoidProxy))
            {
                bool converted = bool.TryParse(ConfigurationManager.AppSettings[AppConfigResource.AvoidProxy].ToString(), out avoidProxy);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToBool, AppConfigResource.AvoidProxy));
                }
            }
            else
            {
                avoidProxy = false;
            }
        }

        /// <summary>
        /// Gets the number of seconds the list of static data can be cached for. Default value is 5 minutes (300 seconds)
        /// </summary>
        private void GetLookupRefreshSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.LookupRefreshSeconds))
            {
                bool converted = int.TryParse(ConfigurationManager.AppSettings[AppConfigResource.LookupRefreshSeconds].ToString(), out lookupRefreshSeconds);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToInteger, AppConfigResource.LookupRefreshSeconds));
                }
            }
            else
            {
                lookupRefreshSeconds = 300;
            }
        }

        /// <summary>
        /// Gets the value indicating whether to delete the PcehrMessageQueue record in the PCEHR Data Store after a successful queued operation. Default is true.
        /// </summary>
        private void GetDeleteQueuedItemOnSuccess()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.DeleteQueuedItemOnSuccess))
            {
                bool converted = bool.TryParse(ConfigurationManager.AppSettings[AppConfigResource.DeleteQueuedItemOnSuccess].ToString(), out deleteQueuedItemOnSuccess);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToBool, AppConfigResource.DeleteQueuedItemOnSuccess));
                }
            }
            else
            {
                deleteQueuedItemOnSuccess = true;
            }
        }

        /// <summary>
        /// Gets the value indicating whether to delete the PcehrMessageQueue record in the PCEHR Data Store after a successful queued operation. Default is one where domain.name must be replaced by local domain name.
        /// </summary>
        private void GetCdaUserIdQualifierFormat()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.CdaUserIdQualifierFormat))
            {
                cdaUserIdQualifierFormat = ConfigurationManager.AppSettings[AppConfigResource.CdaUserIdQualifierFormat].ToString();
            }
            else
            {
                cdaUserIdQualifierFormat = "http://ns.domain.name/id/cda/userid/1.0/{0}";
            }
        }

        /// <summary>
        /// Gets the value indicating the default document format code.
        /// </summary>
        private void GetDefaultDocumentFormatCode()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.DefaultDocumentFormatCode))
            {
                defaultDocumentFormatCode = ConfigurationManager.AppSettings[AppConfigResource.DefaultDocumentFormatCode].ToString();
            }
            else
            {
                defaultDocumentFormatCode = "1.2.36.1.2001.1006.1.20000.16";
            }
        }

        /// <summary>
        /// Gets the value indicating the MLLP Endpoint.
        /// </summary>
        private void GetMLLPEndpoint()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.MLLPUrl))
            {
                mllpEndpoint = ConfigurationManager.AppSettings[AppConfigResource.MLLPUrl].ToString();
            }
            
        }

        /// <summary>
        /// Gets the value indicating the File Location where the acknowledgement will be stored.
        /// </summary>
        private void GetFileLocation()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.FileLocation))
            {
                fileLocation = ConfigurationManager.AppSettings[AppConfigResource.FileLocation].ToString();
            }
            
        }

        /// <summary>
        /// Gets the value indicating the number of minutes to wait between runs of the IHI cleanup process.
        /// The default is 0 which means not to run the IHI cleanup process.
        /// </summary>
        private void GetIhiCleanupProcessMinutes()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.IhiCleanupProcessMinutes))
            {
                ihiCleanupProcessMinutes = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.IhiCleanupProcessMinutes].ToString());
            }
            else
            {
                ihiCleanupProcessMinutes = 0;
            }
        }

        /// <summary>
        /// Gets the value indicating the number of minutes to wait until the connection times out for the DoesPCEHRExist Service.
        /// The default is 60 seconds
        /// </summary>
        private void GetDoesPCEHRExistTimeoutSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.DoesPCEHRExistTimeoutSeconds))
            {
                doesPCEHRExistTimeoutSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.DoesPCEHRExistTimeoutSeconds].ToString());
            }
            else
            {
                doesPCEHRExistTimeoutSeconds = 60;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the Document Production Services
        /// The default is 300 seconds
        /// </summary>
        private void GetDocumentProductionTimeoutSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.DocumentProductionTimeoutSeconds))
            {
                documentProductionTimeoutSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.DocumentProductionTimeoutSeconds].ToString());
            }
            else
            {
                documentProductionTimeoutSeconds = 300;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the Document Consumption Services
        /// The default is 120 seconds
        /// </summary>
        private void GetDocumentConsumptionTimeoutSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.DocumentConsumptionTimeoutSeconds))
            {
                documentConsumptionTimeoutSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.DocumentConsumptionTimeoutSeconds].ToString());
            }
            else
            {
                documentConsumptionTimeoutSeconds = 120;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the IHI Search
        /// The default is 60 seconds
        /// </summary>
        private void GetIhiSearchTimeoutSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.IhiSearchTimeoutSeconds))
            {
                ihiSearchTimeoutSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.IhiSearchTimeoutSeconds].ToString());
            }
            else
            {
                ihiSearchTimeoutSeconds = 60;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of seconds to wait until the connection times out for the HPII Search
        /// The default is 60 seconds
        /// </summary>
        private void GetHpiiSearchTimeoutSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.HpiiSearchTimeoutSeconds))
            {
                hpiiSearchTimeoutSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.HpiiSearchTimeoutSeconds].ToString());
            }
            else
            {
                hpiiSearchTimeoutSeconds = 60;
            }
        }

        /// <summary>
        /// Gets the bypass HI Service app config setting.
        /// </summary>
        private void GetBypassHIService()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.BypassHIService))
            {
                bool converted = bool.TryParse(ConfigurationManager.AppSettings[AppConfigResource.BypassHIService].ToString(), out bypassHIService);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToBool, AppConfigResource.BypassHIService));
                }
            }
            else
            {
                bypassHIService = false;
            }
        }

        /// <summary>
        /// Gets the Registerd Date Of Birth app config setting.
        /// </summary>
        private void GetRegisteredDateOfBirth()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.RegisteredDateOfBirthEnabled))
            {
                bool converted = bool.TryParse(ConfigurationManager.AppSettings[AppConfigResource.RegisteredDateOfBirthEnabled].ToString(), out registeredDateOfBirthEnabled);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToBool, AppConfigResource.RegisteredDateOfBirthEnabled));
                }
            }
            else
            {
                registeredDateOfBirthEnabled = false;
            }
        }

        /// <summary>
        /// Gets a value indicating the number of days to wait until the validation expires for the HPII.
        /// The default value is 1 day.
        /// </summary>
        private void GetHpiiValidationPeriodDays()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.HpiiValidationPeriodDays))
            {
                this.HpiiValidationPeriodDays = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.HpiiValidationPeriodDays].ToString());
            }
            else
            {
                this.HpiiValidationPeriodDays = 1;
            }
        }

        /// <summary>
        /// Gets the value indicating the default document format code for Pathology Report.
        /// </summary>
        private void GetPathologyReportCode()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.PathologyReportDocumentFormatCode))
            {
                pathologyReportDocumentFormatCode = ConfigurationManager.AppSettings[AppConfigResource.PathologyReportDocumentFormatCode].ToString();
            }
            else
            {
                pathologyReportDocumentFormatCode = "1.2.36.1.2001.1006.1.220.4";
            }
        }

        /// <summary>
        /// Gets the value indicating the default document format code for Diagnostic Imaging Report.
        /// </summary>
        private void GetDiagnosticImagingReportCode()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.DiagnosticImagingReportDocumentFormatCode))
            {
                diagnosticImagingReportDocumentFormatCode = ConfigurationManager.AppSettings[AppConfigResource.DiagnosticImagingReportDocumentFormatCode].ToString();
            }
            else
            {
                diagnosticImagingReportDocumentFormatCode = "1.2.36.1.2001.1006.1.222.4";
            }
        }

        /// <summary>
        /// Gets the value indicating the default configured directory for the PDF report.
        /// </summary>
        private void GetConfiguredDirectory()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.ConfiguredDirectory))
            {
                configuredDirectory = ConfigurationManager.AppSettings[AppConfigResource.ConfiguredDirectory].ToString();
            }
            else
            {
                configuredDirectory = @"C:\";
            }
        }

        /// <summary>
        /// Gets the flag indicating if the HL7MessageDateTime should be used for pending message comparison or not
        /// </summary>
        private void GetUseHL7MessageDateTime()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.UseHL7MessageDateTime))
            {
                bool converted = bool.TryParse(ConfigurationManager.AppSettings[AppConfigResource.UseHL7MessageDateTime].ToString(), out useHL7MessageDateTime);
                if (!converted)
                {
                    throw new Exception(string.Format(AppConfigErrorsResource.AppConfigNotSetToBool, AppConfigResource.UseHL7MessageDateTime));
                }
            }
            else
            {
                useHL7MessageDateTime = true;
            }
        }

        /// <summary>
        /// Gets the amount of time to wait in seconds if the calls to the NEHTA library PCEHR Service are mock.
        /// </summary>
        private void GetMockPcehrServiceDocumentUploadWaitSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.MockPcehrServiceDocumentUploadWaitSeconds))
            {
                this.mockPcehrServiceDocumentUploadWaitSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.MockPcehrServiceDocumentUploadWaitSeconds].ToString());
            }
            else
            {
                this.mockPcehrServiceDocumentUploadWaitSeconds = -1;
            }
        }

        /// <summary>
        /// Gets the amount of time to wait in seconds if the calls to the NEHTA library PCEHR Service are mock.
        /// </summary>
        private void GetMockPcehrServiceDocumentRemoveWaitSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.MockPcehrServiceDocumentRemoveWaitSeconds))
            {
                this.mockPcehrServiceDocumentRemoveWaitSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.MockPcehrServiceDocumentRemoveWaitSeconds].ToString());
            }
            else
            {
                this.mockPcehrServiceDocumentRemoveWaitSeconds = -1;
            }
        }

        /// <summary>
        /// Gets the amount of time to wait in seconds if the calls to the NEHTA library PCEHR Service are mock.
        /// </summary>
        private void GetMockPcehrServicePCEHRExistsWaitSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.MockPcehrServicePCEHRExistsWaitSeconds))
            {
                this.mockPcehrServicePCEHRExistsWaitSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.MockPcehrServicePCEHRExistsWaitSeconds].ToString());
            }
            else
            {
                this.mockPcehrServicePCEHRExistsWaitSeconds = -1;
            }
        }
        /// <summary>
        /// Gets the amount of time to wait in seconds if the calls to the NEHTA library HI Service are mock.
        /// </summary>
        private void GetMockHiServiceIHISearchWaitSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.MockHiServiceIHISearchWaitSeconds))
            {
                this.mockHiServiceIHISearchWaitSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.MockHiServiceIHISearchWaitSeconds].ToString());
            }
            else
            {
                this.mockHiServiceIHISearchWaitSeconds = -1;
            }
        }
        /// <summary>
        /// Gets the amount of time to wait in seconds if the calls to the NEHTA library HI Service are mock.
        /// </summary>
        private void GetMockHiServiceHPIISearchWaitSeconds()
        {
            if (ConfigurationManager.AppSettings.AllKeys.Contains(AppConfigResource.MockHiServiceHPIISearchWaitSeconds))
            {
                this.mockHiServiceHPIISearchWaitSeconds = int.Parse(ConfigurationManager.AppSettings[AppConfigResource.MockHiServiceHPIISearchWaitSeconds].ToString());
            }
            else
            {
                this.mockHiServiceHPIISearchWaitSeconds = -1;
            }
        }
        #endregion Methods
    }
}