﻿using System;
using System.Runtime.Serialization;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PatientSchemas
{
    /// <summary>
    /// Represents an episode for a patient in hospital.
    /// </summary>
    [KnownType(typeof(EpisodePatientDetailsResponse))]
    [Serializable]
    [DataContract(Name = "EpisodePatientDetailsResponse")]
    public class EpisodePatientDetailsResponse
    {
        #region Private Members

        private SexEnumerator currentSex = SexEnumerator.NotStatedOrInadequatelyDescribed;
        private HipsResponse hipsResponse = new HipsResponse(HipsResponseIndicator.SystemError, null);

        #endregion Private Members

        /// <summary>
        /// The admission date of the patient.
        /// </summary>
        [DataMember]
        public DateTime AdmissionDate { get; set; }

        /// <summary>
        /// The admission reason of the patient.
        /// </summary>
        [DataMember]
        public string AdmissionReason { get; set; }

        /// <summary>
        /// The current Bed of the patient.
        /// </summary>
        [DataMember]
        public string Bed { get; set; }

        /// <summary>
        /// Whether the patient has withdrawn consent for the most recent matching episode.
        /// </summary>
        [DataMember]
        public bool? ConsentWithdrawn { get; set; }

        /// <summary>
        /// The patient's current sex.
        /// </summary>
        [DataMember]
        public SexEnumerator CurrentSex
        {
            get { return currentSex; }
            set { currentSex = value; }
        }

        /// <summary>
        /// The patients's Date of Birth.
        /// </summary>
        [DataMember]
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// The discharge date of the patient.
        /// </summary>
        [DataMember]
        public DateTime? DischargeDate { get; set; }

        /// <summary>
        /// The patients's DVA Number.
        /// </summary>
        [DataMember]
        public string DvaNumber { get; set; }

        /// <summary>
        /// The patients's family name.
        /// </summary>
        [DataMember]
        public string FamilyName { get; set; }

        /// <summary>
        /// The patient's given names.
        /// </summary>
        [DataMember]
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets the HipsResponse.
        /// Common Response Object.
        /// </summary>
        /// <value>
        /// The hipsResponse
        /// </value>
        [DataMember]
        public HipsResponse HipsResponse
        {
            get { return hipsResponse; }
            set { hipsResponse = value; }
        }

        /// <summary>
        /// The hospital code of the hospital in which the patient is or was admitted.
        /// </summary>
        [DataMember]
        public string HospitalCode { get; set; }

        /// <summary>
        /// The primary key of the hospital in which the patient is or was admitted.
        /// </summary>
        [DataMember]
        public int HospitalId { get; set; }

        /// <summary>
        /// The name of the hospital in which the patient is or was admitted.
        /// </summary>
        [DataMember]
        public string HospitalName { get; set; }

        /// <summary>
        /// The patients's Medicare IRN.
        /// </summary>
        [DataMember]
        public string MedicareIrn { get; set; }

        /// <summary>
        /// The patients's Medicare Number.
        /// </summary>
        [DataMember]
        public string MedicareNumber { get; set; }

        /// <summary>
        /// The patient's MRN at the hospital where the patient was admitted.
        /// </summary>
        [DataMember]
        public string Mrn { get; set; }

        /// <summary>
        /// The family name of the patient's responsible provider.
        /// </summary>
        [DataMember]
        public string ResponsibleProviderFamilyName { get; set; }

        /// <summary>
        /// The given names of the patient's responsible provider.
        /// </summary>
        [DataMember]
        public string ResponsibleProviderGivenNames { get; set; }

        /// <summary>
        /// The identifier of the patient's responsible provider.
        /// </summary>
        [DataMember]
        public string ResponsibleProviderIdentifier { get; set; }

        /// <summary>
        /// The suffix of the patient's responsible provider (such as 'Jr' or 'Sr')
        /// </summary>
        [DataMember]
        public string ResponsibleProviderSuffix { get; set; }

        /// <summary>
        /// The title of the patient's responsible provider (such as 'Dr', 'Ms' or 'Mr').
        /// </summary>
        [DataMember]
        public string ResponsibleProviderTitle { get; set; }

        /// <summary>
        /// The current Room of the patient.
        /// </summary>
        [DataMember]
        public string Room { get; set; }

        /// <summary>
        /// The source system episode id (visit number) for the patient
        /// </summary>
        [DataMember]
        public string SourceSystemEpisodeId { get; set; }

        /// <summary>
        /// The patient's name suffixes (like "Jr")
        /// </summary>
        [DataMember]
        public string Suffix { get; set; }

        /// <summary>
        /// The patients's titles (like "Dr", "Ms" or "Mr")
        /// </summary>
        [DataMember]
        public string Title { get; set; }

        /// <summary>
        /// The current Ward of the patient.
        /// </summary>
        [DataMember]
        public string Ward { get; set; }

        /// <summary>
        /// Withdrawal consent Audit Information
        /// </summary>
        [DataMember]
        public byte[] WithdrawalAuditInformation { get; set; }
    }
}