﻿using System;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Collections.Generic;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the HL7MessageLog table
    /// </summary>
    public class HL7MessageLogDl : BaseDl
    {
        #region Constructors

        public HL7MessageLogDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets the specified HL7MessageLog id.
        /// </summary>
        /// <param name="hl7MessageLogId">The HL7MessageLog id.</param>
        /// <returns></returns>
        public HL7MessageLog Get(int hl7MessageLogId)
        {
            HL7MessageLog result = new HL7MessageLog();
            try
            {
                using (SqlCommand command = GetSqlCommand("[hips].[HL7MessageLogGet]"))
                {
                    command.Parameters.Add(new SqlParameter("@HL7MessageLogId", hl7MessageLogId));
                    PopulateBusinessObject<HL7MessageLog>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorHL7MessageLogGet, ex, User, LogMessage.HIPS_MESSAGE_010);
            }
            return result;
        }

        #endregion Methods

        /// <summary>
        /// Deletes the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Delete(HL7MessageLog item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("[hips].[HL7MessageLogDelete]", transaction))
            {
                return base.Delete<HL7MessageLog>(item, command);
            }
        }

        /// <summary>
        /// Gets the HL7 Message Log entry with the specified sending application,
        /// sending facility, message control ID and date/time of message.
        /// </summary>
        /// <param name="sendingApplication">The sending application.</param>
        /// <param name="sendingFacility">The sending facility.</param>
        /// <param name="messageControlId">The HL7 message control id.</param>
        /// <param name="timeOfMessage">The date and time that the source system created the message</param>
        /// <param name="message">The message log entry.</param>
        /// <returns></returns>
        public bool Get(string sendingApplication, string sendingFacility, string messageControlId, DateTime? timeOfMessage, out HL7MessageLog message)
        {
            message = new HL7MessageLog();
            try
            {
                using (SqlCommand command = GetSqlCommand("[hips].[HL7MessageLogGet]"))
                {
                    command.Parameters.Add(new SqlParameter("@SendingApplication", sendingApplication));
                    command.Parameters.Add(new SqlParameter("@SendingFacility", sendingFacility));
                    command.Parameters.Add(new SqlParameter("@MessageControlId", messageControlId));
                    command.Parameters.Add(new SqlParameter("@MessageTime", timeOfMessage));
                    PopulateBusinessObject<HL7MessageLog>(command.ExecuteReader(), message);
                    command.Connection.Close();
                }
                return true;
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorHL7MessageLogGet, ex, User, LogMessage.HIPS_MESSAGE_131);
            }
            return false;
        }

        /// <summary>
        /// Gets all earlier pending HL7MessageLog items for a specific Patient
        /// </summary>
        /// <param name="hospitalPatientIdentifier">The specific hospital patient identifier</param>
        /// <param name="hospitalId">The hospital Id</param>
        /// <param name="messageDateTime">The existing message date time</param>
        /// <param name="useHL7MessageDateTime">True if using the HL7MessageDateTime column for comparison. False if using the DateCreated column for comparison</param>
        /// <returns>A list of earlier Pending Message Log Items for the specific Patient and message date & time</returns>
        public List<HL7MessageLog> GetEarlierPendingForPatient(string hospitalPatientIdentifier, int hospitalId, DateTime messageDateTime, bool? useHL7MessageDateTime)
        {
            List<HL7MessageLog> results = new List<HL7MessageLog>();
            try
            {
                using (SqlCommand command = GetSqlCommand("[hips].[HL7MessageLogGetPendingForPatient]"))
                {
                    command.Parameters.Add(new SqlParameter("@HospitalPatientIdentifier", hospitalPatientIdentifier));
                    command.Parameters.Add(new SqlParameter("@HospitalId", hospitalId));
                    command.Parameters.Add(new SqlParameter("@MessageDateTime", messageDateTime));
                    command.Parameters.Add(new SqlParameter("@UseHL7MessageDateTime", (useHL7MessageDateTime.HasValue ? useHL7MessageDateTime.Value : false)));
                    results = GetPopulatedBusinessList<HL7MessageLog>(command.ExecuteReader());
                    command.Connection.Close();
                }
                return results;
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorHL7MessageLogGet, ex, User, LogMessage.HIPS_MESSAGE_131);
            }
            return results;
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Insert(HL7MessageLog item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("[hips].[HL7MessageLogInsert]", transaction))
            {
                return base.Insert<HL7MessageLog>(item, command);
            }
        }

        /// <summary>
        /// Updates the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Update(HL7MessageLog item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("[hips].[HL7MessageLogUpdate]", transaction))
            {
                return base.Update<HL7MessageLog>(item, command);
            }
        }
    }
}