﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the name table
    /// </summary>
    public class PatientMasterNameDl : BaseDl
    {
        /// <summary>
        /// Gets all names for a patient.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="transaction">Optional transaction</param>
        /// <returns>List of names</returns>
        public List<PatientMasterName> GetAllByPatient(int patientMasterId, SqlTransaction transaction = null)
        {
            List<PatientMasterName> results = new List<PatientMasterName>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PatientMasterNameGet", transaction))
                {
                    command.Parameters.Add(new SqlParameter("@PatientMasterId", patientMasterId));
                    results = GetPopulatedBusinessList<PatientMasterName>(command.ExecuteReader());
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorPatientMasterNameGetAllByPatient, ex, User, LogMessage.HIPS_MESSAGE_009);
            }
            return results;
        }

        /// <summary>
        /// Updates the specified name.
        /// </summary>
        /// <param name="episode">The name.</param>
        /// <returns></returns>
        public bool Update(PatientMasterName episode, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.PatientMasterNameUpdate", transaction))
            {
                return base.Update<PatientMasterName>(episode, command);
            }
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Insert(PatientMasterName item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.PatientMasterNameInsert", transaction))
            {
                return base.Insert<PatientMasterName>(item, command);
            }
        }

        /// <summary>
        /// Deletes the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Delete(PatientMasterName item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.PatientMasterNameDelete", transaction))
            {
                return base.Delete<PatientMasterName>(item, command);
            }
        }

        /// <summary>
        /// Updates the specified names for a patient.
        /// </summary>
        /// <param name="names">The names.</param>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <returns></returns>
        public bool Update(List<PatientMasterName> names, int patientMasterId, SqlTransaction transaction = null)
        {
            if (names.Count == 0)
            {
                return true;
            }
            bool updateResult = true;

            List<PatientMasterName> originalNames = GetAllByPatient(patientMasterId, transaction);
            List<PatientMasterName> newNames = names.FindAll(name => name.PatientMasterNameId == null);
#if PUMA_CLIENT
            List<PatientMasterName> updatedNames = names.FindAll(name => name.PatientMasterNameId != null && name.IsDirty);
#else
            List<PatientMasterName> updatedNames = names.FindAll(name => name.PatientMasterNameId != null);
#endif

            IEnumerable<int?> originalIds = from result in originalNames
                                            select result.PatientMasterNameId;

            IEnumerable<int?> currentIds = from result in names
                                           select result.PatientMasterNameId;

            IEnumerable<int?> newIds = from result in newNames
                                       select result.PatientMasterNameId;

            IEnumerable<int?> deletedIds = originalIds.Except(currentIds);

            updateResult = DeleteNames(transaction, updateResult, originalNames, deletedIds);

            if (updateResult)
            {
                updateResult = InsertNames(patientMasterId, transaction, updateResult, newNames);
            }

            if (updateResult)
            {
                updateResult = UpdateNames(transaction, updateResult, updatedNames);
            }

            return updateResult;
        }

        /// <summary>
        /// Deletes the old names.
        /// </summary>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="originalNames">The original names.</param>
        /// <param name="deletedIds">The deleted ids.</param>
        /// <returns></returns>
        private bool DeleteNames(SqlTransaction transaction, bool updateResult, List<PatientMasterName> originalNames, IEnumerable<int?> deletedIds)
        {
            foreach (int deletedId in deletedIds)
            {
                updateResult = Delete(originalNames.Find(result => result.PatientMasterNameId == deletedId), transaction);
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Inserts the new names.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="newNames">The new names.</param>
        /// <returns></returns>
        private bool InsertNames(int patientMasterId, SqlTransaction transaction, bool updateResult, List<PatientMasterName> newNames)
        {
            foreach (PatientMasterName name in newNames)
            {
                name.PatientMasterId = patientMasterId;
                updateResult = Insert(name, transaction);
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Updates the existing names.
        /// </summary>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="updatedNames">The updated names.</param>
        /// <returns></returns>
        private bool UpdateNames(SqlTransaction transaction, bool updateResult, List<PatientMasterName> updatedNames)
        {
            foreach (PatientMasterName name in updatedNames)
            {
                updateResult = Update(name, transaction);
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }
    }
}