﻿// -----------------------------------------------------------------------
// <copyright file="Participant.cs"  company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.IhiSchemas.Exceptions;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas.Exceptions;

namespace HIPS.PcehrHiBusinessLogic.Ihi
{
    /// <summary>
    /// Performs IHI searches and validations that were missed while the HI Service was temporarily unavailable.
    /// </summary>
    public class PatientIhiCleanup
    {
        private UserDetails ihiCleanupUser;
        private PatientAccess patientAccess;

        /// <summary>
        /// If a service unavailable exception is caught for this number of patients in a row,
        /// assume the service is truly unavailable rather than just having transient errors,
        /// and skip the rest of this cleanup run.
        /// </summary>
        private const int SERVICE_UNAVAILABLE_THRESHOLD = 20;

        public PatientIhiCleanup()
        {
            ihiCleanupUser = new UserDetails();
            ihiCleanupUser.Role = UserRole.AuthorisedEmployee;
            ihiCleanupUser.AuthorisedEmployeeUserId = "PatientIhiCleanupUserId";
            ihiCleanupUser.Name = "PatientIhiCleanupUserName";
            patientAccess = new PatientAccess(ihiCleanupUser);
        }

        /// <summary>
        /// For each patient in the system with an IHI status of "ServiceUnavailable", attempts to validate the IHI for that patient.
        /// </summary>
        public void Cleanup()
        {
            List<PatientMasterIhi> patients = patientAccess.IhiDataAccess.GetAll(PcehrDataStore.Schemas.Enumerators.IhiStatus.ServiceUnavailable);
            if (patients.Count == 0)
            {
                return;
            }
            Random rand = new Random();
            patients = (from entry in patients.ToDictionary(a => rand.Next()) orderby entry.Key select entry.Value).ToList();
            PatientIhiValidation validation = new PatientIhiValidation();
            string infoMessage = string.Format(ConstantsResource.NumberOfPatientsToClean, patients.Count);
            EventLogger.WriteLog(ConstantsResource.InfoIhiCleanupStart, new Exception(infoMessage), ihiCleanupUser, LogMessage.HIPS_MESSAGE_074);
            int numberCleaned = 0, numberFailed = 0, numberConsecutiveUnavailable = 0;
            Dictionary<string, int> failureMessages = new Dictionary<string, int>();
            foreach (PatientMasterIhi patient in patients)
            {
                PatientMaster patientMaster;
                patientAccess.PatientMasterDataAccess.Get(patient.PatientMasterId.Value, out patientMaster);
                HospitalPatient hospitalPatient = patientAccess.HospitalPatientDataAccess.GetAll(null, patient.PatientMasterId.Value).FirstOrDefault();
                Hospital hospital = HospitalSingleton.Value.Find(hospitalPatient.HospitalId);
                UserDetails user = new UserDetails();
                user.Role = UserRole.AuthorisedEmployee;
                User.PopulateAndValidateUser(hospital, user);
                try
                {
                    validation.RevalidateIhi(patientMaster, hospital, user);
                    numberCleaned++;
                    numberConsecutiveUnavailable = 0;
                }
                catch (IhiServiceUnavailableException ex)
                {
                    HandleException(ref numberFailed, failureMessages, ex);
                    if (ReachedThresholdOnConsecutiveUnavailable(ref numberConsecutiveUnavailable))
                    {
                        break;
                    }
                }
                catch (PcehrServiceUnavailableException ex)
                {
                    HandleException(ref numberFailed, failureMessages, ex);
                    if (ReachedThresholdOnConsecutiveUnavailable(ref numberConsecutiveUnavailable))
                    {
                        break;
                    }
                }
                catch (Exception ex)
                {
                    // Another type of exception, not counted as service unavailable.
                    HandleException(ref numberFailed, failureMessages, ex);
                    numberConsecutiveUnavailable = 0;
                }
            }
            foreach (KeyValuePair<string, int> item in failureMessages)
            {
                string errorMessage = string.Format(ConstantsResource.NumberOfPatientsWithError, item.Value, item.Key);
                EventLogger.WriteLog(ConstantsResource.ErrorIhiCleanupEncounteredError, new Exception(errorMessage), ihiCleanupUser, LogMessage.HIPS_MESSAGE_075);
            }
            infoMessage = string.Format(ConstantsResource.NumberOfPatientsCleanedAndFailed, numberFailed + numberCleaned, numberFailed);
            EventLogger.WriteLog(ConstantsResource.InfoIhiCleanupFinish, new Exception(infoMessage), ihiCleanupUser, LogMessage.HIPS_MESSAGE_076);
        }

        /// <summary>
        /// Counts a case when the service was unavailable. Returns true if the background process should stop this run and go to sleep.
        /// </summary>
        /// <param name="numberConsecutiveUnavailable"></param>
        /// <returns>Whether the background process should stop the run</returns>
        private bool ReachedThresholdOnConsecutiveUnavailable(ref int numberConsecutiveUnavailable)
        {
            numberConsecutiveUnavailable++;
            bool reachedThreshold = numberConsecutiveUnavailable >= SERVICE_UNAVAILABLE_THRESHOLD;

            if (reachedThreshold)
            {
                string infoMessage = string.Format(ConstantsResource.TooManyConsecutiveServiceUnavailable, SERVICE_UNAVAILABLE_THRESHOLD);
                EventLogger.WriteLog(ConstantsResource.InfoIhiCleanupSleep, new Exception(infoMessage), ihiCleanupUser, LogMessage.HIPS_MESSAGE_077);
            }

            return reachedThreshold;
        }

        private static void HandleException(ref int numberFailed, Dictionary<string, int> failureMessages, Exception e)
        {
            if (failureMessages.ContainsKey(e.Message))
            {
                failureMessages[e.Message]++;
            }
            else
            {
                failureMessages.Add(e.Message, 1);
            }
            numberFailed++;
        }
    }
}