﻿using System;
using System.Configuration;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.IhiSchemas.Exceptions;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas.Exceptions;
using nehta.mcaR3.ConsumerSearchIHI;
using HIPS.CommonSchemas.Cda.ParticipatingIndividual;

namespace HIPS.PcehrHiBusinessLogic.Ihi
{
    public class PatientIhiValidation
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="PatientIhiValidation"/> class.
        /// </summary>
        /// <remarks>
        /// The value for the IHI validity period is extracted from the "IhiValidationPeriodDays" configuration setting.
        /// </remarks>
        public PatientIhiValidation()
        {
            this.IhiValidityPeriodHours = PatientIhiValidation.GetIhiValidationPeriodDays() * 24;
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="PatientIhiValidation"/> class.
        /// </summary>
        /// <param name="ihiValidityPeriodHours">The number of hours a locally stored IHI is considered valid for.</param>
        public PatientIhiValidation(int ihiValidityPeriodHours)
        {
            this.IhiValidityPeriodHours = ihiValidityPeriodHours;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the number of hours a locally stored IHI is considered valid for.
        /// </summary>
        private int IhiValidityPeriodHours { get; set; }

        #endregion

        #region Methods

        /// <summary>
        /// Gets the value of the "IhiValidationPeriodDays" configuration setting.
        /// </summary>
        /// <returns>Integer value containing the number of days configured.</returns>
        public static int GetIhiValidationPeriodDays()
        {
            int validationPeriod;
            int.TryParse(System.Configuration.ConfigurationManager.AppSettings["IhiValidationPeriodDays"], out validationPeriod);
            return validationPeriod;
        }

        /// <summary>
        /// Gets the value of the "UseStateIdentifierAsMrn" configuration setting.
        /// </summary>
        /// <returns>Flag indicating to Use State Identifier as MRN to create Hospital Patient.</returns>
        public bool GetUseStateIdentifierAsMrn()
        {
            bool useStateIdentifierAsMrn;
            bool.TryParse(System.Configuration.ConfigurationManager.AppSettings["UseStateIdentifierAsMrn"], out useStateIdentifierAsMrn);
            return useStateIdentifierAsMrn;
        }

        /// <summary>
        /// This is the external interface for clinical systems to request a
        /// valid IHI and registered demographics for use in a clinical
        /// document. This interface requires the date of birth to be provided
        /// as an additional identity check.
        /// </summary>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="hospitalCodeSystem">The code of the code system of the hospital code, such as pasFacCd or labFacCd or cdmFacCd</param>
        /// <param name="mrn">The patient's medical record number at the hospital.</param>
        /// <param name="dateOfBirth">The patient's date of birth (as an additional identity check)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action</param>
        /// <returns>IHI Search Response</returns>
        public ValidatedIhiResponse GetValidatedIhi(PatientIdentifierBase patientIdentifier, DateTime dateOfBirth, UserDetails user)
        {
            ValidatedIhiResponse response = new ValidatedIhiResponse();
            try
            {
                PatientAccess patientAccess = new PatientAccess(user);
                Hospital hospital;
                HospitalPatient hospitalPatient;
                PatientMaster patientMaster;
                response.HipsResponse = patientAccess.GetHospital(patientIdentifier, out hospital);
                if (response.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return response;
                }
                response.HipsResponse = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
                if (response.HipsResponse.Status != HipsResponseIndicator.OK && response.HipsResponse.Status != HipsResponseIndicator.InvalidIhi)
                {
                    return response;
                }
                if (patientMaster.DateOfBirth.Date != dateOfBirth.Date)
                {
                    response.HipsResponse.Status = HipsResponseIndicator.InvalidDateOfBirth;
                    response.HipsResponse.HipsErrorMessage = string.Format(ConstantsResource.DateOfBirthMismatch, dateOfBirth.Date, patientMaster.DateOfBirth.Date);
                    return response;
                }
                IhiSearchResponse ihiSearch = GetValidatedIhi(patientIdentifier, hospital, user, patientMaster, dateOfBirth);
                if (ihiSearch.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
                {
                    // When the HIB calls the IHI Service to retrieve the IHI number
                    // to embed into the CDA document it is important that an IHI
                    // number is returned even if the IHI is stale.  There could be
                    // a situation where the HI Service from Medicare is down where
                    // an IHI is attempted to be revalidated.  In this case the IHI
                    // may be correct, however, we can not prove this.  For this we
                    // need to ensure that we can use the stale IHI for adding to
                    // the CDA.
                    EventLogger.WriteLog(ConstantsResource.ReturningStaleIhi, new Exception(ihiSearch.HipsResponse.HipsErrorMessage), user, LogMessage.HIPS_MESSAGE_080);
                    response.HipsResponse.Status = HipsResponseIndicator.OK;
                    response.HipsResponse.HipsErrorMessage = ConstantsResource.ReturningStaleIhi;
                }
                else
                {
                    if (ihiSearch.HipsResponse.Status != HipsResponseIndicator.OK)
                    {
                        response.HipsResponse = ihiSearch.HipsResponse;
                        return response;
                    }
                    patientAccess.ValidateLocalIhiInformation(patientMaster, response.HipsResponse);
                    if (response.HipsResponse.Status != HipsResponseIndicator.OK)
                    {
                        return response;
                    }
                }
                PopulateResponse(response, patientIdentifier, hospitalPatient, patientMaster);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.IhiRequestFailed, ex, user, LogMessage.HIPS_MESSAGE_081);
                response.HipsResponse = new HipsResponse(HipsResponseIndicator.SystemError);
                response.HipsResponse.HipsErrorMessage = ex.Message;
                response.HipsResponse.ResponseCode = ex.GetType().ToString();
                response.HipsResponse.ResponseCodeDetails = ex.StackTrace;
                if (ex.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDetails = ex.InnerException.Message;
                }
                response.ValidatedIhi = null;
            }
            return response;
        }

        /// <summary>
        /// Populates the MRN, State Patient ID and Validated IHI information into the response object.
        /// </summary>
        /// <param name="response">The response object to be populated.</param>
        /// <param name="patientIdentifier">The original patient identifier.</param>
        /// <param name="hospitalPatient">The hospital patient</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateResponse(ValidatedIhiResponse response, PatientIdentifierBase patientIdentifier, HospitalPatient hospitalPatient, PatientMaster patientMaster)
        {
            response.Mrn = new Mrn(hospitalPatient.Mrn, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            response.StatePatientId = new StatePatientId(patientMaster.StatePatientId, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            response.ValidatedIhi = new ValidatedIhi(
                patientMaster.Ihi,
                (IhiStatus)patientMaster.IhiStatusId,
                (IhiRecordStatus)patientMaster.IhiRecordStatusId,
                patientMaster.IhiLastValidated.Value,
                patientMaster.RegisteredFamilyName,
                patientMaster.RegisteredGivenName,
                patientMaster.DateOfBirth,
                (SexEnumerator)patientMaster.RegisteredSexId,
                patientIdentifier.HospitalCode,
                patientIdentifier.HospitalCodeSystem);
        }

        /// <summary>
        /// This is the internal interface for HIPS services that need to obtain
        /// a valid IHI. If the IHI is stale, i.e. it was last validated outside
        /// the configurable period, this method attempts to validate the IHI by
        /// passing the IHI and key demographic information to the Medicare HI
        /// Service.
        /// This method allows but does not require the date of birth to be
        /// provided as an additional identity check.
        /// If the patient is not found, or has a different date of birth to the
        /// one given, this method returns an empty response.
        /// </summary>
        /// <param name="patientIdentifier">An identifier of the patient (MRN, SAUHI/HCID, IHI or PatientMasterId).</param>
        /// <param name="hospital">The hospital which is requesting the validated IHI</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action</param>
        /// <param name="patientMaster">Optionally supply patient master record if already obtained</param>
        /// <param name="dateOfBirth">Optionally supply the patient's date of birth (for identity verification)</param>
        /// <returns>Validated IHI, registered Medicare first and last HL7Name, and sex code and description</returns>
        public IhiSearchResponse GetValidatedIhi(PatientIdentifierBase patientIdentifier, Hospital hospital, UserDetails user, PatientMaster patientMaster = null, DateTime? dateOfBirth = null)
        {
            IhiSearchResponse response = new IhiSearchResponse();
            response.IhiRecordStatus = IhiRecordStatus.Unknown;
            response.IhiStatus = IhiStatus.Unknown;

            if (!User.PopulateAndValidateUser(hospital, user))
            {
                // Attempted access with incorrect user details should be brought to system administrator's attention.
                EventLogger.WriteLog(ConstantsResource.InvalidUserDetails, null, user, LogMessage.HIPS_MESSAGE_082);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.MissingUserValues;
                return response;
            }

            if (patientMaster == null)
            {
                // Caller has not yet obtained the patient master object, we will do so here.
                HospitalPatient hospitalPatient;
                PatientAccess patientAccess = new PatientAccess(user);
                HipsResponse getPatientResponse = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
                if (getPatientResponse.Status != HipsResponseIndicator.OK)
                {
                    // Unknown patient record should be brought to attention of system administrator.
                    string description = string.Format(ConstantsResource.CannotFindPatient, patientIdentifier, hospital.HospitalId);
                    EventLogger.WriteLog(description, null, user, LogMessage.HIPS_MESSAGE_083);
                    response.HipsResponse = getPatientResponse;
                    return response;
                }
            }

            if (!ValidateInputDetails(patientMaster, patientIdentifier, hospital, dateOfBirth))
            {
                // Incorrect date of birth could be due to unauthorised access and should be brought to attention of system administrator as well.
                string description = string.Format(ConstantsResource.InvalidIhiRequest, patientIdentifier, hospital, dateOfBirth, patientMaster.DateOfBirth);
                EventLogger.WriteLog(description, null, user, LogMessage.HIPS_MESSAGE_084);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidPatient;
                response.HipsResponse.HipsErrorMessage = string.Format(ConstantsResource.DateOfBirthMismatch, dateOfBirth, patientMaster.DateOfBirth);
                return response;
            }

            try
            {
                if (!string.IsNullOrEmpty(patientMaster.Ihi))
                {
                    TimeSpan ts = DateTime.Now.Subtract(patientMaster.IhiLastValidated.Value);

                    if (ts.TotalHours >= this.IhiValidityPeriodHours || patientMaster.IhiStatusId != (int)IhiStatus.Active)
                    {
                        RevalidateIhi(patientMaster, hospital, user);
                    }
                    response.Ihi = patientMaster.Ihi;
                }
                else
                {
                    PatientRegistration registration = new PatientRegistration();
                    response.Ihi = registration.RegisterPatient(patientMaster, hospital, user);
                }
            }
            catch (IhiServiceUnavailableException ex)
            {
                response.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                response.HipsResponse.HipsErrorMessage = HIPS.PcehrHiBusinessLogic.Pcehr.ResponseStrings.IhiRequiresValidation;
                response.HipsResponse.ResponseCodeDescription = ex.Message;
            }
            catch (PcehrServiceUnavailableException ex)
            {
                response.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                response.HipsResponse.HipsErrorMessage = HIPS.PcehrHiBusinessLogic.Pcehr.ResponseStrings.PcehrSystemTemporarilyUnavailable;
                response.HipsResponse.ResponseCodeDescription = ex.Message;
            }
            response.IhiStatus = (IhiStatus)patientMaster.IhiStatusId;
            response.IhiRecordStatus = (IhiRecordStatus)patientMaster.IhiRecordStatusId;
            response.IhiLastValidated = patientMaster.IhiLastValidated;
            response.FirstName = patientMaster.LegalName.GivenNames;
            response.LastName = patientMaster.LegalName.FamilyName;

            ListSingleton lists = ListSingleton.Instance;
            Sex sex = lists.AllSexes.First(result => result.Id == patientMaster.RegisteredSexId);
            if (sex != null)
            {
                response.SexCode = sex.Code;
                response.SexDescription = sex.Description;
            }

            return response;
        }

        /// <summary>
        /// Sends the IHI and key demographic information to the Medicare HI
        /// Service to determine whether the IHI is still valid.
        /// </summary>
        /// <param HL7Name="patientMaster">The patient master.</param>
        /// <param HL7Name="hospital">The hospital whose HPI-O and Registered Officer is accessing the IHI</param>
        /// <param HL7Name="user">The health provider individual, interactive user or authorised employee responsible for this action</param>
        public void RevalidateIhi(PatientMaster patientMaster, Hospital hospital, UserDetails user)
        {
            IhiSearchCriteria searchDetails = SearchHelper.PopulateSearchDetails(patientMaster);
            string resultMessage;
            searchIHIResponse response;
            bool success = Search.ByNumber(hospital, patientMaster, searchDetails, out resultMessage, out response, user, bool.Parse(ConfigurationManager.AppSettings["CheckDoesPcehrExist"].ToString()));
            patientMaster.Ihi = searchDetails.Ihi;
        }

        /// <summary>
        /// Obtains validated IHI information for use when receiving an e-health message.
        /// </summary>
        /// <param name="consumer">Participating Consumer</param>
        /// <param name="hpio">Health Provider Organisation Id</param>
        /// <param name="user">The user responsible for this action</param>
        /// <returns>Validated IHI information</returns>
        public ValidatedIhiResponse CheckReceivedValidatedIhi(ParticipatingConsumer consumer, string hpio, UserDetails user)
        {
            PatientMaster patientMaster;
            Hospital hospital;
            PatientMasterId patientMasterId;

            var patientAccess = new PatientAccess(user);

            // Validate If HPIO exists in HIPS
            var IsHpioExistResponse = patientAccess.ValidateHpio(hpio);

            if (IsHpioExistResponse.Status != HipsResponseIndicator.OK)
            {
                // HPIO does not exists on HIPS
                return new ValidatedIhiResponse() { HipsResponse = IsHpioExistResponse };
            }

            // Retrieve the configured validity period (in days).
            // Determine the validity period in hours. If the configured period is less than 24 hours, we
            // make sure we use that value, otherwise the maximum we can use is 24 hours.
            var validityPeriodDays = PatientIhiValidation.GetIhiValidationPeriodDays();
            var validityPeriodHours = validityPeriodDays == 0 ? 0 : 24;
            
            // Check Response and Get Patient Master
            var response = patientAccess.CheckReceivedValidatedIhi(consumer, out patientMaster, user);

            // If we don't get a response of OK or InvalidIhi and we didn't receive a patientMaster object then return without validating.
            if ((response.Status != HipsResponseIndicator.OK && patientMaster == null) || (response.Status != HipsResponseIndicator.InvalidIhi && patientMaster == null))
            {
                return new ValidatedIhiResponse() { HipsResponse = response };
            }

            // Find a matching hospital using the Patient ID and HPIO
            bool isHospitalMatched = patientAccess.GetHospitalByPatientMasterIdAndHpio(patientMaster.Id.Value, hpio, out hospital, user);

            // If hospital matched
            if (isHospitalMatched)
            {
                // Create a new patient identifier for our local data
                patientMasterId = new PatientMasterId(patientMaster.Id.Value, hospital.Codes[0].Code, hospital.Codes[0].CodeSystemCode, null);

                // Utilise PatientIhiValidation.GetValidatedIhi to perform the validation.
                return new PatientIhiValidation(validityPeriodHours).GetValidatedIhi(patientMasterId, patientMaster.DateOfBirth, user);
            }
            else // If no hospital matched
            {
                // Check if the UseStateIdentifierAsMrn is true
                if (GetUseStateIdentifierAsMrn()) // If true
                {
                    try
                    {                        
                        // Create Hospital Patient Record using the StateIdentifier as the MRN
                        patientAccess.CreatePatientFromStateIdentifier(patientMaster.Id.Value, patientMaster.StatePatientId, hpio, out hospital);
                    }
                    catch (Exception ex)
                    {
                        return new ValidatedIhiResponse() { HipsResponse = new HipsResponse(HipsResponseIndicator.InvalidPatient) };
                    }

                    // Create a new patient identifier for our local data
                    patientMasterId = new PatientMasterId(patientMaster.Id.Value, hospital.Codes[0].Code, hospital.Codes[0].CodeSystemCode, null);

                    // Utilise PatientIhiValidation.GetValidatedIhi to perform the validation.
                    return new PatientIhiValidation(validityPeriodHours).GetValidatedIhi(patientMasterId, patientMaster.DateOfBirth, user);

                }
                else // If setting is False, Throw Invalid IHI response
                {                    
                    return new ValidatedIhiResponse() { HipsResponse = new HipsResponse(HipsResponseIndicator.InvalidIhi) };
                }
            }
        }
        
        /// <summary>
        /// Sends the new Medicare and DVA number to the Medicare HI Service
        /// and checks whether it finds the same IHI as the current IHI for
        /// the patient. If it finds nothing, or finds a different IHI, then
        /// restores the current IHI with a status of MedicareDvaChangeMismatch.
        /// </summary>
        /// <param name="patientMaster">The updated patient master.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="user">The person who is responsible for the action.</param>
        internal HipsResponse RevalidateMedicareDva(PatientMaster patientMaster, Hospital hospital, UserDetails user)
        {
            // Retain the current IHI information for the patient, as it may be overwritten with a different IHI when we search by Medicare or DVA.
            PatientMasterIhi currentIhi = new PatientMasterIhi();
            currentIhi.PatientMasterId = patientMaster.PatientMasterId.Value;
            currentIhi.Ihi = patientMaster.Ihi;
            currentIhi.IhiStatusId = patientMaster.IhiStatusId;
            currentIhi.IhiRecordStatusId = patientMaster.IhiRecordStatusId;
            currentIhi.RegisteredFamilyName = patientMaster.RegisteredFamilyName;
            currentIhi.RegisteredGivenName = patientMaster.RegisteredGivenName;
            currentIhi.RegisteredSexId = patientMaster.RegisteredSexId;
            currentIhi.MedicareNumber = patientMaster.MedicareNumber;
            currentIhi.MedicareNumberSequence = patientMaster.MedicareIrn;
            currentIhi.DvaNumber = patientMaster.DvaNumber;
            currentIhi.DateOfBirth = patientMaster.DateOfBirth;

            // Temporarily clear the IHI and do a search on the new Medicare and DVA numbers
            patientMaster.Ihi = null;
            IhiSearchCriteria searchDetails = SearchHelper.PopulateSearchDetails(patientMaster);
            string resultMessage;
            searchIHIResponse response;
            bool success = Search.ByNumber(hospital, patientMaster, searchDetails, out resultMessage, out response, user, bool.Parse(ConfigurationManager.AppSettings["CheckDoesPcehrExist"].ToString()));

            if (!success || currentIhi.Ihi != patientMaster.Ihi)
            {
                // The search with the new details got a different IHI. Put the current IHI back and mark it as demographic mismatch.
                currentIhi.IhiStatusId = (int)IhiStatus.MedicareDvaChangeMismatch;
                currentIhi.DateLastValidated = DateTime.Now;
                PatientMasterIhiDl dataAccess = new PatientMasterIhiDl(user);
                if (!dataAccess.Update(currentIhi))
                {
                    return new HipsResponse(HipsResponseIndicator.DatabaseError);
                }
                patientMaster.Ihi = currentIhi.Ihi;
                patientMaster.IhiStatusId = currentIhi.IhiStatusId;
                patientMaster.IhiRecordStatusId = currentIhi.IhiRecordStatusId;
                patientMaster.RegisteredFamilyName = currentIhi.RegisteredFamilyName;
                patientMaster.RegisteredGivenName = currentIhi.RegisteredGivenName;
                patientMaster.RegisteredSexId = currentIhi.RegisteredSexId;
                return new HipsResponse(HipsResponseIndicator.InvalidIhi);
            }
            return new HipsResponse(HipsResponseIndicator.OK);
        }

        /// <summary>
        /// Validates the input of the Service call matches the details stored against the patient's details.
        /// This will help to prevent IHIs being returned for the incorrect patient.
        /// The details required for validation are:
        ///     Date Of Birth
        /// </summary>
        /// <param name="patientMasterId">The patient master ID</param>
        /// <param name="mrn">The input Medical Record Number</param>
        /// <param name="pasHospitalCode">The input hospital code</param>
        /// <param name="inDateOfBirth">The input date of birth</param>
        /// <returns>Whether validation was successful</returns>
        private bool ValidateInputDetails(PatientMaster patientMaster, PatientIdentifierBase patientIdentifier, Hospital hospital, DateTime? inDateOfBirth)
        {
            DateTime currentDateOfBirth = patientMaster.DateOfBirth;
            if (inDateOfBirth.HasValue && inDateOfBirth.Value.Date != currentDateOfBirth.Date)
            {
                // This is potentially a data quality issue and should be brought to attention of those processing the IHI lookup alerts.
                IhiLookupAlertBl.Insert(patientMaster.PatientMasterId.Value, string.Format(ConstantsResource.InvalidIhiRequest, patientIdentifier, hospital, inDateOfBirth, currentDateOfBirth), null, null);
                return false;
            }
            return true;
        }

        /// <summary>
        /// Compares the two string values to make sure they are the same.
        /// Strings that are both null or empty are considered the same.
        /// Strings where one is null or empty and the other is not are different.
        /// Otherwise the strings are the same if they contain the same characters
        /// after they are both converted to uppercase.
        /// </summary>
        /// <param name="value1">The first value.</param>
        /// <param name="value2">The other value.</param>
        /// <returns>True if the same, otherwise false.</returns>
        private bool CompareString(string value1, string value2)
        {
            if (string.IsNullOrEmpty(value1) && string.IsNullOrEmpty(value2))
            {
                return true;
            }
            if (string.IsNullOrEmpty(value1) && !string.IsNullOrEmpty(value2))
            {
                return false;
            }
            if (string.IsNullOrEmpty(value2) && !string.IsNullOrEmpty(value1))
            {
                return false;
            }
            if (value1.ToUpper() == value2.ToUpper())
            {
                return true;
            }
            return false;
        }

        #endregion Methods
    }
}