﻿using System;
using System.Linq;
using System.Reflection;
using AutoMapper;

namespace HIPS.PcehrHiBusinessLogic.Mapping
{
    /// <summary>
    /// Provides mapping between object structures.
    /// </summary>
    /// <remarks>
    /// IMPORTANT!
    /// This class AVOIDS using the static "Mapper" instance provided by AutoMapper, in favour of creating and using its own
    /// static AutoMapper.ConfigurationStore and AutoMapper.MappingEngine instances.
    /// This approach supports isolation of mapping configuration within assemblies.
    /// </remarks>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="18 November 2013">Initial version.</change>
    /// </history>
    internal class ObjectMapper
    {
        #region Fields

        private static AutoMapper.ConfigurationStore Configuration = new AutoMapper.ConfigurationStore(new AutoMapper.TypeMapFactory(), AutoMapper.Mappers.MapperRegistry.Mappers);
        private static AutoMapper.MappingEngine Engine = new AutoMapper.MappingEngine(ObjectMapper.Configuration);

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Static constructor.
        /// </summary>
        /// <remarks>Used to initialise AutoMapper mapping profiles.</remarks>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 November 2013">Initial version.</change>
        /// </history>
        static ObjectMapper()
        {
            //Initialise AutoMapper with profiles defined in this assembly only.
            ObjectMapper.AddProfiles(Assembly.GetExecutingAssembly());

            //Add any other profiles that do not have a parameterless constructor here...
            //...

            ObjectMapper.Configuration.Seal();
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Execute a mapping from the source object to a new destination object. The source type is inferred from the source object.
        /// </summary>
        /// <typeparam name="TDestination">Type parameter representing the destination type instance to be created by the mapping.</typeparam>
        /// <param name="source">Source instance to be used as the source for the mapping.</param>
        /// <returns>An instance of TDestination created by executing a mapping from the source type derived from the source instance to the destination type specified by TDestination.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 November 2013">Initial version.</change>
        /// </history>
        public static TDestination Map<TDestination>(object source)
        {
            return ObjectMapper.Engine.Map<TDestination>(source);
        }

        /// <summary>
        /// Execute a mapping from the source object to a new destination object. The source type is inferred from the source object.
        /// </summary>
        /// <typeparam name="TDestination">Type parameter representing the destination type instance to be created by the mapping.</typeparam>
        /// <param name="source">Source instance to be used as the source for the mapping.</param>
        /// <param name="context">Object containing context to be used during mapping.</param>
        /// <returns>An instance of TDestination created by executing a mapping from the source type derived from the source instance to the destination type specified by TDestination.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
        /// </history>
        public static TDestination Map<TDestination>(object source, object context)
        {
            return ObjectMapper.Engine.Map<TDestination>(source, delegate(IMappingOperationOptions m) { m.Items.Add("context", context); });
        }

        /// <summary>
        /// Execute a mapping from the source object to a new destination object. The source type is inferred from the source object.
        /// </summary>
        /// <typeparam name="TDestination">Type parameter representing the destination type instance to be created by the mapping.</typeparam>
        /// <param name="source">Source instance to be used as the source for the mapping.</param>
        /// <param name="mapParameters">IDictionary containing parameters to be used within the map.</param>
        /// <returns>An instance of TDestination created by executing a mapping from the source type derived from the source instance to the destination type specified by TDestination.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
        /// </history>
        public static TDestination Map<TDestination>(object source, System.Collections.Generic.IDictionary<string, object> mapParameters)
        {
            return ObjectMapper.Engine.Map<TDestination>(source, delegate(IMappingOperationOptions m) { mapParameters.ToList().ForEach(p => m.Items.Add(p.Key, p.Value)); });
        }

        /// <summary>
        /// Add the profiles from the provided assembly to the mapping configuration.
        /// </summary>
        /// <remarks>
        /// Only profiles with a parameterless constructor will be added.
        /// </remarks>
        /// <param name="source">Source assembly containing profiles to be added.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="26 March 2014">Initial version.</change>
        /// </history>
        protected static void AddProfiles(System.Reflection.Assembly source)
        {
            var profileType = typeof(Profile);
            // Get an instance of each Profile with a parameterless constructor in the executing assembly.
            var profiles = Assembly.GetExecutingAssembly().GetTypes().Where(t => profileType.IsAssignableFrom(t) && t.GetConstructor(Type.EmptyTypes) != null)
                .Select(Activator.CreateInstance)
                .Cast<Profile>();

            // Initialize AutoMapper with each instance of the profiles found.
            profiles.ToList().ForEach(p => ObjectMapper.Configuration.AddProfile(p));
        }

        #endregion Methods
    }
}