﻿using System.Runtime.Serialization;

using EntLibVal = Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.ServiceContracts.Common.DTO
{
    /// <summary>
    /// Address DTO. This object represents a physical address, which may be a
    /// street address or a postal address, and may be an Australian address or
    /// an international address.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    [EntLibVal.Validators.HasSelfValidation]
    public class Address
    {
        #region Properties

        /// <summary>
        /// Gets or sets the first line of the unstructured address.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.StringLengthValidator(1, 1024)]
        public string AddressLine1 { get; set; }

        /// <summary>
        /// Gets or sets the second line of the unstructured address, if there is more
        /// than one line. This should not include the line that has the suburb, state,
        /// postcode or country, as those must be provided in separate elements.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 1024)]
        [EntLibVal.Validators.IgnoreNulls]
        public string AddressLine2 { get; set; }

        /// <summary>
        /// Gets or sets the address purpose. The valid codes are defined in
        /// AS 5017-2006 in the table "Health Care Client Identifier Address Purpose".
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DomainValidator(AustralianState.None, Negated = true)]
        public AddressPurpose AddressPurpose { get; set; }

        /// <summary>
        /// Gets or sets the Australian state. The valid codes are defined in AS 5017-2006
        /// in the table "Australian State/Territory Identifier - Postal".
        /// This is mandatory for an Australian address, and must not be provided for an
        /// international address.
        /// </summary>
        [DataMember(IsRequired = false)]
        public AustralianState AustralianState { get; set; }

        /// <summary>
        /// Gets or sets the country name. This is mandatory.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.StringLengthValidator(1, 1024)]
        public string CountryName { get; set; }

        /// <summary>
        /// Gets or sets the international state code. This is optional for an international
        /// address, and must not be provided for an Australian address.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 1024)]
        [EntLibVal.Validators.IgnoreNulls]
        public string InternationalStateCode { get; set; }

        /// <summary>
        /// Gets or sets the suburb, town or locality.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 1024)]
        [EntLibVal.Validators.IgnoreNulls]
        public string PlaceName { get; set; }

        /// <summary>
        /// Gets or sets the postcode. This is mandatory for an Australian address, and
        /// optional for an international address.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 1024)]
        [EntLibVal.Validators.IgnoreNulls]
        public string Postcode { get; set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Validate the instance is valid.
        /// </summary>
        /// <param name="results">Validation results to add results of validation to.</param>
        [EntLibVal.Validators.SelfValidation]
        protected void Validate(EntLibVal.ValidationResults results)
        {
            if (this.CountryName != "Australia")
            {
                if (this.AustralianState != Common.AustralianState.None)
                {
                    results.AddResult(new EntLibVal.ValidationResult("When CountryName is not Australia, the value of AustralianState must be None.", this, "Address", string.Empty, null));
                }
            }
            else
            {
                if (this.InternationalStateCode != null)
                {
                    results.AddResult(new EntLibVal.ValidationResult("When CountryName is Australia, the value of InternationalStateCode must be null.", this, "Address", string.Empty, null));
                }
            }
        }

        #endregion Methods
    }
}