﻿#region Using Directives

using System.ServiceModel;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PatientSchemas;

#endregion Using Directives

namespace HIPS.ServiceContracts.ServiceInterfaces
{
    /// <summary>
    /// This interface defines the contract for web service operations upon patients in the HIPS database.
    /// </summary>
    [WCFExtras.Wsdl.Documentation.XmlComments]
    [ServiceContract(Namespace = "http://schemas.HIPS/Services/2012/01", Name = "IPatientService")]
    public interface IPatientService
    {
        /// <summary>
        /// Lists the episodes for a specific patient in a defined hospital
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="sourceSystemEpisodeId">The source system episode identifier for the episode in the hospital</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <returns>
        /// Response containing the a list of the admitted or discharged episodes for the patient with in the specified number of days.,
        /// </returns>
        [OperationContract]
        EpisodePatientDetailsResponse GetEpisodeDetails(UserDetails user, string sourceSystemEpisodeId, PatientIdentifierBase patientIdentifier);

        /// <summary>
        /// Gets the details of the patient’s discloser record for a specific HPIO
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <returns>
        /// Response containing the details of the patient's disclosure.
        /// </returns>
        [OperationContract]
        PatientDisclosureDetailsResponse GetPatientDisclosureDetails(UserDetails user, PatientIdentifierBase patientIdentifier);

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital. Filters are available
        /// to list patients with or without an IHI; with or without a PCEHR; whether to
        /// exclude patients with special values in the Medicare card number field; and the
        /// number of days after discharge that patients are included in the list.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="withIhi">Optional filter on having an active verified IHI, or no IHI.</param>
        /// <param name="withPcehr">Optional filter on having a PCEHR found, or no PCEHR found</param>
        /// <param name="excludeMedicareExclusions">Whether to exclude patients with special Medicare numbers.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <param name="daysAfterService">Number of days after service date that non-inpatients are included in the list. Set to 0 to exclude non-inpatients.</param>
        /// <param name="user">The user who is requesting the information.</param>
        /// <returns>A patient list response with status and a list of patients in hospital.</returns>
        [OperationContract]
        AdmittedPatientListResponse ListAdmittedPatients(string hospitalCodeSystem, string hospitalCode, bool? withIhi, bool? withPcehr, bool excludeMedicareExclusions, int daysDischarged, int daysAfterService, UserDetails user);

        /// <summary>
        /// Lists the episodes for a specific patient in a defined hospital
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="daysDischarged">Number of days after discharge that an episode is included in the list.</param>
        /// <returns>
        /// Response containing the a list of the admitted or discharged episodes for the patient with in the specified number of days.,
        /// </returns>
        [OperationContract]
        PatientEpisodeListResponse ListPatientEpisodesInHospital(UserDetails user, PatientIdentifierBase patientIdentifier, int daysDischarged);

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital with or without a PCEHR.
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="withPcehr">True to list patients with a PCEHR, false to list patients without a PCEHR. Null to include patients with or without a PCEHR.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <returns>A patient list response with status and a list of patients in hospital.</returns>
        [OperationContract]
        PatientListResponse ListPatientsInHospital(UserDetails user, string hospitalCodeSystem, string hospitalCode, bool? withPcehr, int daysDischarged);
    }
}