﻿using HIPS.Client.Proxy;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PatientSchemas;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.ModelInterface.ConsentManagement;

namespace HIPS.Web.Data.Hips.ConsentManagement
{
    /// <summary>
    /// Implements a repository for accessing episodes of care for a patient.
    /// </summary>
    public class EpisodeRepository : RepositoryBase<PatientProxy>, IEpisodeRepository
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="EpisodeRepository"/> class.
        /// </summary>
        /// <param name="daysDischarged">Integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".</param>
        public EpisodeRepository(int daysDischarged)
            : base(null, null)
        {
            // Initialise client proxy ready for use.
            this.ClientProxy = new PatientProxy("PatientEndPoint");
            this.DaysDischarged = daysDischarged;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets an integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".
        /// </summary>
        public int DaysDischarged { get; private set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Retrieves a list of recent episodes for a patient at a hospital.
        /// </summary>
        /// <param name="patientIdentifier">The MRN that identifies the patient at the hospital.</param>
        /// <param name="user">User requesting the operation.</param>
        /// <returns>ServiceResponse containing a PatientEpisodeListResponse containing the results of the operation.</returns>
        public ServiceResponse<PatientEpisodeListResponse> ListPatientEpisodesInHospital(Mrn patientIdentifier, UserDetails user)
        {
            var result = this.ClientProxy.ListPatientEpisodesInHospital(user, patientIdentifier, this.DaysDischarged);
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to retrieve episodes for the specified patient.", MessageLevel.Error);
            }

            return new ServiceResponse<PatientEpisodeListResponse>(result, isSuccessful, messages);
        }

        /// <summary>
        /// Retrieves details of a single episode.
        /// </summary>
        /// <param name="user">User requesting the operation.</param>
        /// <param name="episodeId">The visit number that identifies the episode among all episodes for the patient.</param>
        /// <param name="patientIdentifier">The MRN that identifies the patient at the hospital.</param>
        /// <returns>ServiceResponse containing a EpisodePatientDetailsResponse containing the results of the operation.</returns>
        public ServiceResponse<EpisodePatientDetailsResponse> GetEpisodeDetails(UserDetails user, string episodeId, Mrn patientIdentifier)
        {
            var result = this.ClientProxy.GetEpisodeDetails(user, episodeId, patientIdentifier);
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to retrieve the specified episode for the specified patient.", MessageLevel.Error);
            }

            return new ServiceResponse<EpisodePatientDetailsResponse>(result, isSuccessful, messages);
        }

        #endregion Methods
    }
}