﻿using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;

using AutoMapper;

using HIPS.P2P.Service.Contract.Common;
using HIPS.P2P.Service.Contract.Common.DTO;
using HIPS.P2P.Service.Contract.Common.Fault;
using HIPS.Web.Components.ServiceModel;
using Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.Web.Data.P2P.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for faults.
    /// </summary>
    internal class FaultProfile : Profile
    {
        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }
        
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            /*
            Service --> Common
            */

            // ServiceOperationFault

            this.CreateMap<FaultException<ServiceOperationFault>, List<ResponseMessage>>()
                .ConstructUsing(
                    (System.Func<FaultException<ServiceOperationFault>, List<ResponseMessage>>)
                    (src => new List<ResponseMessage>() { ObjectMapper.Map<ResponseMessage>(src.Detail) }));

            this.CreateMap<ServiceOperationFault, ResponseMessage>()
                .ForMember(dest => dest.Level, opt => opt.UseValue(HIPS.Web.Components.Common.MessageLevel.Error))
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => this.GetFaultDescription(src.Message, src.FaultIdentifier)));

            // InvalidRequestFault
            this.CreateMap<FaultException<InvalidRequestFault>, List<ResponseMessage>>()
                .ConstructUsing(
                    (System.Func<FaultException<InvalidRequestFault>, List<ResponseMessage>>)
                    (src => new List<ResponseMessage>().Concat(new List<ResponseMessage>() { ObjectMapper.Map<ResponseMessage>(src.Detail) }).Concat(ObjectMapper.Map<List<ResponseMessage>>(src.Detail.Messages)).ToList()));

            this.CreateMap<InvalidRequestFault, ResponseMessage>()
                .ForMember(dest => dest.Level, opt => opt.UseValue(HIPS.Web.Components.Common.MessageLevel.Error))
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => this.GetFaultDescription(src.Message, src.FaultIdentifier)));

            // System.Exception
            this.CreateMap<System.Exception, List<ResponseMessage>>()
                .ConstructUsing(
                    (System.Func<System.Exception, List<ResponseMessage>>)
                    (src => new List<ResponseMessage>() { ObjectMapper.Map<ResponseMessage>(src) }));

            this.CreateMap<System.Exception, ResponseMessage>()
                .ForMember(dest => dest.Level, opt => opt.UseValue(HIPS.Web.Components.Common.MessageLevel.Error))
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => src.Message));
        }

        /// <summary>
        /// Constructs and returned a description for a fault based on the provided parameters.
        /// </summary>
        /// <param name="message">Fault message.</param>
        /// <param name="faultIdentifier">Fault identifier.</param>
        /// <returns>String containing the combined description.</returns>
        private string GetFaultDescription(string message, string faultIdentifier)
        {
            return string.Format("{0} [Fault Identifier: {1}]", message, faultIdentifier);
        }     

        #endregion
    }
}