﻿using System;
using System.Collections.Generic;
using System.Linq;

using HIPS.P2P.Service.Contract.Common.Fault;
using HIPS.P2P.Service.Contract.Directory.Configuration.Message;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.P2P.Mapping;
using HIPS.Web.ModelInterface.Directory;

using CommonSchema = HIPS.Web.Model;
using ServiceSchema = HIPS.P2P.Service.Contract.Common.DTO;

namespace HIPS.Web.Data.P2P.Service
{

    /// <summary>
    /// Supports the maintenance of directory configuration.
    /// </summary>
    public class DirectoryConfigurationService : ServiceBase<HIPS.P2P.Client.Proxy.DirectoryConfigurationProxy>, IDirectoryConfigurationService
    {

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the DirectoryConfigurationService class.
        /// </summary>
        public DirectoryConfigurationService() : base()
        {
            this.ClientProxy = new HIPS.P2P.Client.Proxy.DirectoryConfigurationProxy("DirectoryConfigurationEndPoint");
        }

        #endregion

        #region Methods

        #region Payload scheme

        /// <summary>
        /// Returns a list of all configured payload schemes.
        /// </summary>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<List<CommonSchema.Directory.PayloadScheme>> ListPayloadSchemes(CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.ListPayloadSchemes(new ListPayloadSchemesRequest() { User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext) });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<CommonSchema.Directory.PayloadScheme>>(ObjectMapper.Map<List<CommonSchema.Directory.PayloadScheme>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.Directory.PayloadScheme>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Sets the activation state of a specified payload scheme in order to activate or inactivate it.
        /// </summary>
        /// <param name="qualifiedIdentifier">Unique identifier of the payload scheme.</param>
        /// <param name="isActive">Boolean value that indicates whether the payload scheme is active (available for use).</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<object> ActivatePayloadScheme(string qualifiedIdentifier, bool isActive, CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.ActivatePayloadScheme(new ActivatePayloadSchemeRequest() { User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext), QualifiedIdentifier = qualifiedIdentifier, IsActive = isActive });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }
        
        /// <summary>
        /// Returns a list of all document types.
        /// </summary>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<List<CommonSchema.Directory.DocumentType>> ListDocumentTypes(CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.ListDocumentTypes(new ListDocumentTypesRequest() { User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext) });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<CommonSchema.Directory.DocumentType>>(ObjectMapper.Map<List<CommonSchema.Directory.DocumentType>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.Directory.DocumentType>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        #endregion

        #region Delegate

        /// <summary>
        /// Returns a list of all configured delegates.
        /// </summary>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<List<CommonSchema.Directory.Delegate>> ListDelegates(CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.ListDelegates(new ListDelegatesRequest() { User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext) });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<CommonSchema.Directory.Delegate>>(ObjectMapper.Map<List<CommonSchema.Directory.Delegate>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.Directory.Delegate>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Sets the activation state of a specified delegate in order to activate or deactivate it.
        /// </summary>
        /// <param name="qualifiedIdentifier">Unique identifier of the delegate.</param>
        /// <param name="isActive">Boolean value that indicates whether the delegate is active (available for use).</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<object> ActivateDelegate(string qualifiedIdentifier, bool isActive, CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.ActivateDelegate(new ActivateDelegateRequest() { User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext), QualifiedIdentifier = qualifiedIdentifier, IsActive = isActive });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Sets the priority position of the delegate by updating its priority number.
        /// </summary>
        /// <param name="qualifiedIdentifier">Unique identifier of the delegate.</param>
        /// <param name="relativeToQualifiedIdentifier">The reference unique identifier to which the delegate in context will be prioritised over to.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<object> PrioritiseDelegate(string qualifiedIdentifier, string relativeToQualifiedIdentifier, CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.PrioritiseDelegate(new PrioritiseDelegateRequest() { User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext), QualifiedIdentifier = qualifiedIdentifier, RelativeToQualifiedIdentifier = relativeToQualifiedIdentifier });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Returns a list of all configured intermediaries.
        /// </summary>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<List<CommonSchema.Directory.Intermediary>> ListIntermediaries(CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.ListIntermediaries(new ListIntermediariesRequest() { User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext) });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<CommonSchema.Directory.Intermediary>>(ObjectMapper.Map<List<CommonSchema.Directory.Intermediary>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.Directory.Intermediary>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Sets the intermediary for a given delegate.
        /// </summary>
        /// <param name="qualifiedIdentifier">Unique identifier of the delegate.</param>
        /// <param name="intermediaryServiceEndpointUri">The intermediary that will be assigned to the delegate.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>ServiceResponse object.</returns>
        public ServiceResponse<object> MediateDelegate(string qualifiedIdentifier, string intermediaryServiceEndpointUri, CommonSchema.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var result = this.ClientProxy.MediateDelegate(new MediateDelegateRequest() { QualifiedIdentifier = qualifiedIdentifier, IntermediaryServiceEndpointUri = intermediaryServiceEndpointUri, User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext) });

                // HIPS Status should be OK
                isSuccessful = result.Status == HIPS.P2P.Service.Contract.Common.ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        #endregion        

        #endregion        
    }
}
