﻿using System;
using System.Collections.Generic;

using HIPS.P2P.Service.Contract.Common;
using HIPS.P2P.Service.Contract.Messaging.Delivery.Message;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.P2P.Mapping;
using HIPS.Web.ModelInterface.Subscription;

using CommonSchema = HIPS.Web.Model.Subscription;
using ServiceSchema = HIPS.P2P.Service.Contract.Common.DTO;
using HIPS.P2P.Service.Contract.Directory.Subscription.Message;

namespace HIPS.Web.Data.P2P.Service
{
    /// <summary>
    /// Defines the implementation of the Subscription Maintenance.
    /// </summary>
    public class SubscriptionMaintenanceService : ServiceBase<HIPS.P2P.Client.Proxy.SubscriptionProxy>, ISubscriptionService
    {
       #region Constructor

        /// <summary>
        /// Initialises a new instance of the <see cref="SubscriptionMaintenanceService"/> class.
        /// </summary>
        public SubscriptionMaintenanceService()
        {
            this.ClientProxy = new HIPS.P2P.Client.Proxy.SubscriptionProxy("SubscriptionEndPoint");
        }

        #endregion

        #region Methods

        /// <summary>
        /// Searches the NHSD for provider individuals.
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        public ServiceResponse<CommonSchema.SearchResult> FindIndividuals(string familyName, string givenName, string occupation, string suburb, string state, string postCode, int proximityRadius, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                // Generate Request
                var request = this.GenerateIndividualRequest(familyName, givenName, occupation, suburb, state, postCode, proximityRadius, userContext);
                
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext);

                var result = this.ClientProxy.FindIndividuals(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<CommonSchema.SearchResult>(ObjectMapper.Map<CommonSchema.SearchResult>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.SearchResult>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Searches the NHSD for provider organisations.
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        public ServiceResponse<CommonSchema.SearchResult> FindOrganisations(string organisationName, string classificaiton, string suburb, string state, string postCode, int proximityRadius, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                // Generate Request
                var request = this.GenerateOrganisationRequest(organisationName, classificaiton, suburb, state, postCode, proximityRadius, userContext);

                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext);

                var result = this.ClientProxy.FindOrganisations(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<CommonSchema.SearchResult>(ObjectMapper.Map<CommonSchema.SearchResult>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.SearchResult>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// List of Area of Interest definitions.
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        public ServiceResponse<List<CommonSchema.AreaOfInterest>> ListSubscriptionDefinitions(Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new ListSubscriptionDefinitionsRequest()
                {
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.ListSubscriptionDefinitions(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<CommonSchema.AreaOfInterest>>(ObjectMapper.Map<List<CommonSchema.AreaOfInterest>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.AreaOfInterest>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Updates an Area of Interest definition to the accessing organisation’s subscription.
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        public ServiceResponse<CommonSchema.AreaOfInterest> UpdateAreaOfInterest(CommonSchema.AreaOfInterest areaOfInterest, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new UpdateAreaOfInterestRequest()
                {
                    AreaOfInterest = ObjectMapper.Map<ServiceSchema.AreaOfInterest>(areaOfInterest),
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.UpdateAreaOfInterest(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<CommonSchema.AreaOfInterest>(ObjectMapper.Map<CommonSchema.AreaOfInterest>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.AreaOfInterest>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Adds an Area of Interest definition to the accessing organisation’s subscription.
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        public ServiceResponse<CommonSchema.AreaOfInterest> AddAreaOfInterest(CommonSchema.AreaOfInterest areaOfInterest, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new AddAreaOfInterestRequest()
                {
                    AreaOfInterest = ObjectMapper.Map<ServiceSchema.AreaOfInterest>(areaOfInterest),
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.AddAreaOfInterest(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<CommonSchema.AreaOfInterest>(ObjectMapper.Map<CommonSchema.AreaOfInterest>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.AreaOfInterest>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Deletes an Area of Interest definition from the accessing organisation’s subscription.
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        public ServiceResponse<CommonSchema.AreaOfInterest> DeleteAreaOfInterest(CommonSchema.AreaOfInterest areaOfInterest, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new DeleteAreaOfInterestRequest()
                {
                    AreaOfInterest = ObjectMapper.Map<ServiceSchema.AreaOfInterest>(areaOfInterest),
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.DeleteAreaOfInterest(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<CommonSchema.AreaOfInterest>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.AreaOfInterest>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Generates the request for the service FindIndividuals
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        private FindIndividualsRequest GenerateIndividualRequest(string familyName, string givenName, string occupation, string suburb, string state, string postCode, int? proximityRadius, Model.Identity.LocalUser userContext)
        {
            // Generate Request
            var request = new FindIndividualsRequest();
            var address = new ServiceSchema.Search.AddressSearch();

            if (!string.IsNullOrEmpty(familyName))
            {
                request.LastNamePattern = familyName;
            }

            if (!string.IsNullOrEmpty(givenName))
            {
                request.FirstNamePattern = givenName;
            }

            if (!string.IsNullOrEmpty(occupation))
            {
                request.Occupation = new ServiceSchema.IndividualOccupation() { URI = occupation };
            }

            if (!string.IsNullOrEmpty(suburb))
            {
                address.Suburb = suburb;
            }

            if (!string.IsNullOrEmpty(state))
            {
                address.State = state;
            }

            if (!string.IsNullOrEmpty(postCode))
            {
                address.Postcode = postCode;
            }

            if (proximityRadius != 0)
            {
                request.ProximityRadius = proximityRadius;
            }
            else
            {
                request.ProximityRadius = null;
            }

            if (!string.IsNullOrEmpty(suburb))
            {
                request.Address = address;
            }

            return request;
        }

        /// <summary>
        /// Generates the request for the service FindOrganisations
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        private FindOrganisationsRequest GenerateOrganisationRequest(string organisationName, string classificaiton, string suburb, string state, string postCode, int? proximityRadius, Model.Identity.LocalUser userContext)
        {
            // Generate Request
            var request = new FindOrganisationsRequest();
            var address = new ServiceSchema.Search.AddressSearch();

            if (!string.IsNullOrEmpty(organisationName))
            {
                request.NamePattern = organisationName;
            }

            if (!string.IsNullOrEmpty(classificaiton))
            {
                request.Classification = new ServiceSchema.LocationClassification()
                {
                    URI = classificaiton
                };
            }

            if (!string.IsNullOrEmpty(suburb))
            {
                address.Suburb = suburb;
            }

            if (!string.IsNullOrEmpty(state))
            {
                address.State = state;
            }

            if (!string.IsNullOrEmpty(postCode))
            {
                address.Postcode = postCode;
            }

            if (proximityRadius != 0)
            {
                request.ProximityRadius = proximityRadius;
            }
            else
            {
                request.ProximityRadius = null;
            }

            if (!string.IsNullOrEmpty(suburb))
            {
                request.Address = address;
            }

            return request;
        }

        #endregion
    }
}
