#region Using Statements

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Web.Mvc;
using System.Web.Routing;

using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Components.Web;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Directory;
using HIPS.Web.UI.Areas.Directory.ViewModels;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.Shared;

#endregion

namespace HIPS.Web.UI.Areas.Directory.Controllers
{
    /// <summary>
    /// Controller for the maintaining payload schemes as part of the "Directory Maintenance" feature.
    /// </summary>
    [NoCache]
    [HpoRequired]
    public class PayloadSchemesController : HIPS.Web.UI.Controllers.ControllerBase
    {

        #region Fields

        /// <summary>
        /// Gets the directory configuration service to be used by this controller.
        /// </summary>
        private readonly IDirectoryConfigurationService directoryConfigurationService;

        #endregion

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="PayloadSchemesController"/> class.
        /// </summary>
        /// <param name="directoryConfigurationService">Directory configuration service to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public PayloadSchemesController(IDirectoryConfigurationService directoryConfigurationService, ISettingsRepository settingsRepository, ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.directoryConfigurationService = directoryConfigurationService;
        }

        #endregion

        #region Methods

        #region Actions

        /// <summary>
        /// Display payload schemes for maintenance.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult Index()
        {
            return this.View();
        }

        /// <summary>
        /// List payload schemes for maintenance.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult List()
        {
            var m = new ListPayloadSchemesViewModel();
            var result = this.directoryConfigurationService.ListPayloadSchemes(this.GetLocalUser());

            if (result.IsSuccessful)
            {
                m.PayloadSchemes = result.Data.OrderBy(i => i.DocumentType).ThenByDescending(i => i.PriorityNumber).ToList();
            }

            // Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));

            return this.View(m);
        }

        /// <summary>
        /// Activate a specified payload scheme.
        /// </summary>
        /// <param name="m">View model identifying the payload scheme to be activated or deactivated.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        [HttpPost]
        public ActionResult Activate(ActivatePayloadSchemeViewModel m)
        {
            if (!this.ModelState.IsValid)
            {
                var modelErrorMessages = this.ModelState.Values.SelectMany(i => i.Errors).Select(i => new ViewMessage(i.ErrorMessage, MessageLevel.Error));
                m.Messages.AddRange(modelErrorMessages);
            }
            else
            {
                var result = this.directoryConfigurationService.ActivatePayloadScheme(m.QualifiedIdentifier, m.IsActive, this.GetLocalUser());

                // Copy over response messages.
                m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));
            }
            return this.PartialView("ViewMessageList", m.Messages);
        }

        #endregion

        #endregion
    }
}