﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Mime;
using System.Text.RegularExpressions;
using System.Web.Mvc;

using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.Web;
using HIPS.Web.Model;
using HIPS.Web.Model.Common;
using HIPS.Web.Model.Messaging;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Directory;
using HIPS.Web.ModelInterface.Messaging;
using HIPS.Web.UI.Areas.Messaging.ViewModels;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.Shared;
using System.Web.SessionState; 

namespace HIPS.Web.UI.Areas.Messaging.Controllers
{
    /// <summary>
    /// Controller for the maintaining outbound messages as part of the "Messaging" feature.
    /// </summary>    
    [NoCache]
    [HpoRequired]
    public class DeliveryController : HIPS.Web.UI.Controllers.ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Gets the directory configuration service to be used by this controller.
        /// </summary>
        private readonly IDirectoryConfigurationService directoryConfigurationService;

        /// <summary>
        /// Gets the message delivery service to be used by this controller.
        /// </summary>
        private readonly IMessageDeliveryService messageDeliveryService;

        #endregion

        #region Constructor

        /// <summary>
        /// Initialises a new instance of the <see cref="DeliveryController"/> class.
        /// </summary>
        /// <param name="messageDeliveryService">Message Delivery service to be used by this controller.</param>
        /// <param name="directoryConfigurationService">Directory configuration service to be used by this controller.</param>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public DeliveryController(
            IMessageDeliveryService messageDeliveryService,
            IDirectoryConfigurationService directoryConfigurationService,
            IHospitalRepository hospitalRepository,
            ISettingsRepository settingsRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.directoryConfigurationService = directoryConfigurationService;
            this.messageDeliveryService = messageDeliveryService;

            this.hospitalRepository = hospitalRepository;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets an integer value representing the number of months the "From" date will be offset from the current date by default.
        /// </summary>
        private int FromDateOffsetDays
        {
            get
            {
                return int.Parse(this.Settings.GetSettingValue(Setting.SettingCodes.MessageDeliveryViewFromDateOffsetDays));
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// Display initial view for specifying criteria to search.
        /// </summary>
        /// <returns>A view for displaying search criteria</returns>        
        public ActionResult Index()
        {
            var viewModel = new DeliveryFilterViewModel()
            {
                From = DateTime.Today.AddDays(this.FromDateOffsetDays),
                To = DateTime.Today.AddDays(1)
            };

            this.LoadReferenceData(viewModel);

            return this.View(viewModel);
        }

        /// <summary>
        /// Display outbound messages for specified criteria.
        /// </summary>
        /// <param name="model">View model identifying the messages to be displayed.</param>
        /// <returns>A view for displaying messages.</returns>        
        [HttpPost]
        [ValidateAntiForgeryToken]       
        public ActionResult List(DeliveryFilterViewModel model)
        {
            var messages = new ViewMessageList();

            if (ModelState.IsValid)
            {
                var sender = this.SetMessageAddressee(model.SenderHpio);
                var receiver = this.SetMessageAddressee(model.Receiver);
                var deliveryPeriod = new DateTimeRange()
                {
                    Start = model.From,
                    End = new DateTime(model.To.Year, model.To.Month, model.To.Day, 23, 59, 59)
                };

                var request = this.messageDeliveryService.QueryMessages(sender, receiver, deliveryPeriod, model.MessageStatusId, model.DocumentTypeCode, this.GetLocalUser(), this.QueryRecordLimit, this.QueryRecordSkipFrom);
                messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));

                if (request.IsSuccessful)
                {
                    if (request.Data.Count > 0)
                    {
                        var result = ObjectMapper.Map<List<ListDeliveryViewModel>>(request.Data);
                        return this.PartialView(result);
                    }

                    messages.Add("No messages are available for the selected criteria.", MessageLevel.Information);
                }
            }

            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Aborts a specified outbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the outbound message.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>        
        [HttpPost]
        public ActionResult Abort(string invocationIdentifier)
        {
            var request = this.messageDeliveryService.AbortMessage(invocationIdentifier, this.GetLocalUser());
            var messages = new ViewMessageList();
            messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));
            
            if (request.IsSuccessful)
            {
                messages.Add("Message was successfully aborted.", MessageLevel.Information);
            }

            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Retries a specified outbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the outbound message.</param>
        /// <param name="expiryDate">Sets a new expiry date for the outbound message.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>        
        [HttpPost]
        public ActionResult Retry(string invocationIdentifier, DateTime expiryDate)
        {
            var request = this.messageDeliveryService.RetryMessage(invocationIdentifier, expiryDate, this.GetLocalUser());
            var messages = new ViewMessageList();
            messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));
            
            if (request.IsSuccessful)
            {
                messages.Add("Message was successfully submitted for retry.", MessageLevel.Information);
            }

            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Downloads the message content of the specified outbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the outbound message.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        public new ActionResult Content(string invocationIdentifier)
        {
            var request = this.messageDeliveryService.GetMessage(invocationIdentifier, this.GetLocalUser());
            var result = request.Data;

            if (request.IsSuccessful)
            {
                if (request.Data.Payload == null)
                {
                    return new HttpNotFoundResult();
                }

                string filename = "content.bin";
                return this.File(request.Data.Payload, MediaTypeNames.Application.Octet, filename);
            }

            throw new NullReferenceException("Request.Data is null");
        }

        #region Helpers

        /// <summary>
        /// Initialise initial data for the filter
        /// </summary>
        /// <param name="model">View Model to fill up with initial data.</param>
        private void LoadReferenceData(DeliveryFilterViewModel model)
        {
            var hospitals = this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem)
                                                   .GroupBy(h => h.HpiO)
                                                   .ToSelectListItems(
                                                        i => i.Key,
                                                        i => string.Format("{0} ({1})", i.First().HpioName, i.Key.ToHpioFormat()),
                                                        i => i.Key == this.SessionConfiguration.RepresentingHospital.Hpio);

            model.Senders = hospitals;
            model.MessageStatuses = Enum.GetValues(typeof(OutboundMessageStatus))
                                        .Cast<OutboundMessageStatus>()
                                        .ToSelectListItems(o => ((int)o).ToString(), o => o.ToString())
                                        .Where(o => o.Value != "8");

            var request = this.directoryConfigurationService.ListDocumentTypes(this.GetLocalUser());

            if (request.IsSuccessful)
            {
                model.DocumentTypes = request.Data.ToSelectListItems(d => d.Code, d => d.Description);
            }
        }

        /// <summary>
        /// Creates a new instance of Message Addressee.
        /// </summary>
        /// <param name="hpio">A value to be set on Identifier.Value or Organisation.OrganisationName</param>
        /// <returns>A message addressee</returns>
        private MessageAddressee SetMessageAddressee(string hpio)
        {
            if (!string.IsNullOrEmpty(hpio))
            {
                var addressee = new MessageAddressee();
                addressee.Organisation = new ProviderOrganisationIdentity();

                if (this.ValidateHpio(hpio))
                {
                    addressee.Organisation.Identifier = new ProviderOrganisationIdentifier();
                    addressee.Organisation.Identifier.Type = new ProviderIdentifierType();
                    addressee.Organisation.Identifier.Type.Code = "HPIO";
                    addressee.Organisation.Identifier.Value = hpio.ExtractNumbers();
                }
                else
                {
                    addressee.Organisation.OrganisationName = hpio;
                }

                return addressee;
            }

            return null;
        }

        /// <summary>
        /// Checks if the string is a valid HPI-O.
        /// </summary>
        /// <param name="hpio">String to validate.</param>
        /// <returns>Whether the provided string was valid HPI-O.</returns>
        private bool ValidateHpio(string hpio)
        {
            string value = hpio.Replace("-", string.Empty)
                               .Replace(" ", string.Empty)
                               .Trim();

            return Regex.IsMatch(value, @"^\d{16}$");
        }

        #endregion

        #endregion
    }
}
