﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Mime;
using System.Text.RegularExpressions;
using System.Web.Mvc;

using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.Web;
using HIPS.Web.Model;
using HIPS.Web.Model.Common;
using HIPS.Web.Model.Messaging;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Directory;
using HIPS.Web.ModelInterface.Messaging;
using HIPS.Web.UI.Areas.Messaging.ViewModels;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.Areas.Messaging.Controllers
{
    /// <summary>
    /// Controller for the maintaining inbound messages as part of the "Messaging" feature.
    /// </summary>
    [NoCache]
    [HpoRequired]
    public class ReceiptController : HIPS.Web.UI.Controllers.ControllerBase
    {
        #region Fields
        
        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private IHospitalRepository hospitalRepository;

        /// <summary>
        /// Gets the directory configuration service to be used by this controller.
        /// </summary>
        private IDirectoryConfigurationService directoryConfigurationService;

        /// <summary>
        /// Gets the message receipt service to be used by this controller.
        /// </summary>
        private IMessageReceiptService messageReceiptService;

        #endregion

        #region Constructor
        
        /// <summary>
        /// Initialises a new instance of the <see cref="ReceiptController"/> class.
        /// </summary>
        /// <param name="messageReceiptService">Message Receipt service to be used by this controller.</param>
        /// <param name="directoryConfigurationService">Directory configuration service to be used by this controller.</param>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public ReceiptController(
            IMessageReceiptService messageReceiptService,
            IDirectoryConfigurationService directoryConfigurationService,
            IHospitalRepository hospitalRepository,
            ISettingsRepository settingsRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.messageReceiptService = messageReceiptService;
            this.directoryConfigurationService = directoryConfigurationService;

            this.hospitalRepository = hospitalRepository;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets an integer value representing the number of months the "From" date will be offset from the current date by default.
        /// </summary>
        private int FromDateOffsetDays
        {
            get
            {
                return int.Parse(this.Settings.GetSettingValue(Setting.SettingCodes.MessageReceiptViewFromDateOffsetDays));
            }
        }

        #endregion

        #region Methods
        
        /// <summary>
        /// Display initial view for specifying criteria to search.
        /// </summary>
        /// <returns>A view for displaying search criteria</returns>
        public ActionResult Index()
        {
            var viewModel = new ReceiptFilterViewModel()
            {
                From = DateTime.Today.AddDays(this.FromDateOffsetDays),
                To = DateTime.Today.AddDays(1)
            };

            this.LoadReferenceData(viewModel);

            return this.View(viewModel);
        }
        
        /// <summary>
        /// Display inbound messages for specified criteria.
        /// </summary>
        /// <param name="model">View model identifying the messages to be displayed.</param>
        /// <returns>A view for displaying messages.</returns>
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult List(ReceiptFilterViewModel model)
        {
            var messages = new ViewMessageList();
            
            if (ModelState.IsValid)
            {
                var sender = this.SetMessageAddressee(model.Sender);
                var receiver = this.SetMessageAddressee(model.ReceiverHpio);
                var receiptPeriod = new DateTimeRange()
                { 
                    Start = model.From,
                    End = new DateTime(model.To.Year, model.To.Month, model.To.Day, 23, 59, 59)
                };

                var request = this.messageReceiptService.QueryMessages(sender, receiver, receiptPeriod, model.MessageStatusId, model.DocumentTypeCode, this.GetLocalUser(), this.QueryRecordLimit, this.QueryRecordSkipFrom);
                messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));
                
                if (request.IsSuccessful)
                {
                    if (request.Data.Count > 0)
                    {
                        var result = ObjectMapper.Map<List<ListReceiptViewModel>>(request.Data);                        
                        return this.PartialView(result);
                    }
                    
                    messages.Add("No messages are available for the selected criteria.", MessageLevel.Information);
                }
            }
            
            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Aborts a specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        public ActionResult Abort(string invocationIdentifier)
        {
            var request = this.messageReceiptService.AbortResponse(invocationIdentifier, this.GetLocalUser());
            
            var messages = new ViewMessageList();
            messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));

            if (request.IsSuccessful)
            {
                messages.Add("Transport response successfully aborted. You will need to Remove the message to prevent it from publishing.", MessageLevel.Information);
            }

            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Retries a specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>        
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        public ActionResult Retry(string invocationIdentifier)
        {
            var request = this.messageReceiptService.RetryResponse(invocationIdentifier, this.GetLocalUser());

            var messages = new ViewMessageList();
            messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));

            if (request.IsSuccessful)
            {
                messages.Add("Message was successfully submitted for retry.", MessageLevel.Information);
            }

            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Removes a specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>        
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        public ActionResult Remove(string invocationIdentifier)
        {
            var request = this.messageReceiptService.RemoveMessage(invocationIdentifier, this.GetLocalUser());

            var messages = new ViewMessageList();
            messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));

            if (request.IsSuccessful)
            {
                messages.Add("Message was successfully removed.", MessageLevel.Information);
            }

            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Downloads the message content of the specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        public new ActionResult Content(string invocationIdentifier)
        {
            var request = this.messageReceiptService.GetMessage(invocationIdentifier, this.GetLocalUser());
            var result = request.Data;

            if (request.IsSuccessful)
            {
                if (request.Data.Payload == null)
                {
                    return new HttpNotFoundResult();
                }

                string filename = "content.bin";
                return this.File(request.Data.Payload, MediaTypeNames.Application.Octet, filename);
            }

            throw new NullReferenceException("Request.Data is null");
        }

        #region Helpers

        /// <summary>
        /// Initialise initial data for the filter
        /// </summary>
        /// <param name="model">View Model to fill up with initial data.</param>
        private void LoadReferenceData(ReceiptFilterViewModel model)
        {
            var hospitals = this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem)
                                                   .GroupBy(h => h.HpiO)
                                                   .ToSelectListItems(
                                                        i => i.Key,
                                                        i => string.Format("{0} ({1})", i.First().HpioName, i.Key.ToHpioFormat()),
                                                        i => i.Key == this.SessionConfiguration.RepresentingHospital.Hpio);

            model.Receivers = hospitals;
            model.MessageStatuses = Enum.GetValues(typeof(InboundMessageStatus))
                                        .Cast<InboundMessageStatus>()
                                        .ToSelectListItems(i => ((int)i).ToString(), o => o.ToString());

            var request = this.directoryConfigurationService.ListDocumentTypes(this.GetLocalUser());

            if (request.IsSuccessful)
            {
                model.DocumentTypes = request.Data.ToSelectListItems(d => d.Code, d => d.Description);
            }
        }

        /// <summary>
        /// Creates a new instance of Message Addressee.
        /// </summary>
        /// <param name="hpio">A value to be set on Identifier.Value or Organisation.OrganisationName</param>
        /// <returns>A message addressee</returns>
        private MessageAddressee SetMessageAddressee(string hpio)
        {
            if (!string.IsNullOrEmpty(hpio))
            {
                var messageAddressee = new MessageAddressee();
                messageAddressee.Organisation = new ProviderOrganisationIdentity();

                if (this.ValidateHpio(hpio))
                {
                    messageAddressee.Organisation.Identifier = new ProviderOrganisationIdentifier();
                    messageAddressee.Organisation.Identifier.Type = new ProviderIdentifierType();
                    messageAddressee.Organisation.Identifier.Type.Code = "HPIO";
                    messageAddressee.Organisation.Identifier.Value = hpio.ExtractNumbers();
                }
                else
                {
                    messageAddressee.Organisation.OrganisationName = hpio;
                }

                return messageAddressee;
            }

            return null;
        }

        /// <summary>
        /// Checks if the string is a valid HPI-O.
        /// </summary>
        /// <param name="hpio">String to validate.</param>
        /// <returns>Whether the provided HPI-O was valid.</returns>
        private bool ValidateHpio(string hpio)
        {
            string value = hpio.Replace("-", string.Empty)
                               .Replace(" ", string.Empty)
                               .Trim();

            return Regex.IsMatch(value, @"^\d{16}$");
        }

        #endregion

        #endregion
    }
}
