using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Web.Mvc;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.ViewModels.HpiiSearch
{
    /// <summary>
    /// View model for Searching by demographics.
    /// </summary>
    public class SearchByDemographicsViewModel : IValidatableObject
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="SearchByDemographicsViewModel"/> class.
        /// </summary>
        /// <remarks>
        ///     Called automatically by MVC Model Binding.
        ///     Should initialise collections and complex objects to default/empty values.
        /// </remarks>
        public SearchByDemographicsViewModel()
        {
            this.AustralianAddress = new AustralianAddressViewModel();
            this.InternationalAddress = new InternationalAddressViewModel();
            this.Sexes = Enumerable.Empty<SelectListItem>();
            this.ResponseMessages = new ResponseMessages();
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets HPI-O
        /// </summary>
        public string Hpio { get; set; }

        /// <summary>
        /// Gets or sets Family name of the patient.
        /// </summary>
        [Required]
        [StringLength(40, ErrorMessage = "Family Name must be less than 40 characters long.")]
        [Display(Name = "Family Name")]
        public string FamilyName { get; set; }

        /// <summary>
        /// Gets or sets Given name of the patient.
        /// </summary>
        [StringLength(100, ErrorMessage = "Given Name(s) must be less than 100 characters long.")]
        [Display(Name = "Given Name(s)")]
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets Sex code.
        /// </summary>
        [Required]
        [Display(Name = "Sex")]
        public string SexCode { get; set; }

        /// <summary>
        /// Gets or sets Date of birth of the patient.
        /// </summary>
        [Required]
        [Display(Name = "Date of Birth")]
        [DisplayFormat(DataFormatString = "dd/MM/yyyy")]
        public DateTime? DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets Australian address.
        /// </summary>
        public AustralianAddressViewModel AustralianAddress { get; set; }

        /// <summary>
        /// Gets or sets International address.
        /// </summary>
        public InternationalAddressViewModel InternationalAddress { get; set; }

        /// <summary>
        /// Gets response messages.
        /// </summary>
        public ResponseMessages ResponseMessages { get; internal set; }

        #endregion Properties

        #region Data Source Properties

        /// <summary>
        /// Gets Sex drop down.
        /// </summary>
        public IEnumerable<SelectListItem> Sexes { get; internal set; }

        /// <summary>
        /// Gets Unit type drop down.
        /// </summary>
        public IEnumerable<SelectListItem> UnitTypes
        {
            get { return this.AustralianAddress.UnitTypes; }
            internal set { this.AustralianAddress.UnitTypes = value; }
        }

        /// <summary>
        /// Gets Level type drop down.
        /// </summary>
        public IEnumerable<SelectListItem> LevelTypes
        {
            get { return this.AustralianAddress.LevelTypes; }
            internal set { this.AustralianAddress.LevelTypes = value; }
        }

        /// <summary>
        /// Gets Street type drop down.
        /// </summary>
        public IEnumerable<SelectListItem> StreetTypes
        {
            get { return this.AustralianAddress.StreetTypes; }
            internal set { this.AustralianAddress.StreetTypes = value; }
        }

        /// <summary>
        /// Gets Street suffixes drop down.
        /// </summary>
        public IEnumerable<SelectListItem> StreetSuffixes
        {
            get { return this.AustralianAddress.StreetSuffixes; }
            internal set { this.AustralianAddress.StreetSuffixes = value; }
        }

        /// <summary>
        /// Gets Postal delivery type drop down.
        /// </summary>
        public IEnumerable<SelectListItem> PostalDeliveryTypes
        {
            get { return this.AustralianAddress.PostalDeliveryTypes; }
            internal set { this.AustralianAddress.PostalDeliveryTypes = value; }
        }

        /// <summary>
        /// Gets States drop down.
        /// </summary>
        public IEnumerable<SelectListItem> States
        {
            get { return this.AustralianAddress.States; }
            internal set { this.AustralianAddress.States = value; }
        }

        /// <summary>
        /// Gets Countries drop down.
        /// </summary>
        public IEnumerable<SelectListItem> Countries
        {
            get { return this.InternationalAddress.Countries; }
            internal set { this.InternationalAddress.Countries = value; }
        }

        #endregion Data Source Properties

        /// <summary>
        ///     Determines whether the specified object is valid.
        /// </summary>
        /// <returns>
        ///     A collection that holds failed-validation information.
        /// </returns>
        /// <param name="validationContext">The validation context.</param>
        /// <remarks>
        ///     Executed server-side after all annotation validators are successful.
        /// </remarks>
        public IEnumerable<ValidationResult> Validate(ValidationContext validationContext)
        {
            List<ValidationResult> errors = new List<ValidationResult>();

            if (!this.AustralianAddress.HasAnyValue() && !this.InternationalAddress.HasAnyValue())
            {
                errors.Add(new ValidationResult("Please provide either an Australian Address or an International Address."));
            }

            if (this.AustralianAddress.HasAnyValue() && this.InternationalAddress.HasAnyValue())
            {
                errors.Add(new ValidationResult("Please provide only one of an Australian Address or an International Address."));
            }

            return errors;
        }
    }
}