﻿using System.ComponentModel.DataAnnotations;

using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.ViewModels.PcehrView
{
    /// <summary>
    /// Base view model to be used with the PCEHR View feature.
    /// </summary>
    public class PcehrViewModelBase
    {
        #region Properties

        /// <summary>
        /// Gets or sets the list of messages associated with the view model.
        /// </summary>
        public ViewMessageList Messages { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected hospital.
        /// </summary>
        [Required]
        [Display(Name = "Hospital")]
        public string HospitalId { get; set; }

        /// <summary>
        /// Gets or sets the currently selected hospital.
        /// </summary>
        public HospitalViewModel CurrentHospital { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected patient.
        /// </summary>
        public string PatientId { get; set; }

        /// <summary>
        /// Gets or sets the currently selected patient.
        /// </summary>
        public PatientViewModel CurrentPatient { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected document repository.
        /// </summary>
        public string RepositoryId { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected document.
        /// </summary>
        public string DocumentId { get; set; }

        /// <summary>
        /// Gets or sets the currently selected document.
        /// </summary>
        public DocumentViewModel CurrentDocument { get; set; }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        public PcehrViewModelBase()
        {
            this.Messages = new ViewMessageList();
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Determines if the view model contains a selected hospital.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected hospital, false otherwise.</returns>
        public bool IsHospitalSelected()
        {
            return !(string.IsNullOrEmpty(this.HospitalId));
        }

        /// <summary>
        /// Determines if the view model contains a selected patient.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected patient, false otherwise.</returns>
        public bool IsPatientSelected()
        {
            return this.IsHospitalSelected() && (!string.IsNullOrEmpty(this.PatientId)) && this.CurrentPatient != null;
        }

        /// <summary>
        /// Determines if the view model contains a selected document.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected document, false otherwise.</returns>
        public bool IsDocumentSelected()
        {
            return this.IsHospitalSelected() && this.IsPatientSelected() && (!string.IsNullOrEmpty(this.RepositoryId) && !string.IsNullOrEmpty(this.DocumentId));
        }

        #endregion Methods
    }
}