﻿using System;
using System.Collections.Generic;

using HIPS.Client.Proxy;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Patient.Message;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.Hips.Mapping;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.ConsentManagement;

using CommonSchema = HIPS.CommonSchemas;
using ServiceSchema = HIPS.ServiceContracts.Common.DTO;

namespace HIPS.Web.Data.Hips.ConsentManagement
{
    /// <summary>
    /// Implements a repository for accessing episodes of care for a patient.
    /// </summary>
    public class EpisodeRepositoryV2 : RepositoryBase<PatientServiceV2Proxy>, IEpisodeRepositoryV2
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="EpisodeRepositoryV2"/> class.
        /// </summary>
        /// <param name="daysDischarged">Integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".</param>
        public EpisodeRepositoryV2(int daysDischarged)
            : base(null, null)
        {
            this.ClientProxy = new PatientServiceV2Proxy("PatientEndPointV2");
            this.DaysDischarged = daysDischarged;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets an integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".
        /// </summary>
        public int DaysDischarged { get; private set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Retrieves a list of recent episodes for a patient at a hospital.
        /// </summary>
        /// <param name="patientIdentifier">The MRN that identifies the patient at the hospital.</param>
        /// <param name="includeDocuments">A value indicating whether documents should be included as part of episode response.</param>
        /// <param name="documentTypeCode">Code of the document to be included.</param>
        /// <param name="userContext">User requesting the operation.</param>
        /// <returns>ServiceResponse containing a PatientEpisodeListResponse containing the results of the operation.</returns>
        public ServiceResponse<List<PatientEpisode>> ListPatientEpisodesInHospital(CommonSchema.PatientIdentifier.Mrn patientIdentifier, bool includeDocuments, string documentTypeCode, Model.Identity.LocalUser userContext)
        {
            var messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new ListPatientEpisodesInHospitalRequest();
                request.DaysDischarged = this.DaysDischarged;
                request.IncludeDocuments = includeDocuments;
                request.DocumentTypeCode = documentTypeCode;
                request.PatientIdentifier = ObjectMapper.Map<ServiceSchema.PatientIdentifier.Mrn>(patientIdentifier);
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext);

                var result = this.ClientProxy.ListPatientEpisodesInHospital(request);

                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<PatientEpisode>>(ObjectMapper.Map<List<PatientEpisode>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<PatientEpisode>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Retrieves a list of recent episodes for a patient at a hospital.
        /// </summary>
        /// <param name="patientIdentifier">The MRN that identifies the patient at the hospital.</param>
        /// <param name="includeDocuments">A value indicating whether documents should be included as part of episode response.</param>
        /// <param name="documentTypeCode">Code of the document to be included.</param>
        /// <param name="userContext">User requesting the operation.</param>
        /// <returns>ServiceResponse containing a PatientEpisodeListResponse containing the results of the operation.</returns>
        public ServiceResponse<List<PatientEpisode>> ListPatientEpisodesInHospital(CommonSchema.PatientIdentifier.RegisteredEnterprisePatient patientIdentifier, bool includeDocuments, string documentTypeCode, Model.Identity.LocalUser userContext)
        {
            var messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new ListPatientEpisodesInHospitalRequest();
                request.DaysDischarged = this.DaysDischarged;
                request.IncludeDocuments = includeDocuments;
                request.DocumentTypeCode = documentTypeCode;
                request.PatientIdentifier = ObjectMapper.Map<ServiceSchema.PatientIdentifier.RegisteredEnterprisePatient>(patientIdentifier);
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext);

                var result = this.ClientProxy.ListPatientEpisodesInHospital(request);

                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<PatientEpisode>>(ObjectMapper.Map<List<PatientEpisode>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<PatientEpisode>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        #endregion Methods
    }
}