﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using HIPS.Client.Proxy;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Hpii.Message;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.Hips.Mapping;
using HIPS.Web.ModelInterface.HpiiSearch;

using CommonSchema = HIPS.Web.Model.Hpii;
using ServiceSchema = HIPS.ServiceContracts.Common.DTO;

namespace HIPS.Web.Data.Hips.HpiiSearch
{
    /// <summary>
    /// Defines the implementation of the Hpii Search Service (v2) interface.
    /// </summary>
    public class HpiiSearchServiceV2 : IHpiiSearchServiceV2
    {
        /// <summary>
        /// The client proxy used by the service.
        /// </summary>
        private HpiiServiceV2Proxy proxy = new HpiiServiceV2Proxy("HpiiEndpointV2");

        /// <summary>
        /// List all of the current local providers for the specified facility.
        /// </summary>
        /// <param name="hospitalCode">Hospital Code.</param>
        /// <param name="hospitalCodeSystem">Hospital Code System.</param>
        /// <param name="user">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<List<CommonSchema.LocalProvider>> ListLocalProviders(string hospitalCode, string hospitalCodeSystem, Model.Identity.LocalUser user)
        {
            var messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new ListLocalProviderRequest();
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(user);
                request.HospitalIdentifier = new ServiceSchema.HospitalIdentifier() { HospitalCode = hospitalCode, HospitalCodeSystem = hospitalCodeSystem };

                var response = this.proxy.ListLocalProviders(request);

                // Remove local providers without a local provider code. A valid local provider without a local provider code should never exist.
                response.Data.RemoveAll(provider => provider.LocalProviderCode == null);

                // HIPS Status should be OK
                isSuccessful = response.Status == ResponseStatus.OK;
                var result = ObjectMapper.Map<List<CommonSchema.LocalProvider>>(response.Data);

                return new ServiceResponse<List<CommonSchema.LocalProvider>>(result, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.LocalProvider>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Add or Edit a Local Provider.
        /// </summary>
        /// <param name="hospitalCode">Hospital Code.</param>
        /// <param name="hospitalCodeSystem">Hospital Code System.</param>
        /// <param name="hpii">Hpii.</param>
        /// <param name="title">Title.</param>
        /// <param name="familyName">Family Name.</param>
        /// <param name="givenNames">Given Names.</param>
        /// <param name="suffix">Suffix.</param>
        /// <param name="user">User context within which the service is accessed.</param>
        /// <param name="iamProviderIdentifier">Iam Provider Identifier</param>
        /// <param name="localProviderCode">Local Provider Code.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<AddEditLocalProviderResponse> AddEditLocalProvider(
            string hospitalCode,
            string hospitalCodeSystem,
            string hpii,
            string title,
            string familyName,
            string givenNames,
            string suffix,
            Model.Identity.LocalUser user,
            string iamProviderIdentifier = null,
            string localProviderCode = null)
        {
            var messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new AddEditLocalProviderRequest();
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(user);
                request.HospitalIdentifier = new ServiceSchema.HospitalIdentifier() { HospitalCode = hospitalCode, HospitalCodeSystem = hospitalCodeSystem };
                request.Hpii = hpii;                
                request.FamilyName = familyName;
                request.GivenNames = givenNames;
                request.Title = title;
                request.Suffix = suffix;
                request.IamProviderIdentifier = iamProviderIdentifier;
                request.LocalProviderCode = localProviderCode;

                var response = this.proxy.AddEditLocalProvider(request);
                
                // HIPS Status should be OK
                isSuccessful = response.Status == ResponseStatus.OK;

                return new ServiceResponse<AddEditLocalProviderResponse>(response, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));
                
                return new ServiceResponse<AddEditLocalProviderResponse>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Add or Replace a Local Provider Identifier.
        /// </summary>
        /// <param name="localProviderIdentifier">Local Provider Identifier.</param>
        /// <param name="replace">A value indicating whether the Local Provider Identifier will be replaced.</param>
        /// <param name="user">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<AddReplaceLocalProviderIdentifierResponse> AddReplaceLocalProviderIdentifier(CommonSchema.LocalProviderIdentifier localProviderIdentifier, bool replace, Model.Identity.LocalUser user)
        {
            var messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new AddReplaceLocalProviderIdentifierRequest();
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(user);
                request.LocalProviderIdentifier = ObjectMapper.Map<HIPS.ServiceContracts.Hpii.DTO.LocalProviderIdentifier>(localProviderIdentifier);
                request.Replace = replace;

                var response = this.proxy.AddReplaceLocalProviderIdentifier(request);
                
                // HIPS Status should be OK
                isSuccessful = response.Status == ResponseStatus.OK;

                return new ServiceResponse<AddReplaceLocalProviderIdentifierResponse>(response, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<AddReplaceLocalProviderIdentifierResponse>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Deactivates or removes deactivation a local provider.
        /// </summary>
        /// <param name="localProviderCode">Local Provider Code.</param>
        /// <param name="deactivatedDate">Deactivated Date.</param>
        /// <param name="user">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<DeactivateLocalProviderResponse> DeactivateLocalProvider(string hospitalCode,
            string hospitalCodeSystem, string localProviderCode, DateTime? deactivatedDate, Model.Identity.LocalUser user)
        {
            var messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new DeactivateLocalProviderRequest();
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(user);
                request.HospitalIdentifier = new ServiceSchema.HospitalIdentifier() { HospitalCode = hospitalCode, HospitalCodeSystem = hospitalCodeSystem };
                request.LocalProviderCode = localProviderCode;
                request.DeactivatedDate = deactivatedDate;

                var response = this.proxy.DeactivateLocalProvider(request);

                // HIPS Status should be OK
                isSuccessful = response.Status == ResponseStatus.OK;

                return new ServiceResponse<DeactivateLocalProviderResponse>(response, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<DeactivateLocalProviderResponse>(null, isSuccessful, messages, DateTime.Now);
            }
        }
    }
}
