﻿using System;
using System.Collections.Generic;

using HIPS.Client.Proxy;
using HIPS.ServiceContracts.Cda.Message;
using HIPS.ServiceContracts.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.Hips.Mapping;
using HIPS.Web.ModelInterface.PcehrView;

using CommonSchema = HIPS.Web.Model;
using ServiceSchema = HIPS.ServiceContracts.Common.DTO;

namespace HIPS.Web.Data.Hips.PcehrView
{
    /// <summary>
    /// Implements a repository for CDA documents operations.
    /// </summary>
    public class CdaRepository : ICdaRepository
    {
        /// <summary>
        /// The client proxy.
        /// </summary>
        private CdaProxy proxy = new CdaProxy("CdaEndpoint");

        #region Methods

        /// <summary>
        /// Creates a CDA discharge summary document that wraps a PDF document body.
        /// This may be for the purpose of uploading the discharge summary to the
        /// PCEHR or for provider-to-provider (P2P) secure message delivery (SMD).
        /// This takes in the PDF document, required metadata, and any attachments
        /// and creates a CDA document before packaging it.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="metadata">The data that is required to be added to finalise the CDA document.</param>
        /// <param name="attachments">The attached files (GIF, JPEG, PNG, TIFF or PDF) that will be referenced by the CDA document.
        /// Note that it is not necessary to include organisation logo file in the array because HIPS can automatically add the logo
        /// that is configured in the Hospital table of the PCEHR Data Store.</param>
        /// <param name="pdfDocument">PDF discharge summary document.</param>
        /// <param name="userContext">User requesting the operation.</param>
        /// <returns>CdaDocumentDetails containing the results of the operation.</returns>
        public ServiceResponse<CommonSchema.Cda.CdaDocumentDetails> CreateDischargeSummaryLevel1A(
            CommonSchema.PatientIdentifier.Mrn patientIdentifier,
            CommonSchema.Cda.CdaHeaderMetadata metadata,
            List<CommonSchema.Cda.CdaAttachment> attachments,
            byte[] pdfDocument,
            CommonSchema.Identity.LocalUser userContext)
        {
            var messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new CreateDischargeSummaryLevel1ARequest();
                request.PatientIdentifier = ObjectMapper.Map<ServiceSchema.PatientIdentifier.Mrn>(patientIdentifier);
                request.CdaHeaderMetadata = ObjectMapper.Map<ServiceSchema.CdaHeaderMetadata>(metadata);
                request.Attachments = ObjectMapper.Map<List<ServiceSchema.CdaAttachment>>(attachments);
                request.PdfDocument = pdfDocument;

                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext);

                var result = this.proxy.CreateDischargeSummaryLevel1A(request);

                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                return new ServiceResponse<CommonSchema.Cda.CdaDocumentDetails>(ObjectMapper.Map<CommonSchema.Cda.CdaDocumentDetails>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.Cda.CdaDocumentDetails>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Safely disposes the client proxy.
        /// </summary>
        public void Dispose()
        {
            if (this.proxy != null)
            {
                this.proxy.DisposeSafely();
                this.proxy = null;
            }
        }

        #endregion Methods
    }
}