﻿using System;
using System.IO;
using System.Linq;

namespace HIPS.Web.Model.Cda
{
    /// <summary>
    /// Represents an attachment to a CDA document.
    /// </summary>
    public class CdaAttachment
    {
        private const string LOGO = "logo";

        /// <summary>
        /// Gets or sets the binary content of the attachment.
        /// </summary>
        public byte[] Content { get; set; }

        /// <summary>
        /// Gets or sets the file name of the attachment.
        /// </summary>
        public string FileName { get; set; }

        /// <summary>
        /// Gets the caption that will appear on the link to this file
        /// within the CDA document.
        /// </summary>
        public string Caption
        {
            get { return string.Format("{0}: {1}", this.AttachmentType, this.FileName); }
        }

        /// <summary>
        /// Gets the type of attachment, including whether it is the
        /// organisational logo or an additional attached image or document.
        /// </summary>
        public AttachmentType AttachmentType
        {
            get { return this.SetAttachmentType(); }
        }

        private AttachmentType SetAttachmentType()
        {
            string[] validImageExtensions = { ".png", ".gif", ".tiff", ".jpeg", ".jpg" };

            string fileNameWithoutExtension = Path.GetFileNameWithoutExtension(this.FileName);
            string extension = Path.GetExtension(this.FileName);

            if (validImageExtensions.Contains(extension))
            {
                if (fileNameWithoutExtension.Equals(LOGO, StringComparison.OrdinalIgnoreCase) && extension == ".png")
                {
                    return Model.AttachmentType.OrganisationalLogoPng;
                }
                else
                {
                    if (extension == ".png")
                    {
                        return AttachmentType.AttachedImagePng;
                    }
                    else if (extension == ".gif")
                    {
                        return AttachmentType.AttachedImageGif;
                    }
                    else if (extension == ".tiff")
                    {
                        return AttachmentType.AttachedImageTiff;
                    }
                    else if (extension == ".jpeg" || extension == ".jpg")
                    {
                        return AttachmentType.AttachedImageJpeg;
                    }
                }
            }
            else
            {
                return AttachmentType.AttachedDocumentPdf;
            }

            return AttachmentType.None;
        }
    }
}