﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Mime;
using System.Text.RegularExpressions;
using System.Web.Mvc;

using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.Web;
using HIPS.Web.Model;
using HIPS.Web.Model.Common;
using HIPS.Web.Model.Messaging;
using HIPS.Web.Model.Directory.Search;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Directory;
using HIPS.Web.ModelInterface.Messaging;
using HIPS.Web.ModelInterface.Subscription;
using HIPS.Web.UI.Areas.Subscription.ViewModels;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.Shared;
using System.Web.SessionState;
using System.Web.Script.Serialization;
using HIPS.Web.Model.Subscription; 

namespace HIPS.Web.UI.Areas.Subscription.Controllers
{
    /// <summary>
    /// Controller for the Subscription Maintenance.
    /// </summary>    
    [NoCache]
    [HpoRequired]
    public class SubscriptionController : HIPS.Web.UI.Controllers.ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly ISubscriptionService subscriptionRepository;

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IReferenceDataService referenceRepository;

        /// <summary>
        /// Subscription reference repository to be used by this controller.
        /// </summary>
        private readonly ISubscriptionReferenceRepository subscriptionReferenceRepository;

        /// <summary>
        /// Provider Individual Directory reference repository to be used by this controller.
        /// </summary>
        private readonly IProviderIndividualDirectoryService providerIndividualDirectoryRepository;

        /// <summary>
        /// The URI of the State
        /// </summary>
        private const string stateUri = "urn:x-nhsd.au:set:general;16072014;state";

        /// <summary>
        /// The URI of the Suburb
        /// </summary>
        private const string suburbUri = "urn:x-nhsd.au:set:general;16072014;suburb";

        /// <summary>
        /// The URI of the Individual Type
        /// </summary>
        private const string providerIndividualTypeUri = "urn:x-nhsd.au:set:hiservice;16072014;provider_individual_type";

        /// <summary>
        /// The URI of the Individual Specialty
        /// </summary>
        private const string providerIndividualSpecialtyUri = "urn:x-nhsd.au:set:hiservice;16072014;provider_individual_specialty";

        /// <summary>
        /// The URI of the Individual Specialisation
        /// </summary>
        private const string providerIndividualSpecialisationUri = "urn:x-nhsd.au:set:hiservice;16072014;provider_individual_specialisation";

        /// <summary>
        /// The URI of the Organisation Type
        /// </summary>
        private const string providerOrganisationTypeUri = "urn:x-nhsd.au:set:hiservice;16072014;provider_organisation_type";

        /// <summary>
        /// The URI of the Organisation Service Type
        /// </summary>
        private const string providerOrganisationServiceTypeUri = "urn:x-nhsd.au:set:hiservice;16072014;provider_organisation_services_type";

        /// <summary>
        /// The URI of the Organisation Service Unit
        /// </summary>
        private const string providerOrganisationServiceUnitUri = "urn:x-nhsd.au:set:hiservice;16072014;provider_organisation_service_unit";

        /// <summary>
        /// The asterisk used in different parts of the controllers
        /// </summary>
        private const string asterisk = "*";

        #endregion

        #region Constructor

        /// <summary>
        /// Initialises a new instance of the <see cref="DeliveryController"/> class.
        /// </summary>
        /// <param name="messageDeliveryService">Message Delivery service to be used by this controller.</param>
        /// <param name="directoryConfigurationService">Directory configuration service to be used by this controller.</param>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public SubscriptionController(
            ISubscriptionService subscriptionService,            
            ISettingsRepository settingsRepository,
            IReferenceDataService referenceRepository,
            ISubscriptionReferenceRepository subscriptionReferenceRepository,
            IProviderIndividualDirectoryService providerIndividualReferenceRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.subscriptionRepository = subscriptionService;
            this.referenceRepository = referenceRepository;
            this.subscriptionReferenceRepository = subscriptionReferenceRepository;
            this.providerIndividualDirectoryRepository = providerIndividualReferenceRepository;
        }

        #endregion

        #region Methods

        /// <summary>
        /// Display initial view for Areas of Interest.
        /// </summary>
        /// <returns>A view for displaying Areas of Interest</returns>     
        public ActionResult AreasOfInterest()
        {
            var response = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser());
            var model = new ListAreasOfInterestViewModel();
            model.Messages = new ViewMessageList();            

            if (response.IsSuccessful)
            {
                if (response.Data.Count > 0)
                {
                    model.AreasOfInterest = ObjectMapper.Map<List<AreaOfInterestViewModel>>(response.Data);
                }
                else
                {
                    model.Messages.Add("No areas of interest found.", MessageLevel.Information);
                }
            }

            return this.View("AreasOfInterest", model);
        }

        /// <summary>
        /// Display the list Areas of Interest.
        /// </summary>
        /// <returns>A view for displaying Areas of Interest</returns> 
        [HttpPost]
        public ActionResult ListAreasOfInterest()
        {
            var response = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser());
            var model = new ListAreasOfInterestViewModel();
            model.Messages = new ViewMessageList();

            if (response.IsSuccessful)
            {
                if (response.Data.Count > 0)
                {
                    model.AreasOfInterest = ObjectMapper.Map<List<AreaOfInterestViewModel>>(response.Data);
                }
                else
                {
                    model.Messages.Add("No areas of interest found.", MessageLevel.Information);
                }
            }

            return this.PartialView("_ListAreasOfInterest", model);
        }

        /// <summary>
        /// Display the Edit Area of Interest Modal
        /// </summary>
        /// <returns>A view for displaying Areas of Interest</returns>   
        [HttpGet]
        public ActionResult Edit(string code)
        {
            var model = PopulateModal();

            if (!string.IsNullOrEmpty(code))
            {
                var areasOfInterest = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser()).Data.Find(i => i.Code == code);

                model.Description = areasOfInterest.Description;
                model.Code = areasOfInterest.Code;
                model.isEdit = true;

                // Check If the Individual Occupations in Grid exists in the database.
                foreach (var occupation in areasOfInterest.Occupations)
                {
                    var isExist = model.Occupations.Find(i => i.Code == occupation.Code);
                    if (isExist != null)
                    {
                        isExist.isExists = true;
                    }
                }

                // Check If the Organisation Classifications in Grid exists in the database.
                foreach (var classification in areasOfInterest.Classifications)
                {
                    var isExist = model.Classifications.Find(i => i.Code == classification.Code);
                    if (isExist != null)
                    {
                        isExist.isExists = true;
                    }
                }

                model.Individuals.IndiCode = code;
                model.Organisations.OrgCode = code;
            }

            return this.PartialView("Edit", model);
        }

        /// <summary>
        /// Populate the Geographical Areas Grid in the Modal
        /// </summary>
        /// <returns>A partial view returning the results of the action</returns>   
        [HttpPost]
        public ActionResult GeographicalAreasGrid(string code)
        {
            // Initializations
            var model = new AreaOfInterestViewModel();
            var suburbsList = this.referenceRepository.Search(suburbUri, asterisk, asterisk, this.GetLocalUser());
            var statesList = this.referenceRepository.Search(stateUri, asterisk, asterisk, this.GetLocalUser());

            if (suburbsList.IsSuccessful)
            {
                if (suburbsList.Data.Count > 0)
                {
                    model.Areas = ObjectMapper.Map<List<GeographicalAreaViewModel>>(suburbsList.Data);
                }
            }

            if (statesList.IsSuccessful)
            {
                if (statesList.Data.Count > 0)
                {
                    foreach (var state in statesList.Data)
                    {
                        model.Areas.Add(new GeographicalAreaViewModel()
                        {
                            Suburb = asterisk,
                            State = state.Description,
                            Postcode = asterisk,
                            isExists = false
                        });
                    }
                }
            }

            // If code is not null then this is in Edit Mode and display the existing values from the ListSubscriptionDefinitions service
            if (!string.IsNullOrEmpty(code))
            {
                var areasOfInterest = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser()).Data.Find(i => i.Code == code);

                foreach (var area in areasOfInterest.Areas)
                {
                    if (string.IsNullOrEmpty(area.Suburb) && string.IsNullOrEmpty(area.Postcode))
                    {
                        // Display that the territory does exists in the database
                        var stateArea = model.Areas.Find(i => i.State == area.State && i.Suburb == "*" && i.Postcode == "*");
                        if (stateArea != null)
                        {
                            stateArea.isExists = true;
                        }
                    }
                    else
                    {
                        // Display that the suburb / area does exists in the database
                        var geographicalArea = model.Areas.Find(i => i.State == area.State && i.Postcode == area.Postcode && i.Suburb == area.Suburb);
                        if (geographicalArea != null)
                        {
                            geographicalArea.isExists = true;
                        }
                    }
                }
            }

            return this.PartialView("_ListGeographicalAreas", model.Areas);
        }

        /// <summary>
        /// Save or Edit the Area of interest
        /// </summary>
        /// <returns>A view for displaying Areas of Interest</returns>   
        [HttpPost]
        public ActionResult Save(AreaOfInterestViewModel model)
        {
            // Initializations
            var listModel = new ListAreasOfInterestViewModel();
            listModel.Messages = new ViewMessageList();
            
            // Validate Requests
            listModel.Messages = ValidateModel(model);

            if (listModel.Messages.Count > 0)
            {
                return this.PartialView("ViewMessageList", listModel.Messages);
            }

            var areaOfInterest = ObjectMapper.Map<AreaOfInterest>(model);
            var extraMembers = new List<ProviderIdentifierSearch>();

            // Populate Extra Members
            if (model.ExtraIndividual != null)
            {
                foreach (var individual in model.ExtraIndividual.Where(i => i.ExtraMember != null))
                {
                    foreach (var identifier in individual.ExtraMember)
                    {
                        extraMembers.Add(new ProviderIdentifierSearch()
                        {
                            TypeCode = identifier.TypeCode
                            ,
                            Value = identifier.Value
                        });
                    }
                }
            }

            if (model.ExtraOrganisation != null)
            {
                foreach (var organisation in model.ExtraOrganisation.Where(i => i.ExtraMember != null))
                {
                    foreach (var identifier in organisation.ExtraMember)
                    {
                        extraMembers.Add(new ProviderIdentifierSearch()
                        {
                            TypeCode = identifier.TypeCode,
                            Value = identifier.Value
                        });
                    }
                }
            }

            areaOfInterest.ExtraMembers = extraMembers;

            // If Code is not null then the Action is Edit
            if (!string.IsNullOrEmpty(model.Code))
            {
                var response = this.subscriptionRepository.UpdateAreaOfInterest(areaOfInterest, this.GetLocalUser());

                // Check if Response is successful, display success message else display error message
                if (response.IsSuccessful)
                {
                    listModel.Messages.Add("Area of Interest successfully updated.", MessageLevel.Information);
                }
                else
                {
                    listModel.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
                }
            }
            else
            {
                var response = this.subscriptionRepository.AddAreaOfInterest(areaOfInterest, this.GetLocalUser());

                // Check if Response is successful, display success message else display error message
                if (response.IsSuccessful)
                {
                    listModel.Messages.Add("Area of Interest successfully added.", MessageLevel.Information);
                }
                else
                {
                    listModel.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
                }
            } 
            
            return this.PartialView("ViewMessageList", listModel.Messages);
        }

        /// <summary>
        /// Delete the Area of interest
        /// </summary>
        /// <returns>A view for displaying the results of the action</returns>   
        [HttpPost]
        public ActionResult Delete(AreaOfInterestViewModel model)
        {
            // Initializations
            var listModel = new ListAreasOfInterestViewModel();
            listModel.Messages = new ViewMessageList();
            var areaOfInterest = ObjectMapper.Map<AreaOfInterest>(model);

            var response = this.subscriptionRepository.DeleteAreaOfInterest(areaOfInterest, this.GetLocalUser());

            // Check if Response is successful, display success message else display error message
            if (response.IsSuccessful)
            {
                listModel.Messages.Add("Area of Interest successfully deleted.", MessageLevel.Information);
            }
            else
            {
                listModel.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
            }

            return this.PartialView("ViewMessageList", listModel.Messages);
        }

        /// <summary>
        /// Searches for the Extra Individual
        /// </summary>
        /// <returns>A view for displaying the results of the action</returns>   
        [HttpPost]
        public ActionResult SearchIndividual(ExtraIndividualSearchViewModel model)
        {
            // Initializations
            var listModel = new ListAreasOfInterestViewModel();
            listModel.Messages = new ViewMessageList();
            var individuals = new List<ExtraIndividualViewModel>();
            var extraIndividuals = new List<ProviderIdentifierSearchViewModel>();

            // Validate Requests
            listModel.Messages = ValidateSearchIndividualRequest(model);
            if (listModel.Messages.Count > 0)
            {
                return this.PartialView("ViewMessageList", listModel.Messages);
            }

            var response = this.subscriptionRepository.FindIndividuals(model.FamilyName, model.GivenName, model.Occupation, model.Suburb, model.State, model.Postcode, model.ProximityRadius, this.GetLocalUser());

            if (response.IsSuccessful)
            {
                foreach (var individual in response.Data.Individuals)
                {
                    // Declarations
                    bool isExists = false;
                    var occupation = string.Empty;

                    foreach (var occ in individual.Occupations)
                    {
                        occupation = (string.IsNullOrEmpty(occupation) ? occ.DisplayName : occupation + ", " + occ.DisplayName);
                    }

                    // For the Individual Location
                    var location = string.Empty;
                    var locations = from loc in response.Data.Locations
                                    join link in response.Data.Links on loc.LocationId equals link.LocationId
                                    where link.IndividualId == individual.IndividualId
                                    select loc;


                    foreach (var loc in locations)
                    {
                        location = (string.IsNullOrEmpty(location) ? loc.LocationName : location + ", " + loc.LocationName);
                    }

                    foreach (var identifier in individual.Identifiers.Where(i => i.Type.Code.Contains("NHSD")))
                    {
                        extraIndividuals.Add(new ProviderIdentifierSearchViewModel()
                        {
                            TypeCode = identifier.Type.Code,
                            Value = identifier.Value
                        });

                        // This is to validate if the Result is already under the Extra Members of the Area of Interest
                        if (!string.IsNullOrEmpty(model.IndiCode))
                        {
                            var areasOfInterest = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser()).Data.Find(i => i.Code == model.IndiCode);
                            isExists = areasOfInterest.ExtraMembers.Any(i => i.TypeCode == identifier.Type.Code && i.Value == identifier.Value);
                        }
                    }

                    individuals.Add(new ExtraIndividualViewModel()
                    {
                        FamilyName = individual.Name.FamilyName,
                        GivenName = individual.Name.GivenNames,
                        Occupation = occupation,
                        Location = location,
                        isMember = individual.IsInSubscription,
                        ExtraMember = extraIndividuals.ToList(),
                        isExists = isExists
                    });

                    extraIndividuals.Clear();
                }
            }
            else
            {
                listModel.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
                return this.PartialView("ViewMessageList", listModel.Messages);
            }

            return this.PartialView("_ListExtraIndividuals", individuals);
        }

        /// <summary>
        /// Searches for the Extra Organisation
        /// </summary>
        /// <returns>A view for displaying the results of the action</returns>   
        [HttpPost]
        public ActionResult SearchOrganisation(ExtraOrganisationSearchViewModel model)
        {
            // Initializations
            var listModel = new ListAreasOfInterestViewModel();
            listModel.Messages = new ViewMessageList();
            var organisations = new List<ExtraOrganisationViewModel>();
            var extraOrganisations = new List<ProviderIdentifierSearchViewModel>();

            // Validate Requests
            listModel.Messages = ValidateSearchOrganisationRequest(model);
            if (listModel.Messages.Count > 0)
            {
                return this.PartialView("ViewMessageList", listModel.Messages);
            }

            var response = this.subscriptionRepository.FindOrganisations(model.OrganisationName, model.Classification, model.OrgSuburb, model.OrgState, model.OrgPostcode, model.OrgProximityRadius, this.GetLocalUser());

            if (response.IsSuccessful)
            {
                foreach (var organisation in response.Data.Organisations)
                {
                    bool isExists = false;
                    var location = string.Empty;
                    var classification = string.Empty;

                    foreach (var loc in response.Data.Locations.Where(i => i.OrganisationId == organisation.OrganisationId))
                    {
                        location = (string.IsNullOrEmpty(location) ? loc.LocationName : location + ", " + loc.LocationName);
                        foreach (var classifications in loc.Classifications)
                        {
                            classification = (string.IsNullOrEmpty(classification) ? classifications.DisplayName : classification + ", " + classifications.DisplayName);
                        }
                    }

                    foreach (var identifier in organisation.Identifiers.Where(i => i.Type.Code.Contains("NHSD")))
                    {
                        extraOrganisations.Add(new ProviderIdentifierSearchViewModel()
                        {
                            TypeCode = identifier.Type.Code,
                            Value = identifier.Value
                        });

                        // This is to validate if the Result is already under the Extra Members of the Area of Interest
                        if (!string.IsNullOrEmpty(model.OrgCode))
                        {
                            var areasOfInterest = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser()).Data.Find(i => i.Code == model.OrgCode);
                            isExists = areasOfInterest.ExtraMembers.Any(i => i.TypeCode == identifier.Type.Code && i.Value == identifier.Value);
                        }
                    }

                    organisations.Add(new ExtraOrganisationViewModel()
                    {
                        OrganisationName = organisation.LegalName,
                        LocationName = location,
                        Classification = classification,
                        isMember = organisation.IsInSubscription,
                        ExtraMember = extraOrganisations.ToList(),
                        isExists = isExists
                    });

                    extraOrganisations.Clear();
                }
            }
            else
            {
                listModel.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
                return this.PartialView("ViewMessageList", listModel.Messages);
            }

            return this.PartialView("_ListExtraOrganisations", organisations);
        }

        /// <summary>
        /// Searches for the Current Extra Individual
        /// </summary>
        /// <returns>A view for displaying the results of the action</returns>   
        [HttpPost]
        public ActionResult SearchCurrentIndividual(string code)
        {
            // Initializations
            var listModel = new ListAreasOfInterestViewModel();
            listModel.Messages = new ViewMessageList();

            var areasOfInterest = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser()).Data.Find(i => i.Code == code);     

            var entities = new List<Entity>();
            entities.Add(Entity.Individual);
            entities.Add(Entity.Link);
            entities.Add(Entity.Location);
            entities.Add(Entity.IndividualClassification);
            entities.Add(Entity.Organisation);

            var individuals = new List<ExtraIndividualViewModel>();
            var extraMembers = new List<ProviderIdentifierSearchViewModel>();

            foreach (var extraMember in areasOfInterest.ExtraMembers.Where(i => i.Value.Contains("urn:x-nhsd.au:pers")))
            {
                var identifierParam = new ProviderIdentifierSearch()
                {
                    TypeCode = extraMember.TypeCode,
                    Value = extraMember.Value
                };
                var individualParam = new ProviderIndividualSearch()
                {
                    Identifier = identifierParam
                };

                var response = this.providerIndividualDirectoryRepository.Search(individualParam, null, null, null, entities, null, null, this.GetLocalUser());

                if (response.IsSuccessful)
                {
                    foreach (var individual in response.Data.Individuals)
                    {
                        // Declarations
                        var occupation = string.Empty;

                        foreach (var occ in individual.Occupations)
                        {
                            occupation = (string.IsNullOrEmpty(occupation) ? occ.DisplayName : occupation + ", " + occ.DisplayName);
                        }

                        // For the Individual Location
                        var location = string.Empty;
                        var locations = from loc in response.Data.Locations
                                        join link in response.Data.Links on loc.LocationId equals link.LocationId
                                        where link.IndividualId == individual.IndividualId
                                        select loc;

                        foreach (var loc in locations)
                        {
                            location = (string.IsNullOrEmpty(location) ? loc.LocationName : location + ", " + loc.LocationName);
                        }

                        foreach (var iden in individual.Identifiers.Where(i => i.Type.Code.Contains("NHSD")))
                        {
                            extraMembers.Add(new ProviderIdentifierSearchViewModel()
                            {
                                TypeCode = iden.Type.Code,
                                Value = iden.Value
                            });
                        }

                        individuals.Add(new ExtraIndividualViewModel()
                        {
                            FamilyName = individual.Name.FamilyName,
                            GivenName = individual.Name.GivenNames,
                            Occupation = occupation,
                            Location = location,
                            isMember = individual.IsInSubscription,
                            ExtraMember = extraMembers.ToList(),
                            isExists = true
                        });

                        extraMembers.Clear();
                    }
                }
                else
                {
                    listModel.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
                }
            }

            if (listModel.Messages.Count > 0)
            {
                return this.PartialView("ViewMessageList", listModel.Messages);
            }

            return this.PartialView("_ListExtraIndividuals", individuals);
        }

        /// <summary>
        /// Searches for the Current Extra Organisation
        /// </summary>
        /// <returns>A view for displaying the results of the action</returns>   
        [HttpPost]
        public ActionResult SearchCurrentOrganisation(string code)
        {
            // Initializations
            var listModel = new ListAreasOfInterestViewModel();
            listModel.Messages = new ViewMessageList();

            var areasOfInterest = this.subscriptionRepository.ListSubscriptionDefinitions(this.GetLocalUser()).Data.Find(i => i.Code == code);

            var entities = new List<Entity>();
            entities.Add(Entity.Individual);
            entities.Add(Entity.Link);
            entities.Add(Entity.Location);
            entities.Add(Entity.IndividualClassification);
            entities.Add(Entity.Organisation);

            var organisations = new List<ExtraOrganisationViewModel>();
            var extraMembers = new List<ProviderIdentifierSearchViewModel>();

            foreach (var extraMember in areasOfInterest.ExtraMembers.Where(i => i.Value.Contains("urn:x-nhsd.au:org")))
            {
                var identifierParam = new ProviderIdentifierSearch()
                {
                    TypeCode = extraMember.TypeCode,
                    Value = extraMember.Value
                };
                var organisationParam = new ProviderOrganisationSearch()
                {
                    Identifier = identifierParam
                };

                var response = this.providerIndividualDirectoryRepository.Search(null, organisationParam, null, null, entities, null, null, this.GetLocalUser());

                if (response.IsSuccessful)
                {
                    foreach (var organisation in response.Data.Organisations)
                    {
                        var location = string.Empty;
                        var classification = string.Empty;

                        foreach (var loc in response.Data.Locations.Where(i => i.OrganisationId == organisation.OrganisationId))
                        {
                            location = (string.IsNullOrEmpty(location) ? loc.LocationName : location + ", " + loc.LocationName);
                            foreach (var classifications in loc.Classifications)
                            {
                                classification = (string.IsNullOrEmpty(classification) ? classifications.DisplayName : classification + ", " + classifications.DisplayName);
                            }
                        }

                        foreach (var identifier in organisation.Identifiers.Where(i => i.Type.Code.Contains("NHSD")))
                        {
                            extraMembers.Add(new ProviderIdentifierSearchViewModel()
                            {
                                TypeCode = identifier.Type.Code,
                                Value = identifier.Value
                            });
                        }

                        organisations.Add(new ExtraOrganisationViewModel()
                        {
                            OrganisationName = organisation.LegalName,
                            LocationName = location,
                            Classification = classification,
                            isMember = organisation.IsInSubscription,
                            ExtraMember = extraMembers.ToList(),
                            isExists = true
                        });

                        extraMembers.Clear();
                    }
                }
                else
                {
                    listModel.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
                }
            }

            if (listModel.Messages.Count > 0)
            {
                return this.PartialView("ViewMessageList", listModel.Messages);
            }

            return this.PartialView("_ListExtraOrganisations", organisations);
        }

        /// <summary>
        /// Validate the SearchProviderIndividualRequest
        /// </summary>
        /// <returns>A view message lists for the search individual request</returns>   
        private ViewMessageList ValidateSearchIndividualRequest(ExtraIndividualSearchViewModel model)
        {
            var messages = new ViewMessageList();

            // Page Validations
            if (string.IsNullOrEmpty(model.FamilyName) && string.IsNullOrEmpty(model.GivenName) && string.IsNullOrEmpty(model.Occupation)
                && (string.IsNullOrEmpty(model.Suburb) && string.IsNullOrEmpty(model.State) && string.IsNullOrEmpty(model.Postcode)))
            {
                messages.Add("At least one of first name, last name, address or occupation must be provided.", MessageLevel.Warning);
                return messages;
            }

            if (!string.IsNullOrEmpty(model.Suburb) || !string.IsNullOrEmpty(model.State) || !string.IsNullOrEmpty(model.Postcode))
            {
                if (string.IsNullOrEmpty(model.Suburb))
                {
                    messages.Add("Suburb must be provided.", MessageLevel.Warning);
                }
                if (string.IsNullOrEmpty(model.State))
                {
                    messages.Add("State must be provided.", MessageLevel.Warning);
                }
                if (string.IsNullOrEmpty(model.Postcode))
                {
                    messages.Add("Postcode must be provided.", MessageLevel.Warning);
                }
                if (messages.Count > 0)
                {
                    return messages;
                }
            }

            if ((model.FamilyName != null && model.FamilyName.Length < 3) || (model.GivenName != null && model.GivenName.Length < 3))
            {
                messages.Add("Family name and given name have a minimum length of 3.", MessageLevel.Warning);
                return messages;
            }

            int familyNameWildCards = (model.FamilyName != null) ? model.FamilyName.Split('*').Count() : 0;
            int givenNameWildCards = (model.GivenName != null) ? model.GivenName.Split('*').Count() : 0;

            if (familyNameWildCards > 2 || givenNameWildCards > 2)
            {
                messages.Add("Family name and given name may contain no more than one wildcard character (asterisk).", MessageLevel.Warning);
                return messages;
            }

            if (model.Postcode != null && model.Postcode.Length != 4)
            {
                messages.Add("Postcode is a 4 digit number.", MessageLevel.Warning);
                return messages;
            }

            return messages;
        }

        /// <summary>
        /// Validate the SearchProviderOrganisationRequest
        /// </summary>
        /// <returns>A view message lists for the search organisation request</returns>   
        private ViewMessageList ValidateSearchOrganisationRequest(ExtraOrganisationSearchViewModel model)
        {
            var messages = new ViewMessageList();

            // Page Validations
            if (string.IsNullOrEmpty(model.OrganisationName) && string.IsNullOrEmpty(model.Classification)
                && (string.IsNullOrEmpty(model.OrgSuburb) && string.IsNullOrEmpty(model.OrgState) && string.IsNullOrEmpty(model.OrgPostcode)))
            {
                messages.Add("At least one of organisation name, address or classification must be provided.", MessageLevel.Warning);
                return messages;
            }

            if (!string.IsNullOrEmpty(model.OrgSuburb) || !string.IsNullOrEmpty(model.OrgState) || !string.IsNullOrEmpty(model.OrgPostcode))
            {
                if (string.IsNullOrEmpty(model.OrgSuburb))
                {
                    messages.Add("Suburb must be provided.", MessageLevel.Warning);
                }
                if (string.IsNullOrEmpty(model.OrgState))
                {
                    messages.Add("State must be provided.", MessageLevel.Warning);
                }
                if (string.IsNullOrEmpty(model.OrgPostcode))
                {
                    messages.Add("Postcode must be provided.", MessageLevel.Warning);
                }
                if (messages.Count > 0)
                {
                    return messages;
                }
            }

            if ((model.OrganisationName != null && model.OrganisationName.Length < 3))
            {
                messages.Add("Organisation name has a minimum length of 3.", MessageLevel.Warning);
                return messages;
            }

            int organisationNameWildcards = (model.OrganisationName != null) ? model.OrganisationName.Split('*').Count() : 0;

            if (organisationNameWildcards > 2)
            {
                messages.Add("Organisation name may contain no more than one wildcard character (asterisk).", MessageLevel.Warning);
                return messages;
            }

            if (model.OrgPostcode != null && model.OrgPostcode.Length != 4)
            {
                messages.Add("Postcode is a 4 digit number.", MessageLevel.Warning);
                return messages;
            }

            return messages;
        }

        /// <summary>
        /// Validate the add or edit request
        /// </summary>
        /// <returns>A view for displaying Areas of Interest</returns>   
        private ViewMessageList ValidateModel(AreaOfInterestViewModel model)
        {
            var messages = new ViewMessageList();
            var areaOfInterest = ObjectMapper.Map<AreaOfInterest>(model);

            // Page Validations
            if (string.IsNullOrEmpty(areaOfInterest.Description))
            {
                messages.Add("Name cannot be empty.", MessageLevel.Warning);
            }

            if (areaOfInterest.Areas.Count == 0)
            {
                messages.Add("At least one geographical area is required.", MessageLevel.Warning);
            }

            if (areaOfInterest.Occupations.Count == 0 && areaOfInterest.Classifications.Count == 0)
            {
                messages.Add("At least one occupation or classification is required.", MessageLevel.Warning);
            }

            return messages;
        }

        /// <summary>
        /// Populate the Area Of Interest View Model that will be used in the Modal Pop-up
        /// </summary>
        /// <returns>A view model for displaying Areas of Interest modal</returns> 
        private AreaOfInterestViewModel PopulateModal()
        {
            var statesList = this.referenceRepository.Search(stateUri, asterisk, asterisk, this.GetLocalUser());
            var providerIndividualTypeList = this.referenceRepository.Search(providerIndividualTypeUri, asterisk, asterisk, this.GetLocalUser());
            var providerIndividualSpecialtyList = this.referenceRepository.Search(providerIndividualSpecialtyUri, asterisk, asterisk, this.GetLocalUser());
            var providerIndividualSpecialisationList = this.referenceRepository.Search(providerIndividualSpecialisationUri, asterisk, asterisk, this.GetLocalUser());
            var providerOrganisationTypeList = this.referenceRepository.Search(providerOrganisationTypeUri, asterisk, asterisk, this.GetLocalUser());
            var providerOrganisationServiceTypeList = this.referenceRepository.Search(providerOrganisationServiceTypeUri, asterisk, asterisk, this.GetLocalUser());
            var providerOrganisationServiceUnitList = this.referenceRepository.Search(providerOrganisationServiceUnitUri, asterisk, asterisk, this.GetLocalUser());

            var model = new AreaOfInterestViewModel();
            model.Occupations = new List<IndividualOccupationViewModel>();
            model.Classifications = new List<OrganisationClassificationViewModel>();
            model.Individuals = new ExtraIndividualSearchViewModel();
            model.Organisations = new ExtraOrganisationSearchViewModel();
            var occupationContainer = new List<SelectListItem>();
            var classificationContainer = new List<SelectListItem>();

            // Popupate Individual Occupation Grid
            if (providerIndividualTypeList.IsSuccessful)
            {
                foreach (var type in providerIndividualTypeList.Data.OrderBy(m => m.DisplayName))
                {
                    model.Occupations.Add(new IndividualOccupationViewModel()
                    {
                        Set = "Type",
                        Code = type.Code,
                        Description = type.DisplayName,
                        isExists = false
                    });

                    // This populate the Occupation Dropdown in the Extra Individuals Tab
                    occupationContainer.Add(new SelectListItem()
                    {
                        Text = type.DisplayName,
                        Value = type.Uri
                    });

                    var specialtyOfThisType = providerIndividualSpecialisationList.Data.Where(i => i.Code.StartsWith(type.Code)).OrderBy(i => i.DisplayName);
                    foreach (var specialty in specialtyOfThisType)
                    {
                        occupationContainer.Add(new SelectListItem()
                        {
                            Text = "s-" + specialty.DisplayName,
                            Value = specialty.Uri
                        });

                        var specialisationOfThisSpecialty = providerIndividualSpecialisationList.Data.Where(i => i.Code.StartsWith(specialty.Code)).OrderBy(i => i.DisplayName);
                        foreach (var specialisation in specialisationOfThisSpecialty)
                        {
                            occupationContainer.Add(new SelectListItem()
                            {
                                Text = "sl-" + specialisation.DisplayName,
                                Value = specialisation.Uri
                            });
                        }
                    }
                }
            }

            if (providerIndividualSpecialtyList.IsSuccessful)
            {
                foreach (var specialty in providerIndividualSpecialtyList.Data)
                {
                    model.Occupations.Add(new IndividualOccupationViewModel()
                    {
                        Set = "Specialty",
                        Code = specialty.Code,
                        Description = specialty.DisplayName,
                        isExists = false
                    });
                }
            }

            if (providerIndividualSpecialisationList.IsSuccessful)
            {
                foreach (var specialisation in providerIndividualSpecialisationList.Data)
                {
                    model.Occupations.Add(new IndividualOccupationViewModel()
                    {
                        Set = "Specialisation",
                        Code = specialisation.Code,
                        Description = specialisation.DisplayName,
                        isExists = false
                    });
                }
            }

            // Assign Values to Extra Individuals Occupation Dropdownlist
            model.Individuals.OccupationList = occupationContainer;

            // Populate Organisation Classifications
            if (providerOrganisationTypeList.IsSuccessful)
            {
                foreach (var organisationType in providerOrganisationTypeList.Data.OrderBy(m => m.DisplayName))
                {
                    model.Classifications.Add(new OrganisationClassificationViewModel()
                    {
                        Set = "Type",
                        Code = organisationType.Code,
                        Description = organisationType.DisplayName,
                        isExists = false
                    });

                    // This populate the Classification Dropdown in the Extra Organisations Tab
                    classificationContainer.Add(new SelectListItem()
                    {
                        Text = organisationType.DisplayName,
                        Value = organisationType.Uri
                    });

                    var serviceOfThisType = providerOrganisationServiceTypeList.Data.Where(i => i.Code.StartsWith(organisationType.Code)).OrderBy(i => i.DisplayName);
                    foreach (var service in serviceOfThisType)
                    {
                        classificationContainer.Add(new SelectListItem()
                        {
                            Text = "s-" + service.DisplayName,
                            Value = service.Uri
                        });

                        var unitOfThisService = providerOrganisationServiceUnitList.Data.Where(i => i.Code.StartsWith(service.Code)).OrderBy(i => i.DisplayName);
                        foreach (var unit in unitOfThisService)
                        {
                            classificationContainer.Add(new SelectListItem()
                            {
                                Text = "u-" + unit.DisplayName,
                                Value = unit.Uri
                            });
                        }
                    }
                }
            }

            // Assign Values to Extra Organisations Classificaiton Dropdownlist
            model.Organisations.ClassificationList = classificationContainer;

            if (providerOrganisationServiceTypeList.IsSuccessful)
            {
                foreach (var serviceType in providerOrganisationServiceTypeList.Data)
                {
                    model.Classifications.Add(new OrganisationClassificationViewModel()
                    {
                        Set = "Service",
                        Code = serviceType.Code,
                        Description = serviceType.DisplayName,
                        isExists = false
                    });
                }
            }

            if (providerOrganisationServiceUnitList.IsSuccessful)
            {
                foreach (var serviceUnit in providerOrganisationServiceUnitList.Data)
                {
                    model.Classifications.Add(new OrganisationClassificationViewModel()
                    {
                        Set = "Unit",
                        Code = serviceUnit.Code,
                        Description = serviceUnit.DisplayName,
                        isExists = false
                    });
                }
            }

            if (statesList.IsSuccessful)
            {
                model.Individuals.StateList = ObjectMapper.Map<IEnumerable<SelectListItem>>(statesList.Data);
                model.Organisations.OrgStateList = ObjectMapper.Map<IEnumerable<SelectListItem>>(statesList.Data);
            }

            model.Individuals.ProximityRadiusList = this.subscriptionReferenceRepository.GetProximityRadius().ToSelectListItems(c => c.Code, c => c.Description);
            model.Organisations.OrgProximityRadiusList = this.subscriptionReferenceRepository.GetProximityRadius().ToSelectListItems(c => c.Code, c => c.Description);

            // Populate the Filter Dropdown
            List<SelectListItem> filter = new List<SelectListItem>();
            filter.Add(new SelectListItem()
            {
                Text = "Included",
                Selected = true
            });
            filter.Add(new SelectListItem()
            {
                Text = "All"
            });

            model.FilterList = filter.ToList();

            return model;
        }

        #endregion
    }
}
