using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;

using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.Web;
using HIPS.Web.Model.AssistedRegistration;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.AssistedRegistration;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.UI.Conversion.AssistedRegistration;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.AssistedRegistration;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// <see cref="Controller"/> for the PCEHR Assisted Registration functionality.
    /// </summary>
    [HpoRequired]
    public class AssistedRegistrationController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// Hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Assisted Registration Reference repository to be used by this controller.
        /// </summary>
        private readonly IAssistedRegistrationReferenceRepository assistedRegistrationReferenceRepository;

        /// <summary>
        /// Patients without Pcehr repository to be used by this controller.
        /// </summary>
        private readonly IPatientRepository patientsRepository;

        /// <summary>
        /// Assisted Registration service to be used by this controller.
        /// </summary>
        private readonly IAssistedRegistrationService assistedRegistrationService;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="AssistedRegistrationController" /> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="assistedRegistrationReferenceRepository">Assisted Registration Reference repository to be used by this controller.</param>
        /// <param name="patientRepository">Patients without Pcehr repository to be used by this controller.</param>
        /// <param name="registrationService">Assisted Registration service to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public AssistedRegistrationController(
            IHospitalRepository hospitalRepository,
            IAssistedRegistrationReferenceRepository assistedRegistrationReferenceRepository,
            IPatientRepository patientRepository,
            IAssistedRegistrationService registrationService,
            ISettingsRepository settingsRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
            this.assistedRegistrationReferenceRepository = assistedRegistrationReferenceRepository;
            this.patientsRepository = patientRepository;
            this.assistedRegistrationService = registrationService;
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Displays a list of unregistered patients in the selected hospital.
        /// </summary>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Unregistered()
        {
            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            var model = new UnregisteredViewModel();

            // Only load patients after a hospital has been selected.
            if (!string.IsNullOrEmpty(hospitalId))
            {
                // Load patients without PCEHR
                var response = this.patientsRepository.ListPatientsCurrentlyInHospital(this.DefaultHospitalCodeSystem, hospitalId, false, this.GetCurrentUserDetails());

                if (response.IsSuccessful)
                {
                    if (response.Data.PatientInHospitalList.Count > 0)
                    {
                        model.UnregisteredPatients.AddRange(ObjectMapper.Map<IEnumerable<UnregisteredPatientViewModel>>(response.Data.PatientInHospitalList));
                    }
                    else
                    {
                        model.Messages.Add("There are no patients at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";

                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));

                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    model.Messages.Add(errorMessage, MessageLevel.Error);
                }
            }

            return this.View(model);
        }

        /// <summary>
        /// Display a form for registering an Adult patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <returns>The view Result.</returns>
        [HttpGet]
        public ActionResult Register(string hospitalId, string patientId)
        {
            return this.Register(hospitalId, patientId, false);
        }

        /// <summary>
        /// Display a form for registering a child patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <returns>The view Result.</returns>
        [HttpGet]
        public ActionResult RegisterDependant(string hospitalId, string patientId)
        {
            return this.Register(hospitalId, patientId, true);
        }

        /// <summary>
        /// Display an embedded form for registering an Adult patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient (MRN).</param>
        /// <param name="statePatientId">Identifier of the patient (State Patient Id).</param>
        /// <returns>The view Result.</returns>
        [HttpGet]
        public ActionResult EmbeddedEnterpriseRegister(string hospitalId, string patientId, string statePatientId)
        {
            return this.EmbeddedEnterpriseRegister(hospitalId, patientId, statePatientId, false);
        }

        /// <summary>
        /// Display a form for registering a child patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient (MRN).</param>
        /// <param name="statePatientId">Identifier of the patient (State Patient Id).</param>
        /// <returns>The view Result.</returns>
        [HttpGet]
        public ActionResult EmbeddedEnterpriseRegisterDependant(string hospitalId, string patientId, string statePatientId)
        {
            return this.EmbeddedEnterpriseRegister(hospitalId, patientId, statePatientId, true);
        }

        /// <summary>
        /// Process the AJAX request to register a adult patient.
        /// </summary>
        /// <param name="registration">Model that contains data to be processed.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        public ActionResult Register(RegisterViewModel registration)
        {
            return this.Register(registration, false);
        }

        /// <summary>
        /// Process the AJAX request to register a child patient.
        /// </summary>
        /// <param name="registration">Model that contains data to be processed.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        public ActionResult RegisterDependant(RegisterViewModel registration)
        {
            return this.Register(registration, true);
        }

        /// <summary>
        /// Display a form for registering a adult/child patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="asDependant">A value indicating whether the if patient is dependant.</param>
        /// <returns>The View Result.</returns>
        private ActionResult Register(string hospitalId, string patientId, bool asDependant)
        {
            var model = new RegisterViewModel();

            if (string.IsNullOrEmpty(hospitalId))
            {
                model.HospitalId = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            }

            // If patient indicated attempt to load
            if (!string.IsNullOrWhiteSpace(patientId))
            {
                var patientIdentifier = new Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem);
                var patient = this.patientsRepository.GetPatientInHospital(patientIdentifier, false, this.GetCurrentUserDetails());

                // Check if the patient is in hospital, with an IHI and no PCEHR
                if (patient != null)
                {
                    model.Applicant = ObjectMapper.Map<PersonDemographicViewModel>(patient);
                }
                else
                {
                    model.HasPcehrOrInvalidPatient = true;

                    // Get the patient record ignoring IHI and PCEHR status
                    var checkPatient = this.patientsRepository.GetAdmittedPatient(patientIdentifier, this.GetCurrentUserDetails());

                    // Check if the patient exists at this hospital
                    if (checkPatient == null)
                    {
                        model.Messages.Add(string.Format("No patient {0} was found.", patientId), MessageLevel.Information);
                    }
                    else
                    {
                        // Check if the patient has an IHI
                        if (checkPatient.IhiStatus == PcehrDataStore.Schemas.Enumerators.IhiStatus.Unknown)
                        {
                            model.Messages.Add(string.Format("No IHI was found for patient {0}.", patientId), MessageLevel.Information);
                        }
                        // Check if the patient has a PCEHR
                        else if (checkPatient.PcehrDisclosed.Value)
                        {
                            model.Messages.Add(string.Format("Patient {0} has a digital health record.", patientId), MessageLevel.Information);
                        }
                        else
                        {
                            model.Messages.Add(string.Format("Could not load patient {0}.", patientId), MessageLevel.Information);
                        }
                    }
                }
            }

            this.LoadReferences(model);
            if (asDependant)
            {
                model.Applicant.ShowDvaFileNumber = false;
                return this.View("RegisterDependant", model);
            }

            return this.View("Register", model);
        }

        /// <summary>
        /// Display a form for registering a adult/child patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient (MRN).</param>
        /// <param name="statePatientId">Identifier of the patient (State Patient Id).</param>
        /// <param name="asDependant">A value indicating whether the if patient is dependant.</param>
        /// <returns>The View Result.</returns>
        private ActionResult EmbeddedEnterpriseRegister(string hospitalId, string patientId, string statePatientId, bool asDependant)
        {
            var model = new RegisterViewModel();

            if (string.IsNullOrWhiteSpace(statePatientId))
            {
                statePatientId = patientId;
            }

            // If patient indicated attempt to load
            if (!string.IsNullOrWhiteSpace(patientId))
            {
                var patientIdentifier = new RegisteredEnterprisePatient(patientId, statePatientId, hospitalId, this.DefaultHospitalCodeSystem);

                // Get all patients without PCEHR
                var patientsResponse = this.patientsRepository.ListPatientsCurrentlyInHospital(this.DefaultHospitalCodeSystem, hospitalId, false, this.GetCurrentUserDetails());

                // Get patient details
                var patientResponse = this.patientsRepository.GetPatientDisclosureDetails(this.GetCurrentUserDetails(), patientIdentifier);

                if (!patientResponse.IsSuccessful)
                {
                    model.Messages.Add(string.Format("No patient {0} was found.", patientId), MessageLevel.Information);
                    model.HasPcehrOrInvalidPatient = true;
                }
                else
                {
                    if (patientResponse.Data.AdmittedPatient.IhiStatus == PcehrDataStore.Schemas.Enumerators.IhiStatus.Unknown)
                    {
                        model.Messages.Add(string.Format("No IHI was found for patient {0}.", patientId), MessageLevel.Information);
                        model.HasPcehrOrInvalidPatient = true;
                    }
                    else if (patientResponse.Data.AdmittedPatient.PcehrDisclosed.HasValue && patientResponse.Data.AdmittedPatient.PcehrDisclosed.Value)
                    {
                        model.Messages.Add(string.Format("Patient {0} has a digital health record.", patientId), MessageLevel.Information);
                        model.HasPcehrOrInvalidPatient = true;
                    }
                    else
                    {
                        // Get patient from List of patients without PCEHR
                        var patient = patientsResponse.Data.PatientInHospitalList.SingleOrDefault(p => p.Mrn == patientResponse.Data.AdmittedPatient.Mrn);
                        if (patient != null)
                        {
                            model.Applicant = ObjectMapper.Map<PersonDemographicViewModel>(patient);
                        }
                        else
                        {
                            model.Messages.Add(string.Format("Could not load patient {0}.", patientId), MessageLevel.Information);
                            model.HasPcehrOrInvalidPatient = true;
                        }
                    }
                }
            }

            this.LoadReferences(model);
            if (asDependant)
            {
                model.Applicant.ShowDvaFileNumber = false;
                return this.View("RegisterDependant", model);
            }

            return this.View("Register", model);
        }

        /// <summary>
        /// Process the AJAX request to register a adult/child patient.
        /// </summary>
        /// <param name="registration">Model that contains data to be processed.</param>
        /// <param name="asDependant">A value indicating whether the if patient is dependant.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        private ActionResult Register(RegisterViewModel registration, bool asDependant)
        {
            // As returning JSON ensure the request was AJAX.
            // TODO: Support non-ajax requests to this method (currently not enabled by client).
            // NB: Shouldn't be hard, priorities currently deferred (MM)
            if (!this.Request.IsAjaxRequest())
            {
                return this.Content("JavaScript is required for this functionality.");
            }

            // Check binding validation errors
            if (!this.ModelState.IsValid)
            {
                return this.Json(new { Echo = registration, Errors = ModelState.ToErrorDictionary() });
            }

            // Load Data
            var sexes = this.assistedRegistrationReferenceRepository.GetSexes();
            var indigenousStatuses = this.assistedRegistrationReferenceRepository.GetIndigenousStatuses();
            var identityVerificationMethods = this.assistedRegistrationReferenceRepository.GetIdentityVerificationMethods();
            var medicareConsents = this.assistedRegistrationReferenceRepository.GetMedicareConsents();
            var ivcDeliveryMethods = this.assistedRegistrationReferenceRepository.GetIvcDeliveryMethods();

            // If identifying a pre-validated applicant
            if (!string.IsNullOrWhiteSpace(registration.Applicant.Mrn))
            {
                var mrn = new Mrn(registration.Applicant.Mrn, registration.HospitalId, this.DefaultHospitalCodeSystem);
                var patient = this.patientsRepository.GetPatientInHospital(mrn, false, this.GetCurrentUserDetails());

                // Ensure patient found
                if (patient == null)
                {
                    ModelState.AddModelError(string.Empty, "Could not find the pre-validated patient record.");
                    return this.Json(new { Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }

                var validatedPatient = this.CreateValidatedPatient(patient, sexes, registration.HospitalId);

                if (asDependant)
                {
                    var response = this.assistedRegistrationService.RegisterDependant(
                        validatedPatient,
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.Representative.ToPersonDemographicModel(sexes),
                        registration.ConsentDeclared,
                        registration.MedicareConsents.GetMedicareConsents(medicareConsents),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return this.Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }
                else
                {
                    var response = this.assistedRegistrationService.RegisterApplicant(
                        validatedPatient,
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.MedicareConsents.GetMedicareConsents(medicareConsents),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return this.Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }
            }
            else
            {
                // A "visitor" applicant
                // NB: Configuration point for "Visitor" functionality?
                var hospitals = this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem);
                var hospital = hospitals.Select(h => new Hospital(h.GetCode(this.DefaultHospitalCodeSystem), this.DefaultHospitalCodeSystem, h.Name, h.HpiO, h.HpioName))
                                        .FirstOrDefault(h => h.HospitalFacilityCode == registration.HospitalId);

                if (asDependant)
                {
                    var response = this.assistedRegistrationService.RegisterDependant(
                        registration.Applicant.ToPersonDemographicModel(sexes),
                        hospital,
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.Representative.ToPersonDemographicModel(sexes),
                        registration.ConsentDeclared,
                        registration.MedicareConsents.GetMedicareConsents(medicareConsents),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return this.Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }
                else
                {
                    var response = this.assistedRegistrationService.RegisterApplicant(
                        registration.Applicant.ToPersonDemographicModel(sexes),
                        hospital,
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.MedicareConsents.GetMedicareConsents(medicareConsents),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return this.Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }
            }
        }

        #region Helpers

        /// <summary>
        /// Load references
        /// </summary>
        /// <param name="model">View model to load references into.</param>
        private void LoadReferences(RegisterViewModel model)
        {
            var sexes = this.assistedRegistrationReferenceRepository.GetSexes();
            var indigenousStatuses = this.assistedRegistrationReferenceRepository.GetIndigenousStatuses();
            var identityVerificationMethods = this.assistedRegistrationReferenceRepository.GetIdentityVerificationMethods();
            var medicareConsents = this.assistedRegistrationReferenceRepository.GetMedicareConsents();

            model.Applicant.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            model.Applicant.IndigenousStatuses = indigenousStatuses.ToSelectListItems(i => i.Code, i => i.Description);

            model.Representative.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            model.Representative.IndigenousStatuses = indigenousStatuses.ToSelectListItems(i => i.Code, i => i.Description);

            model.IdentityVerificationMethods = identityVerificationMethods.ToSelectListItems(i => i.Code, i => i.Description);
            model.MedicareConsents.MedicareConsentOptions = medicareConsents.ToNestedSelectList();
        }

        /// <summary>
        /// Creates a validated patient.
        /// </summary>
        /// <param name="patient">A Patient.</param>
        /// <param name="sexes">List of Sexes.</param>
        /// <param name="hospitalCode">Hospital Code.</param>
        /// <returns>A Validated Patient.</returns>
        private ValidatedPatient CreateValidatedPatient(HIPS.PatientSchemas.PatientInHospital patient, List<Sex> sexes, string hospitalCode)
        {
            var sex = sexes.FirstOrDefault(s => s.SexId == (int)patient.RegisteredSex);

            var personDemographic = new PersonDemographic(patient.FamilyName, patient.GivenNames, patient.DateOfBirth, sex, patient.MedicareNumber, patient.MedicareIrn, patient.DvaNumber);
            var verifiedIhi = new VerifiedIhi(patient.Ihi, (int)patient.IhiStatus, (int)patient.IhiRecordStatus, patient.IhiLastValidated.Value);
            var hospitalLocation = new HospitalLocation(hospitalCode, this.DefaultHospitalCodeSystem, patient.Ward, patient.Room, patient.Bed);

            return new ValidatedPatient(personDemographic, verifiedIhi, hospitalLocation);
        }

        #endregion Helpers

        #endregion Methods
    }
}