﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web;

// Simon Tewsi shows a table here, which is a useful reference to understand various .NET encoding schemes
// http://stackoverflow.com/questions/575440/url-encoding-using-c-sharp

namespace HIPS.Web.UI.Conversion.Custom
{
    #region Interface

    /// <summary>
    /// Base encoder interface
    /// </summary>
    internal interface IEncoder
    {
        /// <summary>
        /// Encodes the string.
        /// </summary>        
        /// <returns>Returns the encoded string.</returns>
        /// <param name="s">String to encode.</param>
        string Encode(string s);

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        string Decode(string s);
    }

    #endregion

    #region JtmlSafeId

    /// <summary>
    /// Utility class for custom encoding string so that they are safe to use for JQuery and HTML
    /// when directly using the string as the value for an HTML element's ID attribute
    /// </summary>
    public static class JtmlSafeId
    {
        #region Properties

        /// <summary>
        /// Private static field to hold the token collection
        /// </summary>
        private static IEncoder encoder = Factory.CreateObject();

        #endregion

        #region Methods

        /// <summary>
        /// Encodes the string.
        /// </summary>
        /// <param name="s">String to encode.</param>
        /// <returns>Returns the encoded string.</returns>
        public static string Encode(string s)
        {
            return encoder.Encode(s);
        }

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        public static string Decode(string s)
        {
            return encoder.Decode(s);
        }

        #endregion
    }

    #endregion

    #region Factory

    /// <summary>
    /// Factory object
    /// </summary>
    internal static class Factory
    {
        /// <summary>
        /// Encoding scheme that will be used
        /// </summary>
        private static EncodingType type = EncodingType.ModifiedBase64;

        /// <summary>
        /// Encoding type
        /// </summary>
        private enum EncodingType
        {
            /// <summary>
            /// Base64 encoding type
            /// </summary>
            Base64,

            /// <summary>
            /// ModifiedBase64 encoding type
            /// </summary>
            ModifiedBase64,

            /// <summary>
            /// WebUtility encoding type
            /// </summary>
            WebUtility,

            /// <summary>
            /// HttpUtility encoding type
            /// </summary>
            HttpUtility,

            /// <summary>
            /// Uri encoding type
            /// </summary>
            Uri,

            /// <summary>
            /// Simple encoding type
            /// </summary>
            Simple
        }

        /// <summary>
        /// Factory creation method
        /// </summary>
        /// <returns>Returns an instance of the IEncoder interface</returns>
        public static IEncoder CreateObject()
        {
            switch (type)
            {
                case EncodingType.Base64:
                    return new Base64Encoder();
                case EncodingType.WebUtility:
                    return new WebUtilityEncoder();
                case EncodingType.HttpUtility:
                    return new HttpUtilityEncoder();
                case EncodingType.Uri:
                    return new UriEncoder();
                case EncodingType.ModifiedBase64:
                    return new ModifiedBase64Encoder();
                default:
                    return new SimpleCustomEncoder();
            }
        }
    }

    #endregion

    #region Built-in .NET Encoders

    #region Base64Encoder

    /// <summary>
    /// Base64 implementation
    /// </summary>
    internal class Base64Encoder : IEncoder
    {
        /// <summary>
        /// Encodes the string.
        /// </summary>        
        /// <returns>Returns the encoded string.</returns>
        /// <param name="s">String to encode.</param>
        public string Encode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                var bytes = System.Text.Encoding.UTF8.GetBytes(s);
                return System.Convert.ToBase64String(bytes);
            }
        }

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        public string Decode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                var bytes = System.Convert.FromBase64String(s);
                return System.Text.Encoding.UTF8.GetString(bytes);
            }
        }
    }

    #endregion

    #region WebUtilityEncoder

    /// <summary>
    /// WebUtility implementation
    /// </summary>
    internal class WebUtilityEncoder : IEncoder
    {
        /// <summary>
        /// Encodes the string.
        /// </summary>        
        /// <returns>Returns the encoded string.</returns>
        /// <param name="s">String to encode.</param>
        public string Encode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                return WebUtility.HtmlEncode(s);
            }
        }

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        public string Decode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                return WebUtility.HtmlDecode(s);
            }
        }
    }

    #endregion

    #region HttpUtilityEncoder

    /// <summary>
    /// HttpUtility implementation
    /// </summary>
    internal class HttpUtilityEncoder : IEncoder
    {
        /// <summary>
        /// Encodes the string.
        /// </summary>        
        /// <returns>Returns the encoded string.</returns>
        /// <param name="s">String to encode.</param>
        public string Encode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                return HttpUtility.UrlEncode(s);
            }
        }

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        public string Decode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                return HttpUtility.UrlDecode(s);
            }
        }
    }

    #endregion

    #region UriEncoder

    /// <summary>
    /// Uri implementation
    /// </summary>
    internal class UriEncoder : IEncoder
    {
        /// <summary>
        /// Encodes the string.
        /// </summary>        
        /// <returns>Returns the encoded string.</returns>
        /// <param name="s">String to encode.</param>
        public string Encode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                return Uri.EscapeDataString(s);
            }
        }

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        public string Decode(string s)
        {
            if (string.IsNullOrEmpty(s))
            {
                return string.Empty;
            }
            else
            {
                return Uri.UnescapeDataString(s);
            }
        }
    }

    #endregion

    #endregion

    #region Custom Encoders

    #region ModifiedBase64Encoder

    /// <summary>
    /// Modified version of the Base64 implementation.
    /// </summary>
    internal class ModifiedBase64Encoder : IEncoder
    {
        /// <summary>
        /// Instance of the Base64 encoder
        /// </summary>
        private IEncoder base64 = new Base64Encoder();

        /// <summary>
        /// Encodes the string.
        /// </summary>        
        /// <returns>Returns the encoded string.</returns>
        /// <param name="s">String to encode.</param>
        public string Encode(string s)
        {
            return this.base64.Encode(s).Replace("=", "__");
        }

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        public string Decode(string s)
        {
            return this.base64.Decode(s.Replace("__", "="));
        }
    }

    #endregion

    #region SimpleCustomEncoder

    /// <summary>
    /// Internal class that manages the collection of special tokens 
    /// </summary>
    internal class SimpleCustomEncoder : IEncoder
    {
        #region Fields

        /// <summary>
        /// Private field to hold the token collection
        /// </summary>
        private Dictionary<string, string> d = new Dictionary<string, string>();

        #endregion

        #region Constructor

        /// <summary>
        /// Initialises a new instance of the SimpleEncoder class
        /// </summary>
        public SimpleCustomEncoder()
        {
            this.d.Add(@":", "--CO--");
            this.d.Add(@".", "--PE--");
            this.d.Add(@"/", "--FS--");
            this.d.Add(@"\", "--BS--");
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets the Tokens in the tokens collection
        /// </summary>
        private Dictionary<string, string> Tokens
        {
            /// <summary>
            /// Gets the Tokens in the tokens collection
            /// </summary>
            get { return this.d; }
            set { }
        }

        #endregion

        #region Methods

        /// <summary>
        /// Encodes the string.
        /// </summary>        
        /// <returns>Returns the encoded string.</returns>
        /// <param name="s">String to encode.</param>
        public string Encode(string s)
        {
            var enc = s;
            foreach (var item in this.Tokens)
            {
                enc = enc.Replace(item.Key, item.Value);
            }
            return enc;
        }

        /// <summary>
        /// Decodes the string.
        /// </summary>
        /// <param name="s">String to decode.</param>
        /// <returns>Returns the decoded string.</returns>
        public string Decode(string s)
        {
            var dec = s;
            foreach (var item in this.Tokens)
            {
                dec = dec.Replace(item.Value, item.Key);
            }
            return dec;
        }

        #endregion
    }

    #endregion

    #endregion
}