﻿using AutoMapper;
using HIPS.Web.Components.Common;
using HIPS.Web.Model.Messaging;
using HIPS.Web.UI.Areas.Messaging.ViewModels;

namespace HIPS.Web.UI.Helpers.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for messaging view models.
    /// </summary>
    public class MessagingProfile : Profile
    {
        #region Properties

        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>        
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion

        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>        
        protected override void Configure()
        {
            // Data --> ViewModel:
            this.CreateMap<OutboundSealedMessage, ListDeliveryViewModel>()
                .ForMember(dest => dest.InvocationIdentifier, opt => opt.MapFrom(src => src.InvocationIdentifier))
                .ForMember(dest => dest.MessageTimestamp, opt => opt.MapFrom(src => src.MessageTimestamp))
                .ForMember(dest => dest.MessageStatus, opt => opt.MapFrom(src => src.MessageStatus))
                .ForMember(dest => dest.DocumentType, opt => opt.MapFrom(src => src.PayloadScheme.DocumentType))
                .ForMember(dest => dest.Receiver, opt => opt.ResolveUsing<ReceiverResolver>())
                .ForMember(dest => dest.Details, opt => opt.MapFrom(src => Mapper.Map<OutboundSealedMessage, DetailsViewModel>(src)));

            this.CreateMap<InboundSealedMessage, ListReceiptViewModel>()
                .ForMember(dest => dest.InvocationIdentifier, opt => opt.MapFrom(src => src.InvocationIdentifier))
                .ForMember(dest => dest.MessageTimestamp, opt => opt.MapFrom(src => src.MessageTimestamp))
                .ForMember(dest => dest.MessageStatus, opt => opt.MapFrom(src => src.MessageStatus))
                .ForMember(dest => dest.DocumentType, opt => opt.MapFrom(src => src.PayloadScheme.DocumentType))
                .ForMember(dest => dest.Sender, opt => opt.ResolveUsing<SenderResolver>())
                .ForMember(dest => dest.Details, opt => opt.MapFrom(src => Mapper.Map<InboundSealedMessage, DetailsViewModel>(src)));

            Mapper.CreateMap<TransportResponse, TransportResponseViewModel>()
                  .ForMember(dest => dest.ResponseTime, opt => opt.MapFrom(src => src.ResponseTime))
                  .ForMember(dest => dest.ResponseClass, opt => opt.MapFrom(src => src.ResponseClass))
                  .ForMember(dest => dest.ResponseStatus, opt => opt.MapFrom(src => src.ResponseStatus))
                  .ForMember(dest => dest.ResponseMessage, opt => opt.MapFrom(src => src.Message.Replace("/", "/\u200b")))
                  .ForMember(dest => dest.Final, opt => opt.MapFrom(src => src.Final));

            Mapper.CreateMap<SealedMessage, DetailsViewModel>()
                  .ForMember(dest => dest.CreationTime, opt => opt.MapFrom(src => src.MessageTimestamp))
                  .ForMember(dest => dest.PayloadScheme, opt => opt.MapFrom(src => src.PayloadScheme.Name))
                  .ForMember(dest => dest.SenderOrganisation, opt => opt.MapFrom(src => src.Sender.Organisation))
                  .ForMember(dest => dest.SenderIndividual, opt => opt.MapFrom(src => src.Sender.Individual))
                  .ForMember(dest => dest.ReceiverOrganisation, opt => opt.MapFrom(src => src.Receiver.Organisation))
                  .ForMember(dest => dest.ReceiverIndividual, opt => opt.MapFrom(src => src.Receiver.Individual))
                  .ForMember(dest => dest.SubjectOfCare, opt => opt.MapFrom(src => src.SubjectOfCare))
                  .ForMember(dest => dest.TransportResponses, opt => opt.MapFrom(src => src.TransportResponses));

            Mapper.CreateMap<ProviderOrganisationIdentity, string>()
                  .ConvertUsing<ProviderOrganisationIdentityConverter>();

            Mapper.CreateMap<ProviderIndividualIdentity, string>()
                  .ConvertUsing<ProviderIndividualIdentityConverter>();

            Mapper.CreateMap<ConsumerIdentity, string>()
                  .ConvertUsing<ConsumerIdentityConverter>();
        }

        /// <summary>
        /// Sets and format the organisation name (sender/receiver) to be mapped.
        /// </summary>
        /// <param name="organisationName">Organisation Name to be set.</param>
        /// <param name="identifier">The identifier (HPI-O) to be set.</param>
        /// <returns>Formatted Organisation Name</returns>
        private static string SetOrganisationName(string organisationName, ProviderIdentifier identifier)
        {
            string value = identifier.Value;

            if (string.IsNullOrEmpty(organisationName))
            {
                return value;
            }

            if (!string.IsNullOrEmpty(value) && identifier.Type.Code == "HPIO")
            {
                value = identifier.Value.ToHpioFormat();
            }

            return string.Format("{0} ({1})", organisationName, value);
        }

        #endregion

        #region Resolvers / Converters

        /// <summary>
        /// Resolves the value of the Receiver based on a provided OutboundSealedMessage.
        /// </summary>
        internal class ReceiverResolver : ValueResolver<OutboundSealedMessage, string>
        {
            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">Source to be used when resolving for the correct value.</param>
            /// <returns>Resolved value</returns>
            protected override string ResolveCore(OutboundSealedMessage source)
            {
                return MessagingProfile.SetOrganisationName(source.Receiver.Organisation.OrganisationName, source.Receiver.Organisation.Identifier);
            }
        }

        /// <summary>
        /// Resolves the value of the Sender based on a provided InboundSealedMessage.
        /// </summary>
        internal class SenderResolver : ValueResolver<InboundSealedMessage, string>
        {
            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">Source to be used when resolving for the correct value.</param>
            /// <returns>Resolved value</returns>
            protected override string ResolveCore(InboundSealedMessage source)
            {
                return MessagingProfile.SetOrganisationName(source.Sender.Organisation.OrganisationName, source.Sender.Organisation.Identifier);
            }
        }

        /// <summary>
        /// Converts the value of the ProviderOrganisationIdentity to string.
        /// </summary>
        internal class ProviderOrganisationIdentityConverter : TypeConverter<ProviderOrganisationIdentity, string>
        {
            /// <summary>
            /// Convert the required value.
            /// </summary>
            /// <param name="source">Source to be used when converting.</param>
            /// <returns>Converted value</returns>
            protected override string ConvertCore(ProviderOrganisationIdentity source)
            {
                if (source == null)
                {
                    return "Not specified";
                }

                string value = source.Identifier.Value;

                if (!string.IsNullOrEmpty(value) && source.Identifier.Type.Code == "HPIO")
                {
                    value = source.Identifier.Value.ToHpioFormat();
                }

                if (string.IsNullOrEmpty(source.OrganisationName))
                {
                    return string.Format("[HPI-O] {0}", value);
                }
                else
                {
                    return string.Format("{0} [HPI-O] {1}", source.OrganisationName, value);
                }
            }
        }

        /// <summary>
        /// Converts the value of the ProviderIndividualIdentity to string.
        /// </summary>
        internal class ProviderIndividualIdentityConverter : TypeConverter<ProviderIndividualIdentity, string>
        {
            /// <summary>
            /// Convert the required value.
            /// </summary>
            /// <param name="source">Source to be used when converting.</param>
            /// <returns>Converted value</returns>
            protected override string ConvertCore(ProviderIndividualIdentity source)
            {
                if (source == null)
                {
                    return "Not specified";
                }

                string value = source.Identifier.Value;

                if (!string.IsNullOrEmpty(value) && source.Identifier.Type.Code == "HPII")
                {
                    value = source.Identifier.Value.ToHpioFormat();
                }

                if (source.Name == null)
                {
                    return string.Format("[HPI-I] {0}", value);
                }
                else
                {
                    return string.Format("{0}, {1} [HPI-I] {2}", source.Name.FamilyName, source.Name.GivenNames, value);
                }
            }
        }

        /// <summary>
        /// Converts the value of ConsumerIdentity to string.
        /// </summary>
        internal class ConsumerIdentityConverter : TypeConverter<ConsumerIdentity, string>
        {
            /// <summary>
            /// Convert the required value.
            /// </summary>
            /// <param name="source">Source to be used when converting.</param>
            /// <returns>Converted value</returns>
            protected override string ConvertCore(ConsumerIdentity source)
            {
                if (source == null)
                {
                    return "Not specified";
                }

                if (!string.IsNullOrEmpty(source.Ihi))
                {
                    return string.Format("{0}, {1} [IHI] {2}", source.FamilyName, source.GivenNames, source.Ihi.ToHpioFormat());
                }

                return string.Format("{0}, {1}", source.FamilyName, source.GivenNames);
            }
        }

        #endregion
    }
}