﻿using System;
using HIPS.ConsentSchemas;

namespace HIPS.Web.UI.ViewModels.DisclosureManagement
{
    /// <summary>
    /// View model for a particular patient.
    /// </summary>
    public class PatientViewModel
    {
        #region Properties

        /// <summary>
        /// Gets or sets the name of the patient.
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// Gets or sets the date of birth of the patient.
        /// </summary>
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets the ward of the patient's most recent episode.
        /// </summary>
        public string Ward { get; set; }

        /// <summary>
        /// Gets or sets the room of the patient's most recent episode.
        /// </summary>
        public string Room { get; set; }

        /// <summary>
        /// Gets or sets the bed of the patient's most recent episode.
        /// </summary>
        public string Bed { get; set; }

        /// <summary>
        /// Gets or sets the MRN of the patient.
        /// </summary>
        public string MRN { get; set; }

        /// <summary>
        /// Gets or sets the PCEHR Participation Status of the patient.
        /// </summary>
        public ParticipationStatus PcehrParticipationStatus { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether the patient has disclosed the existence of their PCEHR.
        /// </summary>
        public bool PcehrDisclosed { get; set; }

        /// <summary>
        /// Gets or sets whether the patient has withdrawn consent to upload documents for their most recent episode.
        /// </summary>
        public bool? ConsentWithdrawn { get; set; }

        /// <summary>
        /// Gets or sets the admission date and time of the patient's most recent episode.
        /// </summary>
        public DateTime AdmissionDate { get; set; }

        /// <summary>
        /// Gets or sets the admission reason of the patient's most recent episode.
        /// </summary>
        public string AdmissionReason { get; set; }

        /// <summary>
        /// Gets or sets the discharge date and time of the patient's most recent episode.
        /// </summary>
        public DateTime? DischargeDate { get; set; }

        /// <summary>
        /// Gets or sets the number of episodes for the patient within the configured number of days since discharge.
        /// </summary>
        public int EpisodeCount { get; set; }

        /// <summary>
        /// Gets or sets the family name of the responsible provider of the patient's most recent episode.
        /// </summary>
        public string ResponsibleProviderFamilyName { get; set; }

        /// <summary>
        /// Gets or sets the given names of the responsible provider of the patient's most recent episode.
        /// </summary>
        public string ResponsibleProviderGivenNames { get; set; }

        /// <summary>
        /// Gets or sets the name suffix of the responsible provider of the patient's most recent episode.
        /// </summary>
        public string ResponsibleProviderSuffix { get; set; }

        /// <summary>
        /// Gets or sets the name title of the responsible provider of the patient's most recent episode.
        /// </summary>
        public string ResponsibleProviderTitle { get; set; }

        /// <summary>
        /// Gets or sets the visit number of the patient's most recent episode.
        /// </summary>
        public string EpisodeId { get; set; }

        #region Calculated Properties

        /// <summary>
        /// Gets the location (ward:room:bed) of the patient's most recent episode.
        /// </summary>
        public string Location
        {
            get
            {
                return string.Format("{0}:{1}:{2}", this.Ward, this.Room, this.Bed);
            }
        }

        /// <summary>
        /// Gets a description of the PCEHR status for the patient.
        /// </summary>
        public string PcehrStatus
        {
            get
            {
                if (this.PcehrDisclosed)
                {
                    return "Disclosed";
                }
                switch (this.PcehrParticipationStatus)
                {
                    case ParticipationStatus.NoValidIhi:
                        return "Not Found";

                    case ParticipationStatus.PcehrNotAdvertised:
                        return "Not Found";

                    case ParticipationStatus.PcehrAdvertised:
                        return "Found";

                    case ParticipationStatus.RequestedUpload:
                        return "Disclosed";

                    default:
                        return string.Empty;
                }
            }
        }

        /// <summary>
        /// Gets the name of the responsible provider for the patient's most recent episode.
        /// </summary>
        public string ResponsibleProvider
        {
            get
            {
                return string.Format(
                    "{0} {1} {2} {3}",
                    this.ResponsibleProviderTitle,
                    this.ResponsibleProviderGivenNames,
                    this.ResponsibleProviderFamilyName,
                    this.ResponsibleProviderSuffix);
            }
        }

        #endregion Calculated Properties

        #endregion Properties
    }
}