﻿using System;
using System.Collections.Generic;

using HIPS.P2P.Client.Proxy;
using HIPS.P2P.Service.Contract.Common;
using HIPS.P2P.Service.Contract.Messaging.Receipt.Message;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.P2P.Mapping;
using HIPS.Web.ModelInterface.Messaging;

using CommonSchema = HIPS.Web.Model.Messaging;
using ServiceSchema = HIPS.P2P.Service.Contract.Common.DTO;

namespace HIPS.Web.Data.P2P.Service
{
    /// <summary>
    /// Defines the implementation of the Message Receipt interface.
    /// </summary>
    public class MessageReceiptService : ServiceBase<MessageReceiptProxy>, IMessageReceiptService
    {
        #region Constructor
        
        /// <summary>
        /// Initialises a new instance of the <see cref="MessageReceiptService"/> class.
        /// </summary>
        public MessageReceiptService()
        {
            this.ClientProxy = new MessageReceiptProxy("MessageReceiptEndPoint");
        }
        
        #endregion

        #region Methods

        /// <summary>
        /// Returns a list of inbound sealed messages.
        /// </summary>
        /// <param name="sender">Sender of the inbound sealed message to be return.</param>
        /// <param name="receiver">Receiver of the inbound sealed message to be return.</param>
        /// <param name="receiptPeriod">Start/End of the inbound sealed message to be return.</param>
        /// <param name="messageStatusId">Message status Id of inbound sealed message to be return.</param>        
        /// <param name="documentType">Document type of the inbound sealed message to be return.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<List<CommonSchema.InboundSealedMessage>> QueryMessages(
            CommonSchema.MessageAddressee sender,
            CommonSchema.MessageAddressee receiver,
            CommonSchema.DateTimeRange receiptPeriod,
            int? messageStatusId,
            string documentType,
            Model.Identity.LocalUser userContext,
            int recordLimit,
            int recordSkipFrom)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;
            
            try
            {
                var request = new QueryMessagesRequest();
                request.Receivers.Add(ObjectMapper.Map<ServiceSchema.MessageAddressee>(receiver));
                request.ReceiptPeriod = ObjectMapper.Map<ServiceSchema.DateTimeRange>(receiptPeriod);
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext);
                request.RecordLimit = recordLimit;
                request.RecordSkipFrom = recordSkipFrom; 

                if (sender != null)
                {
                    request.Senders.Add(ObjectMapper.Map<ServiceSchema.MessageAddressee>(sender));
                }

                if (!string.IsNullOrEmpty(documentType))
                {
                    request.DocumentTypes.Add(documentType);
                }

                if (messageStatusId.HasValue)
                {
                    request.MessageStatuses.Add((InboundMessageStatus)messageStatusId.Value);
                }

                var result = this.ClientProxy.QueryMessages(request);

                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<CommonSchema.InboundSealedMessage>>(ObjectMapper.Map<List<CommonSchema.InboundSealedMessage>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.InboundSealedMessage>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Aborts a specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<object> AbortResponse(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new AbortResponseRequest()
                {
                    InvocationIdentifier = invocationIdentifier,
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.AbortResponse(request);
                
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Retries a specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<object> RetryResponse(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new RetryResponseRequest()
                {
                    InvocationIdentifier = invocationIdentifier,
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.RetryResponse(request);

                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Removes a specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<object> RemoveMessage(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new RemoveMessageRequest()
                {
                    InvocationIdentifier = invocationIdentifier,
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.RemoveMessage(request);

                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Downloads the message content of the specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<CommonSchema.BinaryMessageContent> GetMessage(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;
            
            try
            {
                var request = new GetMessageRequest()
                {
                    InvocationIdentifier = invocationIdentifier,
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext),
                    MarkAsRead = false
                };

                var result = this.ClientProxy.GetMessage(request);
                
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<CommonSchema.BinaryMessageContent>(ObjectMapper.Map<CommonSchema.BinaryMessageContent>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.BinaryMessageContent>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Downloads the CDA message content of the specified inbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the inbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<CommonSchema.CdaMessageContent> GetCdaMessage(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            throw new NotImplementedException();
        }

        #endregion
    }
}
