﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;

using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.Web;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.DataIntegrity;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for the Data Integrity feature.
    /// </summary>
    [NoCache]
    [HpoRequired]
    public class DataIntegrityController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Gets the patient repository to be used by this controller.
        /// </summary>
        private readonly IPatientRepository patientRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="DataIntegrityController" /> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="patientRepository">Patient repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public DataIntegrityController(
            IHospitalRepository hospitalRepository,
            IPatientRepository patientRepository,
            ISettingsRepository settingsRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
            this.patientRepository = patientRepository;
        }

        #endregion Constructors

        #region Methods

        #region Actions

        /// <summary>
        /// Display a printable report with a list of patients for a selected hospital.
        /// </summary>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult PatientsWithoutIhiReport()
        {
            return this.PatientsWithoutIhi(usePrintStyles: true);
        }

        /// <summary>
        /// Display a list of patients for a selected hospital.
        /// </summary>
        /// <param name="usePrintStyles">Whether to use print styles.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult PatientsWithoutIhi(bool usePrintStyles = false)
        {
            // Create ViewModel:
            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            var m = new PatientsViewModel() { HospitalId = hospitalId, UsePrintStyles = usePrintStyles };
            this.LoadCurrentContext(m);

            // Load reference data:
            var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                // Update ViewModel with reference data:
                m.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => h.Name);
            }
            else
            {
                m.Messages.Add("No hospitals available for selection.", MessageLevel.Error);
            }

            // Don't allow load for all patients:
            if (hospitalId != null)
            {
                // Load patients for selected hospital.
                var response = this.patientRepository.ListAdmittedPatients(this.GetCurrentUserDetails(), this.DefaultHospitalCodeSystem, hospitalId, withIhi: false, withPcehr: null, excludeMedicareExclusions: true);

                // Ensure loading was successful.
                if (response.IsSuccessful)
                {
                    DateTime oneMonthAgo = DateTime.Now - new TimeSpan(30, 0, 0, 0);
                    var patientsAgedOverOneMonth =
                        from p in response.Data.AdmittedPatientList
                        where p.DateOfBirth <= oneMonthAgo
                        orderby p.Ward, p.FamilyName, p.GivenNames
                        select p;

                    if (patientsAgedOverOneMonth.Count() > 0)
                    {
                        // Update ViewModel with patients.
                        m.Patients.AddRange(ObjectMapper.Map<IEnumerable<PatientViewModel>>(patientsAgedOverOneMonth));
                    }
                    else
                    {
                        m.Messages.Add("There are no patients without IHI at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";
                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    m.Messages.Add(errorMessage, MessageLevel.Error);
                }
            }

            return this.View("PatientsWithoutIhi", m);
        }

        #endregion Actions

        #region Helpers

        /// <summary>
        /// Loads required context into the provided view model.
        /// </summary>
        /// <param name="model">View model to load context into.</param>
        private void LoadCurrentContext(DataIntegrityViewModelBase model)
        {
            // Load current hospital.
            if (!string.IsNullOrEmpty(model.HospitalId))
            {
                var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

                if ((hospitals != null) && (hospitals.Count() > 0))
                {
                    model.CurrentHospital = hospitals.FirstOrDefault(h => h.Code == model.HospitalId);
                }
            }
        }

        #endregion Helpers

        #endregion Methods
    }
}