﻿using System.Linq;

using AutoMapper;
using CommonSchema = HIPS.Web.Model.Common;
using ResourceSchema = HIPS.PcehrDataStore.Schemas;
using UISchema = HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.Helpers.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Hospital entity.
    /// </summary>
    internal class HospitalProfile : Profile
    {
        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Resource --> UI:
            this.CreateMap<ResourceSchema.Hospital, UISchema.HospitalViewModel>()
                .ForMember(dest => dest.CodeSystemCode, opt => opt.ResolveUsing<HospitalCodeSystemResolver>())
                .ForMember(dest => dest.Code, opt => opt.ResolveUsing<HospitalCodeResolver>())
                .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.Name))
                .ForMember(dest => dest.Hpio, opt => opt.MapFrom(src => src.HpiO))
                .ForMember(dest => dest.HpioName, opt => opt.MapFrom(src => src.HpioName))
                .ForMember(dest => dest.HpioLogo, opt => opt.MapFrom(src => src.Logo));

            // Common --> UI:
            this.CreateMap<CommonSchema.Hospital, UISchema.HospitalViewModel>()
                .ForMember(d => d.Code, o => o.MapFrom(s => s.HospitalFacilityCode))
                .ForMember(d => d.CodeSystemCode, o => o.MapFrom(s => s.HospitalFacilityCodeSystem))
                .ForMember(d => d.Hpio, o => o.MapFrom(s => s.Hpio))
                .ForMember(d => d.HpioLogo, o => o.MapFrom(s => s.HpioLogo))
                .ForMember(d => d.HpioName, o => o.MapFrom(s => s.HpioName))
                .ForMember(d => d.Name, o => o.MapFrom(s => s.HospitalName));

            // UI --> Common:
            this.CreateMap<UISchema.HospitalViewModel, CommonSchema.Hospital>()
                .ForMember(d => d.HospitalFacilityCode, o => o.MapFrom(s => s.Code))
                .ForMember(d => d.HospitalFacilityCodeSystem, o => o.MapFrom(s => s.CodeSystemCode))
                .ForMember(d => d.Hpio, o => o.MapFrom(s => s.Hpio))
                .ForMember(d => d.HpioLogo, o => o.MapFrom(s => s.HpioLogo))
                .ForMember(d => d.HpioName, o => o.MapFrom(s => s.HpioName))
                .ForMember(d => d.HospitalName, o => o.MapFrom(s => s.Name));
        }

        #endregion Methods

        #region Declarations

        #region Resolvers

        /// <summary>
        /// Resolves the value of the HospitalCode based on a provided hospital.
        /// </summary>
        internal abstract class HospitalCodeResolverBase : IValueResolver
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            public virtual ResolutionResult Resolve(ResolutionResult source)
            {
                if (source == null)
                {
                    throw new System.ArgumentNullException("source");
                }

                HIPS.PcehrDataStore.Schemas.HospitalCode hospitalCode;

                var context = source.Context.Options.Items.Values.OfType<Context.HospitalMappingContext>().SingleOrDefault();

                if (!string.IsNullOrEmpty(context.HospitalCodeSystem))
                {
                    // Use hospital code system provided in context.
                    hospitalCode = (source.Value as HIPS.PcehrDataStore.Schemas.Hospital).Codes.SingleOrDefault(c => c.CodeSystemCode == context.HospitalCodeSystem);
                }
                else
                {
                    // Use first code as fallback.
                    hospitalCode = (source.Value as HIPS.PcehrDataStore.Schemas.Hospital).Codes[0];
                }

                return source.New(hospitalCode);
            }

            #endregion Methods
        }

        /// <summary>
        /// Resolves the value of the HospitalCodeSystem based on a provided hospital.
        /// </summary>
        internal class HospitalCodeSystemResolver : HospitalCodeResolverBase
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            public override ResolutionResult Resolve(ResolutionResult source)
            {
                return source.New((base.Resolve(source).Value as HIPS.PcehrDataStore.Schemas.HospitalCode).CodeSystemCode);
            }

            #endregion Methods
        }

        /// <summary>
        /// Resolves the value of the HospitalCode based on a provided hospital.
        /// </summary>
        internal class HospitalCodeResolver : HospitalCodeResolverBase
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            public override ResolutionResult Resolve(ResolutionResult source)
            {
                return source.New((base.Resolve(source).Value as HIPS.PcehrDataStore.Schemas.HospitalCode).Code);
            }

            #endregion Methods
        }

        #endregion Resolvers

        #endregion Declarations
    }
}