﻿using System;
using System.Text;

namespace HIPS.Web.UI.ViewModels.ConsentManagement
{
    /// <summary>
    /// View model for a single episode of care when the patient was admitted to hospital.
    /// </summary>
    public class EpisodeViewModel
    {
        #region Properties

        /// <summary>
        /// Gets or sets the date and time when the patient was admitted.
        /// </summary>
        public DateTime AdmissionDate { get; set; }

        /// <summary>
        /// Gets or sets the reason why the patient was admitted.
        /// </summary>
        public string AdmissionReason { get; set; }

        /// <summary>
        /// Gets or sets the bed component of the assigned location for the patient within this episode.
        /// </summary>
        public string Bed { get; set; }

        /// <summary>
        /// Gets or sets a value that indicates whether consent has been withdrawn for this episode.
        /// </summary>
        public bool? ConsentWithdrawn { get; set; }

        /// <summary>
        /// Gets or sets the date and time when the patient was discharged, or null if the patient is still in hospital.
        /// </summary>
        public DateTime? DischargeDate { get; set; }

        /// <summary>
        /// Gets or sets the family name of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderFamilyName { get; set; }

        /// <summary>
        /// Gets or sets the given name of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderGivenNames { get; set; }

        /// <summary>
        /// Gets or sets the name suffix of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderSuffix { get; set; }

        /// <summary>
        /// Gets or sets the name title of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderTitle { get; set; }

        /// <summary>
        /// Gets or sets the room component of the assigned location for the patient within this episode.
        /// </summary>
        public string Room { get; set; }

        /// <summary>
        /// Gets or sets the visit number that identifies this episode.
        /// </summary>
        public string SourceSystemEpisodeId { get; set; }

        /// <summary>
        /// Gets or sets the ward component of the assigned location for the patient within this episode.
        /// </summary>
        public string Ward { get; set; }

        /// <summary>
        /// Gets or sets the notes that were entered when the patient withdrew consent to upload documents for this episode.
        /// </summary>
        public byte[] WithdrawalAuditInformation { get; set; }

        #endregion Properties

        #region Calculated Properties

        /// <summary>
        /// Gets the name of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProvider
        {
            get
            {
                return string.Format(
                    "{0} {1} {2} {3}",
                    this.ResponsibleProviderTitle,
                    this.ResponsibleProviderGivenNames,
                    this.ResponsibleProviderFamilyName,
                    this.ResponsibleProviderSuffix);
            }
        }

        /// <summary>
        /// Gets the last known assigned location of the patient for this episode, formatted as Ward:Room:Bed.
        /// </summary>
        public string Location
        {
            get
            {
                return string.Format("{0}:{1}:{2}", this.Ward, this.Room, this.Bed);
            }
        }

        /// <summary>
        /// Gets the notes that were entered when the patient withdrew consent to upload documents for this episode.
        /// </summary>
        public string WithdrawalNotes
        {
            get
            {
                return this.WithdrawalAuditInformation == null ? "(None)" : Encoding.UTF8.GetString(this.WithdrawalAuditInformation);
            }
        }

        #endregion Calculated Properties
    }
}