﻿using System;
using System.ComponentModel.DataAnnotations;

using HIPS.ConsentSchemas;

namespace HIPS.Web.UI.ViewModels.DocumentUpload
{
    /// <summary>
    /// View model for a particular patient.
    /// </summary>
    public class PatientViewModel
    {
        #region Properties

        /// <summary>
        /// Gets or sets the MRN of the patient.
        /// </summary>
        public string Mrn { get; set; }

        /// <summary>
        /// Gets or sets the Family name of the patient.
        /// </summary>
        public string FamilyName { get; set; }

        /// <summary>
        /// Gets or sets the Given names of the patient.
        /// </summary>
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets the date of birth of the patient.
        /// </summary>
        [Display(Name = "Date of Birth")]
        [DisplayFormat(DataFormatString = "{0:dd/MM/yyyy}")]
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets the ward of the patient's most recent episode.
        /// </summary>
        public string Ward { get; set; }

        /// <summary>
        /// Gets or sets the room of the patient's most recent episode.
        /// </summary>
        public string Room { get; set; }

        /// <summary>
        /// Gets or sets the bed of the patient's most recent episode.
        /// </summary>
        public string Bed { get; set; }

        /// <summary>
        /// Gets or sets the PCEHR Participation Status of the patient.
        /// </summary>
        public ParticipationStatus ParticipationStatus { get; set; }

        #region Calculated Properties

        /// <summary>
        /// Gets the name of the patient.
        /// </summary>
        public string Name
        {
            get { return string.Format("{0}, {1}", this.FamilyName, this.GivenNames); }
        }

        /// <summary>
        /// Gets the name of the patient formatted with MRN.
        /// </summary>
        [Display(Name = "Patient")]
        public string NameWithMrn
        {
            get { return string.Format("{0} ({1})", this.Name, this.Mrn); }
        }

        /// <summary>
        /// Gets whether if the patient has PCEHR.
        /// </summary>
        public string HasPcehr
        {
            get { return (this.ParticipationStatus == ParticipationStatus.RequestedUpload || this.ParticipationStatus == ParticipationStatus.PcehrAdvertised) ? "Yes" : "No"; }
        }

        /// <summary>
        /// Gets the location (ward:room:bed) of the patient's most recent episode.
        /// </summary>
        public string Location
        {
            get
            {
                return string.Format("{0}:{1}:{2}", this.Ward, this.Room, this.Bed);
            }
        }

        #endregion Calculated Properties

        #endregion Properties
    }
}