﻿using System;
using System.Collections.Generic;
using System.ServiceModel;

using HIPS.AppServer.HIPSServiceHost.Services;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.AppServer.ServiceHost.Utility;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Cda;
using HIPS.CommonBusinessLogic.Service;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.Cda;
using HIPS.ServiceContracts.Cda.Message;
using HIPS.ServiceContracts.Common.DTO;
using HIPS.ServiceContracts.Common.Fault;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services that operate on CDA documents.
    /// </summary>
    public class CdaService : BaseService, ICdaService
    {
        #region Methods

        /// <summary>
        /// Package a CDA document.
        /// Used by the P2P module to package a CDA® document for sending via the SMD. 
        /// All validation is done as part of the P2P module before the CDA® Packaging is invoked.
        /// A successful response indicates the document has been sent to the SMD service successfully. 
        /// The SMD component will attempt to send the message and any errors are handled in this component.
        /// </summary>
        /// <param name="request">Request Message</param>
        /// <returns>Response Message</returns>
        public PackageCdaResponse PackageCdaDocument(PackageCdaRequest request)
        {
            UserDetails user = null;
            try
            {
                user = ObjectMapper.Map<UserDetails>(request.User);
                CdaDocumentService businessLogic = new CdaDocumentService(user);
                var metadata = ObjectMapper.Map<CommonSchemas.Cda.CdaMetadata>(request);
                var businessResult = businessLogic.Package(metadata);
                var serviceResult = new ServiceContracts.Common.DTO.CdaPackage()
                {
                    Content = businessResult,
                    Format = request.Format
                };
                return new PackageCdaResponse()
                {
                    Data = serviceResult,
                    Status = ServiceContracts.Common.ResponseStatus.OK
                };
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToPackageCdaDocument, ex, user, PcehrDataStore.Schemas.Enumerators.LogMessage.HIPS_MESSAGE_164);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Unpackage a CDA document.
        /// Used by the P2P module to unpack a CDA® document that has been sent via the SMD Component. 
        /// The result is sent back as the Unpackage CDA® Document response. 
        /// </summary>
        /// <param name="request">Request Message</param>
        /// <returns>Response Message</returns>
        public UnpackageCdaResponse UnpackageCdaDocument(UnpackageCdaRequest request)
        {
            UserDetails user = null;
            try
            {
                user = ObjectMapper.Map<UserDetails>(request.User);
                CdaDocumentService businessLogic = new CdaDocumentService(user);
                var businessFormat = request.Package.Format.ToCommonModel();
                var businessResult = businessLogic.Unpackage(request.Package.Content, businessFormat);
                var serviceResult = ObjectMapper.Map<ServiceContracts.Common.DTO.CdaDocument>(businessResult);
                return new UnpackageCdaResponse()
                {
                    Data = serviceResult,
                    Status = ServiceContracts.Common.ResponseStatus.OK
                };
            }
            catch (InvalidCertificateException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUnpackageCdaDocument, ex, user, PcehrDataStore.Schemas.Enumerators.LogMessage.HIPS_MESSAGE_165);
                throw ObjectMapper.Map<FaultException<InvalidCertificateFault>>(ex);
            }
            catch (CdaPackagingException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUnpackageCdaDocument, ex, user, PcehrDataStore.Schemas.Enumerators.LogMessage.HIPS_MESSAGE_166);
                throw ObjectMapper.Map<FaultException<CdaPackagingFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUnpackageCdaDocument, ex, user, PcehrDataStore.Schemas.Enumerators.LogMessage.HIPS_MESSAGE_167);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Get the content from a provided CDA unpackaged document.
        /// </summary>
        /// <param name="request">Request message.</param>
        /// <returns>Response message.</returns>
        public GetCdaContentResponse GetCdaContent(GetCdaContentRequest request)
        {
            UserDetails user = null;
            try
            {
                user = ObjectMapper.Map<UserDetails>(request.User);
                CdaDocumentService businessLogic = new CdaDocumentService(user);
                var businessResult = businessLogic.GetContent(ObjectMapper.Map<CommonSchemas.Cda.CdaDocument>(request.Document));
                var serviceResult = ObjectMapper.Map<CdaDocumentContent>(businessResult);
                return new GetCdaContentResponse()
                {
                    Data = serviceResult,
                    Status = ServiceContracts.Common.ResponseStatus.OK
                };
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToGetCdaDocumentContent, ex, user, PcehrDataStore.Schemas.Enumerators.LogMessage.HIPS_MESSAGE_168);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Creates a CDA discharge summary document that wraps a PDF document body.
        /// This may be for the purpose of uploading the discharge summary to the
        /// PCEHR or for provider-to-provider (P2P) secure message delivery (SMD).
        /// This takes in the PDF document, required metadata, and any attachments
        /// and creates a CDA document before packaging it.
        /// </summary>
        /// <param name="request">Request containing a PDF and required metadata for creating the CDA package.</param>
        /// <returns>Response containing details of the created document.</returns>
        public CreateDischargeSummaryLevel1AResponse CreateDischargeSummaryLevel1A(CreateDischargeSummaryLevel1ARequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;
            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var metadata = ObjectMapper.Map<HIPS.CommonSchemas.Cda.CdaHeaderMetadata>(request.CdaHeaderMetadata);
                var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(request.PatientIdentifier);
                var attachments = ObjectMapper.Map<List<HIPS.CommonSchemas.Cda.CdaAttachment>>(request.Attachments);

                // Invoke Business Service
                var businessService = new DischargeSummaryService(user);
                var businessResponse = businessService.CreateDischargeSummaryLevel1A(request.PdfDocument, metadata, patientIdentifier, attachments);

                // Map Common Model to DTO
                var svcResponse = ObjectMapper.Map<CreateDischargeSummaryLevel1AResponse>(businessResponse);
                return svcResponse;
            }
            catch (PcehrServiceException ex)
            {
                // Catch any business exceptions, log them in the “SystemErrorLog”
                // and map them to SOAP faults using AutoMapper. Ensure that the
                // message or code in each business exception is returned in the
                // SOAP fault.
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<PcehrServiceFault>>(ex);
            }
            catch (HiServiceException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<HiServiceFault>>(ex);
            }
            catch (ItemNotFoundException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<ItemNotFoundFault>>(ex);
            }
            catch (InvalidUserException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<InvalidUserFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToCreateDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        #endregion Methods
    }
}