﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using AutoMapper;
using HIPS.PcehrDataStore.Schemas;
using HIPS.CommonBusinessLogic.Singleton;

using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;
using System.Security.Cryptography;


namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Related Document.
    /// </summary>
    internal class RelatedDocumentProfile : Profile
    {
        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }
        
        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA

            #region Common
            // Pathology/Diagbostic Imaging - SCSContent - RelatedDocument
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.RelatedDocument>()
                .ForMember(dest => dest.CustomNarrative, opt => opt.MapFrom(src => src.CdaDocumentDetails.EventNarrativeText))
                .ForMember(dest => dest.ExaminationResultRepresentation, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.DocumentDetails, opt => opt.MapFrom(src => src));
            

            // Diagnostic Imaging - SCSContent - RelatedDocument - ExaminationResultRepresentation
            // Pathology - SCSContent - RelatedDocument - ExaminationResultRepresentation
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.Common.Entities.ExternalData>()
                .ForMember(dest => dest.ExternalDataMediaType, opt => opt.UseValue(NehtaSchemas.Generator.Enums.MediaType.PDF))
                .ForMember(dest => dest.ByteArrayInput, opt => opt.MapFrom(src => src.CdaDocumentDetails.Attachments.SingleOrDefault(a => a.AttachmentType != ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng)));
                
            this.CreateMap<ModelSchemas.Cda.CdaAttachment, NehtaSchemas.SCSModel.Common.Entities.ByteArrayInput>()
                .ForMember(dest => dest.ByteArray, opt => opt.MapFrom(src => src.Content))
                .ForMember(dest => dest.FileName, opt => opt.MapFrom(src => src.FileName));
            
            // CommonSchemas.Cda --> PcehrSchemas
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, PcehrSchemas.Attachment>()
                .ForMember(dest => dest.Contents, opt => opt.MapFrom(src => src.CdaDocumentDetails.Attachments.SingleOrDefault(a => a.AttachmentType != ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng).Content))
                .ForMember(dest => dest.FileName, opt => opt.MapFrom(src => src.CdaDocumentDetails.Attachments.SingleOrDefault(a => a.AttachmentType != ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng).FileName));

            // Pathology/Diagnostic Imaging - SCSContent - RelatedDocument - DocumentDetails
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, CDA.Generator.Common.SCSModel.Entities.DocumentDetails>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.BaseCDAModel.CreateDocumentDetails())
                .ForMember(dest => dest.ReportDescription, opt => opt.MapFrom(src => src.OrderDetails.PathologyRequestedTestName))
                .ForMember(dest => dest.ReportDate, opt => opt.MapFrom(src => src.CdaDocumentDetails.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.ReportStatus, opt => opt.ResolveUsing<ReportStatusResolver>().FromMember(src => src.RelatedDocumentDetails))
                .ForMember(dest => dest.ReportIdentifier, opt => opt.ResolveUsing<ReportIdentifierResolver>().FromMember(src => src));
            
            #endregion
        }

        /// <summary>
        /// Resolves the value of ReportStatus.
        /// </summary>
        internal class ReportStatusResolver : ValueResolver<ModelSchemas.Cda.RelatedDocumentDetails, NehtaSchemas.SCSModel.ICodableText>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override NehtaSchemas.SCSModel.ICodableText ResolveCore(ModelSchemas.Cda.RelatedDocumentDetails source)
            {
                switch (source.ResultStatus)
                {
                    case "C":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.CorrectionToResults);
                        
                    case "F":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.FinalResults);
                        
                    case "I":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoResultsAvailableSpecimenReceivedProcedureIncomplete);
                                           
                    case "O":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.OrderReceived);
                        
                    case "P":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.Preliminary);
                                            
                    case "R":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.ResultsStored);
                                           
                    case "S":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoResultsAvailableProcedureScheduledButNotDone);
                                            
                    case "A":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.SomeButNotAllResultsAvailable);
                        
                    case "X":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoResultsAvailableOrderCanceled);
                        
                    case "Y":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoOrderOnRecordForThisTest);
                        
                    case "Z":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoRecordOfThisPatient);
                   
                }

                // If we're here then it didn't map correctly so add a Null Flavour for the codeable text
                return NehtaSchemas.Common.BaseCDAModel.CreateCodableText(NehtaSchemas.Common.Enums.NullFlavour.NoInformation);
            }


            #endregion Methods
        }

        /// <summary>
        /// Resolves the value of ReportIdentifier.
        /// </summary>
        internal class ReportIdentifierResolver : ValueResolver<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.Common.Identifier>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override NehtaSchemas.SCSModel.Common.Identifier ResolveCore(ModelSchemas.Cda.CdaPathDIDocumentDetails source)
            {
                if (source.PathologyTestResult != null && source.PathologyTestResult.Count > 0) // Pathology Report add an Identifier
                    return NehtaSchemas.Common.PathologyResultReport.CreateIdentifier(string.Format("1.2.36.1.2001.1005.54.{0}", source.CdaDocumentDetails.Hospital.HpiO), string.Format("{0}.{1}", source.CdaDocumentDetails.DocumentSetId.Extension, source.CdaDocumentDetails.DocumentId.Extension));
               
                return (NehtaSchemas.SCSModel.Common.Identifier)(null); // otherwise return a null identifier
            }

           
            #endregion Methods
        }
    }
}
