﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.Base.Schemas.Enumerators;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonBusinessLogic.HL7;
using HIPS.CommonBusinessLogic.Hpii;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.CommonSchemas.Cda.ParticipatingIndividual;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.HL7.Common.Segment;
using HIPS.HL7.Common.SegmentGroup;
using HIPS.HL7.Common.Enumerators;
using HIPS.HL7.Common.DataStructure;
using HIPS.HpiiSchemas;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrDataStore.DataAccess.DataAccess;
using HIPS.PcehrDataStore.Schemas.Schemas;
using HIPS.PcehrSchemas;
using hips = HIPS.PcehrDataStore.Schemas.Schemas;
using System.Transactions;
using HIPS.Configuration.Tracing;
using nehta.mcaR50.ProviderSearchForProviderIndividual;
using HIPS.HpiiSchemas.Enumerators;

namespace HIPS.CommonBusinessLogic
{
    /// <summary>
    /// Class to validate the patient information
    /// </summary>
    public class PatientAccess : BaseDl
    {
        #region Private Fields

        /// <summary>
        /// The episode type code that is used for ValidatedIhi stub episodes.
        /// </summary>
        private const string EPISODE_STUB_CODE = "STUB";

        /// <summary>
        /// The tolerance for how closely the admission date/time must be specified
        /// in order to match to an existing episode record. Currently set to 1.0
        /// minutes.
        /// </summary>
        private const double MAXIMUM_ADMISSION_TIME_DIFFERENCE_MINUTES = 1.0;   

        /// <summary>
        /// Hospital Data Access
        /// </summary>
        private HospitalDl hospitalDataAccess;
        
        /// <summary>
        /// Patient Data Access
        /// </summary>
        private HospitalPatientDl patientDataAccess;
        
        /// <summary>
        /// Patient Master Data Access
        /// </summary>
        private PatientMasterDl masterDataAccess;
        
        /// <summary>
        /// Episode Data Access
        /// </summary>
        private EpisodeDl episodeDataAccess;
        
        /// <summary>
        /// IHI Data access
        /// </summary>
        private PatientMasterIhiDl ihiDataAccess;
        
        /// <summary>
        /// Document Data Access
        /// </summary>
        private ClinicalDocumentDl documentDataAccess;
        
        /// <summary>
        /// Version Data Access
        /// </summary>
        private ClinicalDocumentVersionDl versionDataAccess;
        
        /// <summary>
        /// Health Provider Organisation Patient Data Access
        /// </summary>
        private HealthProviderOrganisationPatientDl healthProviderOrganisationPatientDataAccess;
        
        /// <summary>
        /// Hospitals List
        /// </summary>
        private List<Hospital> hospitals;
        
        /// <summary>
        /// PendingIhiPcehrLookup Data Access
        /// </summary>
        private PendingIhiPcehrLookupDl pendingIhiPcehrLookupDataAccess;

        /// <summary>
        /// The default SqlTransaction
        /// </summary>
        private SqlTransaction transaction;

        // Defaults to the Home code for addresstype
        private const string UnknownAddressType = "H";

        // Placeholder values for situations when information is unknown or not supplied.
        private const int UnknownForeignKey = -1;

        #endregion Private Fields

        #region Constructors

        public PatientAccess(UserDetails user)
        {
            base.User = user;
        }

        #endregion Constructors

        #region Properties

        public ClinicalDocumentDl DocumentDataAccess
        {
            get
            {
                if (documentDataAccess == null)
                {
                    documentDataAccess = new ClinicalDocumentDl(User);
                }
                return documentDataAccess;
            }
        }

        public EpisodeDl EpisodeDataAccess
        {
            get
            {
                if (episodeDataAccess == null)
                {
                    episodeDataAccess = new EpisodeDl(User);
                }
                return episodeDataAccess;
            }
        }

        public HealthProviderOrganisationPatientDl HealthProviderOrganisationPatientDataAccess
        {
            get
            {
                if (healthProviderOrganisationPatientDataAccess == null)
                {
                    healthProviderOrganisationPatientDataAccess = new HealthProviderOrganisationPatientDl(User);
                }
                return healthProviderOrganisationPatientDataAccess;
            }
        }

        public HospitalDl HospitalDataAccess
        {
            get
            {
                if (hospitalDataAccess == null)
                {
                    hospitalDataAccess = new HospitalDl();
                }

                return hospitalDataAccess;
            }
        }

        public HospitalPatientDl HospitalPatientDataAccess
        {
            get
            {
                if (patientDataAccess == null)
                {
                    patientDataAccess = new HospitalPatientDl(User);
                }

                return patientDataAccess;
            }
        }

        public List<Hospital> Hospitals
        {
            get
            {
                if (hospitals == null)
                {
                    hospitals = HospitalDataAccess.GetAll();
                }
                return hospitals;
            }
        }

        public PatientMasterDl PatientMasterDataAccess
        {
            get
            {
                if (masterDataAccess == null)
                {
                    masterDataAccess = new PatientMasterDl(User);
                }
                return masterDataAccess;
            }
        }

        public PatientMasterIhiDl IhiDataAccess
        {
            get
            {
                if (ihiDataAccess == null)
                {
                    ihiDataAccess = new PatientMasterIhiDl(User);
                }

                return ihiDataAccess;
            }
        }

        public PendingIhiPcehrLookupDl PendingIhiPcehrLookupDataAccess
        {
            get
            {
                if (pendingIhiPcehrLookupDataAccess == null)
                {
                    pendingIhiPcehrLookupDataAccess = new PendingIhiPcehrLookupDl(User);
                }
                return pendingIhiPcehrLookupDataAccess;
            }
        }     

        public ClinicalDocumentVersionDl VersionDataAccess
        {
            get
            {
                if (versionDataAccess == null)
                {
                    versionDataAccess = new ClinicalDocumentVersionDl(User);
                }

                return versionDataAccess;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Attempts to find the matching episode given an source system episode id.
        /// If the hospital patient has more than one episode with the given id then none match.
        /// </summary>
        /// <param name="sourceSystemEpisodeId">The source system episode identifier.</param>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <returns>The matched episode, or null if none matched</returns>
        public Episode GetEpisode(string sourceSystemEpisodeId, HospitalPatient hospitalPatient)
        {
            // Specific Episode must be found from the SourceSystemEpisode (Visit Number) in this case, and so all episodes
            // can be returned, including cancelled, as the result will be filtered specifically by the sourceSystemEpisodeId.
            var matched = from episode in EpisodeDataAccess.GetAll(hospitalPatient.PatientId, null)
                          where episode.SourceSystemEpisodeId == sourceSystemEpisodeId
                          select episode;
            if (matched.Count() == 1)
            {
                return matched.First();
            }
            return null;
        }

        /// <summary>
        /// Attempts to find the matching episode given an source system episode id and the Patient Id.
        /// If the hospital patient has more than one episode with the given source system episode id then none match
        /// and an error will be returned.
        /// </summary>
        /// <param name="admissionDate">The admission date/time</param>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <returns>The matched episode, or null if none matched</returns>
        public EpisodePatientExtendedDetails GetEpisodePatientExtendedDetails(string sourceSystemEpisodeId, HospitalPatient hospitalPatient, PatientIdentifierBase patientIdentifier)
        {
            // Specific Episode must be found from the SourceSystemEpisode (Visit Number) in this case, and so all episodes
            // can be returned, including cancelled, as the result will be filtered specifically by the sourceSystemEpisodeId.
            EpisodePatientExtendedDetails episode = EpisodeDataAccess.GetEpisodePatientExtendedDetails(sourceSystemEpisodeId, hospitalPatient, patientIdentifier.HospitalCodeSystem);
            return episode;
        }

        /// <summary>
        /// Attempts to find (or create) the matching episode given a document
        /// set ID and/or admission date/time.
        ///
        /// If a set ID is provided and a stub episode has been created for this
        /// set ID, then that episode is returned. In addition, if its admission
        /// date/time is outside the configured tolerance from the provided
        /// admission date/time, then the stub's admission date/time is changed
        /// to the provided one.
        ///
        /// If the hospital patient has more than one episode within the
        /// configured tolerance then none match. The episodes will need to be
        /// merged, either on the hospital PAS if supported, or by manual
        /// submission of an A35 message to HIPS, or by direct database
        /// manipulation.  Any Cancelled episodes will not be returned
        ///
        /// If the patient identifier is a validated IHI and the episode is not
        /// found through either of the mechanisms described above, then this
        /// method will create and insert a stub record for the episode. If a
        /// document set ID was provided (e.g. for a document upload operation)
        /// then the stub will be identified by the document set ID. Otherwise,
        /// there is no set ID (e.g. for a consent withdrawal), and so the stub
        /// will be identified by the admission date/time.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The admission date/time</param>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <param name="sourceSystemSetId">The set ID of a previously uploaded document (optional)</param>
        /// <returns>The matched episode, or null if none matched</returns>
        /// <exception cref="System.Exception">If a database error occurs while updating or inserting a stub episode</exception>
        public Episode GetEpisodeWithoutCancelled(PatientIdentifierBase patientIdentifier, DateTime admissionDate, HospitalPatient hospitalPatient, string sourceSystemSetId = null)
        {
            List<Episode> episodes = EpisodeDataAccess.GetAll(hospitalPatient.PatientId, null, true);
            EpisodeType stubType = ListSingleton.Instance.AllEpisodeTypes.FirstOrDefault(a => a.Code == EPISODE_STUB_CODE);
            Episode stub = episodes.FirstOrDefault(a => a.SourceSystemEpisodeId == sourceSystemSetId && a.EpisodeTypeId == stubType.EpisodeTypeId.Value);
            if (stub != null)
            {
                if (Math.Abs((stub.AdmissionDate - admissionDate).TotalMinutes) > MAXIMUM_ADMISSION_TIME_DIFFERENCE_MINUTES)
                {
                    stub.AdmissionDate = admissionDate;
                    if (!EpisodeDataAccess.Update(stub, null))
                    {
                        throw new Exception(string.Format(ResponseStrings.DatabaseError, EpisodeDataAccess.GetType().FullName));
                    }
                }
                return stub;
            }
            var matched = from ep in episodes
                          where Math.Abs((ep.AdmissionDate - admissionDate).TotalMinutes) < MAXIMUM_ADMISSION_TIME_DIFFERENCE_MINUTES
                          select ep;
            if (matched.Count() == 1)
            {
                return matched.First();
            }
            if (matched.Count() == 0 && patientIdentifier is ValidatedIhi)
            {
                stub = new Episode();
                stub.PatientId = hospitalPatient.PatientId.Value;
                stub.AdmissionDate = admissionDate;
                stub.SourceSystemEpisodeId = sourceSystemSetId ?? admissionDate.ToString("yyyyMMddHHmmss");
                stub.ResponsibleProviderId = -1;
                stub.EpisodeLifecycleId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.EpisodeLifecycle.Admitted;
                stub.EpisodeTypeId = stubType.EpisodeTypeId.Value;
                if (!EpisodeDataAccess.Insert(stub, null))
                {
                    throw new Exception(string.Format(ResponseStrings.DatabaseError, EpisodeDataAccess.GetType().FullName));
                }
                return stub;
            }
            return null;
        }

        /// <summary>
        /// Finds the hospital using either the internal hospital ID (if the supplied patient identifier is of PatientMasterId type) or the hospital code and hospital code system.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="hospital">Out. The hospital that was found.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        public HipsResponse GetHospital(PatientIdentifierBase patientIdentifier, out Hospital hospital)
        {
            if (HospitalSingleton.Value.IsHospitalListEmpty())
            {
                // Try to recover from database being down on HIPS startup.
                ListSingleton.Instance.Refresh(force: true);
            }

            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            PatientMasterId patientMasterId = patientIdentifier as PatientMasterId;
            if (patientMasterId != null && patientMasterId.HospitalId.HasValue)
            {
                hospital = HospitalSingleton.Value.Find(patientMasterId.HospitalId.Value);
            }
            else
            {
                hospital = HospitalSingleton.Value.Find(patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            }

            if (hospital == null)
            {
                if (HospitalSingleton.Value.IsHospitalListEmpty())
                {
                    response.Status = HipsResponseIndicator.DatabaseError;
                    response.HipsErrorMessage = ResponseStrings.UnableToGetHospitalList;
                }
                else
                {
                    response.Status = HipsResponseIndicator.InvalidHospital;
                    response.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                }

                return response;
            }

            return response;
        }

        /// <summary>
        /// Finds the visitor hospital using either the internal hospital ID (if the supplied patient identifier is of PatientMasterId type) or the hospital code and hospital code system.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="hospital">Out. The hospital that was found.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        public HipsResponse GetVisitorHospital(PatientIdentifierBase patientIdentifier, out Hospital hospital)
        {
            if (HospitalSingleton.Value.IsHospitalListEmpty())
            {
                // Try to recover from database being down on HIPS startup.
                ListSingleton.Instance.Refresh(force: true);
            }

            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            PatientMasterId patientMasterId = patientIdentifier as PatientMasterId;
            if (patientMasterId != null && patientMasterId.HospitalId.HasValue)
            {
                hospital = HospitalSingleton.Value.FindVisitorHospital(patientMasterId.HospitalId.Value);
            }
            else
            {
                hospital = HospitalSingleton.Value.FindVisitorHospital(patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            }

            if (hospital == null)
            {
                if (HospitalSingleton.Value.IsHospitalListEmpty())
                {
                    response.Status = HipsResponseIndicator.DatabaseError;
                    response.HipsErrorMessage = ResponseStrings.UnableToGetHospitalList;
                }
                else
                {
                    response.Status = HipsResponseIndicator.InvalidHospital;
                    string notfound = patientMasterId != null && patientMasterId.HospitalId.HasValue
                        ? string.Format(ResponseStrings.VisitorHospitalNotFound, patientIdentifier.HospitalCode, patientMasterId.HospitalId)
                        : string.Format(ResponseStrings.VisitorHospitalNotFoundByCode, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
                    response.HipsErrorMessage = notfound;
                }
                return response;
            }

            return response;
        }

        /// <summary>
        /// Based on the type of patient identifier supplied, reads the hospital patient and patient master from the database.
        /// In case of the type Validated IHI, it will automatically create minimal records with a dummy MRN for those
        /// jurisdictions who plan to avoid using HIPS as an HI Service platform.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="hospitalPatient">Returns the hospital patient.</param>
        /// <param name="patientMaster">Returns the hospital patient.</param>
        /// <returns>Success (OK) or reason for failure (DatabaseError or InvalidPatient or InvalidIhi or UnresolvedIhiAlert or SystemError)</returns>
        public HipsResponse GetPatient(PatientIdentifierBase patientIdentifier, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster, bool allowDemographics = false)
        {
            HipsResponse response;
            if (patientIdentifier is Mrn)
            {
                response = GetPatient((patientIdentifier as Mrn).Value, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is ValidatedIhi)
            {
                response = ReadCreateOrUpdatePatient(patientIdentifier as ValidatedIhi, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is StatePatientId)
            {
                response = GetPatient(patientIdentifier as StatePatientId, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is PatientMasterId)
            {
                response = GetPatient(patientIdentifier as PatientMasterId, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is RegisteredEnterprisePatient)
            {
                response = GetPatient(patientIdentifier as RegisteredEnterprisePatient, hospital, out hospitalPatient, out patientMaster);
            }
            else if (patientIdentifier is Demographic)
            {
                if (allowDemographics)
                {
                    response = GetPatient(patientIdentifier as Demographic, hospital, out hospitalPatient, out patientMaster);
                }
                else
                {
                    patientMaster = null;
                    hospitalPatient = null;
                    response = new HipsResponse(HipsResponseIndicator.SystemError, ResponseStrings.DemographicsNotAllowedForThisService);
                }
            }
            else
            {
                patientMaster = null;
                hospitalPatient = null;
                response = new HipsResponse(HipsResponseIndicator.SystemError, ResponseStrings.UnknownPatientIdentifierType);
            }

            if (response.Status == HipsResponseIndicator.OK)
            {
                ValidateLocalIhiInformation(patientMaster, response);
            }

            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given patient master ID.
        /// </summary>
        /// <param name="patientMasterId">The patient master ID</param>
        /// <param name="hospital">The hospital to get the hospital patient record from.</param>
        /// <param name="hospitalPatient">Out. The hospital record for this patient.</param>
        /// <param name="patientMaster">Out. The patient master record for this patient.</param>
        /// <returns>Success (OK) or reason for failure (DatabaseError or InvalidPatient)</returns>
        public HipsResponse GetPatient(PatientMasterId patientMasterId, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = PatientMasterDataAccess.Get(patientMasterId.Value, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }

            response = HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }

            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given state patient identifier.
        /// </summary>
        /// <param name="statePatientId">The state patient identifier.</param>
        /// <param name="hospital">The hospital to get the hospital patient record from.</param>
        /// <param name="hospitalPatient">Out. The hospital record for this patient.</param>
        /// <param name="patientMaster">Out. The patient master record for this patient.</param>
        /// <returns>Information about success or failure.</returns>
        public HipsResponse GetPatient(StatePatientId statePatientId, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = PatientMasterDataAccess.GetByStatePatientId(statePatientId.Value, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                patientMaster = null;
                hospitalPatient = null;
                return response;
            }

            response = HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given MRN.
        /// </summary>
        /// <param name="mrn"> The facility-level local identifier for the patient.</param>
        /// <param name="hospital">The hospital to get the hospital patient record from.</param>
        /// <param name="hospitalPatient">Out. The hospital record for this patient.</param>
        /// <param name="patientMaster">Out. The patient master record for this patient.</param>
        /// <returns>Success (OK) or reason for failure (DatabaseError or InvalidPatient)</returns>
        public HipsResponse GetPatient(string mrn, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = HospitalPatientDataAccess.Get(hospital.HospitalId.Value, mrn, out hospitalPatient);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            response = PatientMasterDataAccess.Get(hospitalPatient.PatientMasterId, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;
                return response;
            }
            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given demographics.
        /// </summary>
        /// <param name="demographic">The demographic details for the target patient.</param>
        /// <param name="hospital">The hospital to get the hospital patient record from.</param>
        /// <param name="hospitalPatient">Out. The hospital record for this patient.</param>
        /// <param name="patientMaster">Out. The patient master record for this patient.</param>
        /// <returns>Success (OK) or reason for failure (DatabaseError or InvalidPatient)</returns>
        public HipsResponse GetPatient(Demographic demographic, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.SystemError);
            this.LookupByDemographicsWithinHospital(demographic.FamilyName, demographic.GivenName, demographic.DateOfBirth, (int)demographic.Sex, demographic.MedicareNumber, demographic.MedicareIrn, demographic.DvaNumber, hospital, out patientMaster);
            if (patientMaster == null)
            {
                response.Status = HipsResponseIndicator.SystemError;
                hospitalPatient = new HospitalPatient();
                return response;
            }

            if (!patientMaster.PatientMasterId.HasValue)
            {
                // no matching patient so add them
                this.CreatePatient(demographic, hospital, out hospitalPatient, out patientMaster);
                response.Status = HipsResponseIndicator.OK;
            }
            else
            {
                response = HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    response.Status = HipsResponseIndicator.SystemError;
                    return response;
                }
            }

            return response;
        }

        /// <summary>
        /// Reads the patient master and hospital patient for the given Registered Enterprise Patient.
        /// </summary>
        /// <param name="registeredEnterprisePatient">The enterprise-level identifier for the patient.</param>
        /// <param name="hospital">The hospital to get the hospital patient record from.</param>
        /// <param name="hospitalPatient">Out. The hospital record for this patient.</param>
        /// <param name="patientMaster">Out. The patient master record for this patient.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        public HipsResponse GetPatient(RegisteredEnterprisePatient registeredEnterprisePatient, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            HipsResponse response;
            hospitalPatient = null;
            patientMaster = null;

            // 1. Use the “GetByStatePatientId” method of the “PatientMasterDataAccess” object to look up the patient master.
            // a. If the response is not “OK”, the patient does not exist or there was a database error. Set the output parameters to null and return the response.
            response = PatientMasterDataAccess.GetByStatePatientId(registeredEnterprisePatient.StatePatientId, hospital.HealthProviderOrganisationNetworkId, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK)
            {
                hospitalPatient = null;
                patientMaster = null;

                return response;
            }

            // 2. Use the “Get” method of the “HospitalPatientDataAccess” object to look up the existing hospital patient with the specified MRN.
            response = HospitalPatientDataAccess.Get(hospital.HospitalId.Value, registeredEnterprisePatient.Mrn, out hospitalPatient);

            // a. If the response is “OK” and the hospital patient’s PatientMasterId is the same as the patient master’s PatientMasterId, the lookup is successful so return the response.
            // b. If the response is “OK” and the hospital patient’s PatientMasterId is different to the patient master’s PatientMasterId, it means the hospital patient is on the wrong patient master.
            if (response.Status == HipsResponseIndicator.OK)
            {
                if (patientMaster.PatientMasterId == hospitalPatient.PatientMasterId)
                {
                    return response;
                }
                else
                {
                    // i. Format an error message “Patient with MRN {0} is registered in hospital {1}, but has ID {2} different to the given ID {3}”, using the MRN, hospital description, and the two StatePatientIds as parameters.
                    // ii. Write the error message to the system error log.
                    // iii. Set the response’s Status to “IncorrectStatePatientId” and set the HipsErrorMessage to the error message.
                    PatientMaster wrongPatientMaster;
                    PatientMasterDataAccess.Get(hospitalPatient.PatientMasterId, hospital.HealthProviderOrganisationNetworkId, out wrongPatientMaster);
                    string errorMessage = string.Format(EventLoggerStrings.RegisteredEnterprisePatientHasMultipleIds, registeredEnterprisePatient.Mrn, hospital.Description, wrongPatientMaster.StatePatientId, registeredEnterprisePatient.StatePatientId);
                    EventLogger.WriteLog(errorMessage, null, User, LogMessage.HIPS_MESSAGE_172);
                    response.Status = HipsResponseIndicator.IncorrectStatePatientId;
                    response.HipsErrorMessage = errorMessage;
                }
            }
            else
            {
                // 3. If the response is not “OK”, the specified MRN doesn’t yet exist. Use the “GetActive” method of the “HospitalPatientDataAccess” object to look up the current MRN of the patient in the hospital.
                //    a. If the response was “OK”, it means the patient is currently registered using a different MRN in this hospital.
                //      i. Format an error message “Patient with ID {0} is already registered in hospital {1}, but has an MRN {2} different to the given MRN {3}”, using the StatePatientId, hospital description and the two MRNs as parameters.
                //      ii. Write the error message to the system error log.
                //      iii. Set the response’s Status to “IncorrectMrn” and set the HipsErrorMessage to the error message.

                response = HospitalPatientDataAccess.GetActive((int)hospital.HospitalId.Value, (int)patientMaster.PatientMasterId, out hospitalPatient);
                if (response.Status == HipsResponseIndicator.OK)
                {
                    string errorMessage = string.Format(EventLoggerStrings.RegisteredEnterprisePatientAlreadyResigiteredAtHospital, registeredEnterprisePatient.StatePatientId, hospital.Description, hospitalPatient.Mrn, registeredEnterprisePatient.Mrn);
                    EventLogger.WriteLog(errorMessage, null, User, LogMessage.HIPS_MESSAGE_173);
                    response.Status = HipsResponseIndicator.IncorrectMrn;
                    response.HipsErrorMessage = errorMessage;
                }
                else if (response.Status == HipsResponseIndicator.InvalidPatient)
                {
                    // b. If the response was “InvalidPatient”, this means we can create the HospitalPatient object:
                    //  i. Create a new “HospitalPatient” object.
                    //  ii. Set the PatientMasterId from the “patientMaster" that was found.
                    //  iii. Set the HospitalId from the “hospital” parameter.
                    //  iv. Set the Mrn from the “patientIdentifier” parameter.
                    //  v. Invoke the “Insert” method of the “HospitalPatientDataAccess” object.
                    //  vi. If the insert fails, set the response Status to “DatabaseError”, set the response HipsErrorMessage to “Database access failed when creating patient record.” and write this message to the system error log.

                    hospitalPatient = new HospitalPatient();
                    hospitalPatient.PatientMasterId = (int)patientMaster.PatientMasterId;
                    hospitalPatient.HospitalId = (int)hospital.HospitalId.Value;
                    hospitalPatient.Mrn = registeredEnterprisePatient.Mrn;

                    bool isSuccess = HospitalPatientDataAccess.Insert(hospitalPatient);
                    if (!isSuccess)
                    {
                        response.Status = HipsResponseIndicator.DatabaseError;
                        response.HipsErrorMessage = EventLoggerStrings.RegisteredEnterprisePatientDatabaseFailure;
                        EventLogger.WriteLog(EventLoggerStrings.RegisteredEnterprisePatientDatabaseFailure, null, User, LogMessage.HIPS_MESSAGE_174);
                    }
                    else
                    {
                        // vii. Invoke “ValidateLocalIhiInformation” to determine if the patient has an IHI with unresolved alerts. If the response is neither OK nor InvalidIhi, then return.
                        ValidateLocalIhiInformation(patientMaster, response);
                        if (response.Status != HipsResponseIndicator.OK && response.Status != HipsResponseIndicator.InvalidIhi)
                        {
                            return response;
                        }

                        // viii. Invoke the “GetValidatedIhi” method of the “PatientIhiValidation” business logic. This will revalidate the IHI with the HI service if it was not validated within the configured period, 
                        // and it will set the IHI status to service unavailable if the IHI lookup fails. If the response is not OK then return.
                        PatientIhiValidation patientIhiValidation = new PatientIhiValidation();
                        IhiSearchResponse ihiSearchResponse = patientIhiValidation.GetValidatedIhi(registeredEnterprisePatient, hospital, User, patientMaster);

                        if (ihiSearchResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                        {
                            response.Status = ihiSearchResponse.HipsResponse.Status;
                            response.HipsErrorMessage = ihiSearchResponse.HipsResponse.HipsErrorMessage;
                            return response;
                        }

                        // ix. Invoke the “PcehrExists” method of the “DoesPcehrExist” business logic.
                        // This will obtain and store the PCEHR status for the new hospital’s HPI-O,
                        // and it will set the IHI status to service unavailable if the PCEHR lookup fails.
                        DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
                        DoesPcehrExistResponse doesPcehrExistResponse = doesPcehrExist.PcehrExists(registeredEnterprisePatient, hospital, patientMaster, User);
                        if (doesPcehrExistResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                        {
                            response.Status = doesPcehrExistResponse.HipsResponse.Status;
                            response.HipsErrorMessage = doesPcehrExistResponse.HipsResponse.HipsErrorMessage;
                            return response;
                        }
                    }
                }
            }

            return response;
        }      

        /// <summary>
        /// Get list of all non cancelled episodes for a patient
        /// </summary>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <param name="admissionDateStart">The admission date start for the query</param>
        /// <param name="admissionDateEnd">The admission date end for the query</param>
        /// <returns>
        /// A list of episodes for the patient with the optional admission start and end date, or null if none matched
        /// </returns>
        /// <exception cref="System.Exception">If a database error occurs while updating or inserting a stub episode</exception>
        public List<Episode> ListEpisodesWithoutCancelled(HospitalPatient hospitalPatient, DateTime? admissionDateStart, DateTime? admissionDateEnd)
        {
            List<Episode> episodes = EpisodeDataAccess.GetAll(hospitalPatient.PatientId, null, true);
            return episodes;
        }

        /// <summary>
        /// Reads, creates or updates a patient master and hospital patient for the given validated IHI information.
        /// </summary>
        /// <param name="validatedIhi">The IHI along with its last validated date and demographic information for revalidation</param>
        /// <param name="hospital">The hospital to read the hospital patient record from.</param>
        /// <param name="hospitalPatient">Out. The hospital record for this patient.</param>
        /// <param name="patientMaster">Out. The patient master record for this patient.</param>
        /// <returns>Information about success or failure</returns>
        public HipsResponse ReadCreateOrUpdatePatient(ValidatedIhi validatedIhi, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            hospitalPatient = null;
            patientMaster = null;
            SqlTransaction transaction = null;
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                // Validate the ihi information
                Sex sex;
                response = ValidateValidatedIhiInformation(validatedIhi, out sex);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    return response;
                }

                // Populate the patient master from the ihi provided
                PatientMasterDataAccess.GetByIhi(validatedIhi.Ihi, hospital.HealthProviderOrganisationNetworkId, out patientMaster);

                using (SqlCommand command = base.Command)
                {
                    transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;

                    // Insert the patientmaster if it does not already exist
                    if (!patientMaster.PatientMasterId.HasValue)
                    {
                        patientMaster.DateOfBirth = validatedIhi.DateOfBirth;
                        patientMaster.CurrentSexId = sex.SexId.Value;
                        if (!PatientMasterDataAccess.Insert(patientMaster, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, PatientMasterDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    // Add in all IHI info to new or gathered patientmaster item
                    patientMaster.Ihi = validatedIhi.Ihi;
                    patientMaster.IhiStatusId = (int)validatedIhi.IhiStatus;
                    patientMaster.IhiRecordStatusId = (int)validatedIhi.IhiRecordStatus;
                    patientMaster.IhiLastValidated = validatedIhi.IhiLastValidated;
                    patientMaster.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;

                    // If the DOB has changed then we need to create a flag so that we know to process an update
                    bool patientMasterDOBUpdated = (patientMaster.DateOfBirth != validatedIhi.DateOfBirth);
                    if (!Settings.Instance.RegisteredDateOfBirthEnabled)
                    {
                        patientMaster.DateOfBirth = validatedIhi.DateOfBirth;
                    }

                    patientMaster.CurrentSexId = sex.SexId.Value;
                    bool patientMasterNameUpdated = patientMaster.SetNewCurrentName(null, validatedIhi.GivenName, validatedIhi.FamilyName, null);

                    // Get the ihi information for the patient
                    PatientMasterIhi patientMasterIhi;
                    IhiDataAccess.Get(patientMaster.PatientMasterId.Value, hospital.HealthProviderOrganisationNetworkId, out patientMasterIhi);

                    // If any of the values for the PatientMasterIhi object have been changed then update it
                    if ((patientMasterIhi.Ihi != validatedIhi.Ihi) ||
                            (patientMasterIhi.DateLastValidated != validatedIhi.IhiLastValidated) ||
                            (patientMasterIhi.IhiStatusId != (int)validatedIhi.IhiStatus) ||
                            (patientMasterIhi.IhiRecordStatusId != (int)validatedIhi.IhiRecordStatus) ||
                            (patientMasterIhi.RegisteredFamilyName != validatedIhi.FamilyName) ||
                            (patientMasterIhi.RegisteredGivenName != validatedIhi.GivenName) ||
                            (patientMasterIhi.RegisteredSexId != sex.SexId.Value) ||
                            (patientMasterDOBUpdated))
                    {
                        // Populate with the passed in values ready for insert or update
                        patientMasterIhi.PatientMasterId = patientMaster.PatientMasterId.Value;
                        patientMasterIhi.Ihi = validatedIhi.Ihi;
                        patientMasterIhi.DateLastValidated = validatedIhi.IhiLastValidated;
                        patientMasterIhi.IhiStatusId = (int)validatedIhi.IhiStatus;
                        patientMasterIhi.IhiRecordStatusId = (int)validatedIhi.IhiRecordStatus;
                        patientMasterIhi.RegisteredFamilyName = validatedIhi.FamilyName;
                        patientMasterIhi.RegisteredGivenName = validatedIhi.GivenName;
                        patientMasterIhi.RegisteredSexId = sex.SexId.Value;
                        patientMasterIhi.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;
                        patientMasterIhi.DateOfBirth = validatedIhi.DateOfBirth;

                        if (Settings.Instance.RegisteredDateOfBirthEnabled)
                        {
                            patientMasterIhi.RegisteredDateOfBirth = validatedIhi.DateOfBirth;
                        }

                        // Insert or Update the patient master ihi
                        if (!IhiDataAccess.Update(patientMasterIhi, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, IhiDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    // If any of the PatientMaster object has changed then update it
                    if ((patientMaster.Ihi != validatedIhi.Ihi) ||
                            (patientMaster.IhiStatusId != (int)validatedIhi.IhiStatus) ||
                            (patientMaster.IhiRecordStatusId != (int)validatedIhi.IhiRecordStatus) ||
                            (patientMaster.IhiLastValidated != validatedIhi.IhiLastValidated) ||
                            (patientMasterDOBUpdated) ||
                            (patientMaster.CurrentSexId != sex.SexId.Value) ||
                            (patientMasterNameUpdated)
                        )
                    {
                        // Update the patient master with added information or changes as passed through
                        if (!PatientMasterDataAccess.Update(patientMaster, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, PatientMasterDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    // Get the active hospital patient and if it does not exists then insert
                    HospitalPatientDataAccess.GetActive(hospital.HospitalId.Value, patientMaster.PatientMasterId.Value, out hospitalPatient);
                    if (hospitalPatient == null || !hospitalPatient.PatientId.HasValue)
                    {
                        // This jurisdiction doesn't want to provide hospital MRNs so we will generate a dummy one.
                        hospitalPatient = new HospitalPatient();
                        hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;
                        hospitalPatient.HospitalId = hospital.HospitalId.Value;
                        hospitalPatient.Mrn = System.Guid.NewGuid().ToString().Substring(0, 20);  
                        if (!HospitalPatientDataAccess.Insert(hospitalPatient, transaction))
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, HospitalPatientDataAccess.GetType().FullName);
                            throw new PatientStubException(message);
                        }
                    }

                    transaction.Commit();
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                if (transaction != null)
                {
                    transaction.Rollback();
                    if (transaction.Connection != null)
                    {
                        transaction.Connection.Close();
                    }
                }
                EventLogger.WriteLog(ResponseStrings.UnableToInsertPatientStub, ex, User, LogMessage.HIPS_MESSAGE_066);
                response.Status = HipsResponseIndicator.DatabaseError;
                response.HipsErrorMessage = ResponseStrings.UnableToInsertPatientStub;
            }
            return response;
        }

        /// <summary>
        /// Validates the locally-stored IHI information for the patient before access to the PCEHR.
        /// This does not go to the Medicare HI Service. It checks whether an IHI is assigned to the
        /// patient, that it is 16 digits, that the record status is Verified, that the status is
        /// Active or Deceased, and that there are no unresolved alerts.
        /// </summary>
        /// <param name="patientMaster">The local patient demographics</param>
        /// <param name="response">Stores information about validation success or failure</param>
        public void ValidateLocalIhiInformation(PatientMaster patientMaster, HipsResponse response)
        {
            // Need to check status first because outstanding alerts should trump invalid IHI.
            // If the alert statuses are present, we return immediately.
            switch ((IhiStatus)patientMaster.IhiStatusId)
            {
                case IhiStatus.DuplicateIhi:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusDuplicateIhi;
                    return;

                case IhiStatus.DuplicatePatient:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusDuplicatePatient;
                    return;

                case IhiStatus.MergeConflict:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusMergeConflict;
                    return;

                case IhiStatus.MedicareDvaChangeMismatch:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusMedicareDvaChangeMismatch;
                    return;

                case IhiStatus.DemographicMismatch:
                    response.Status = HipsResponseIndicator.UnresolvedIhiAlert;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusDemographicMismatch;
                    return;

                case IhiStatus.Active:
                    response.Status = HipsResponseIndicator.OK; // OK
                    break;

                case IhiStatus.Deceased:
                    response.Status = HipsResponseIndicator.OK; // Can still access PCEHR
                    break;

                case IhiStatus.Expired:
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusExpired;
                    break;

                case IhiStatus.Resolved:

                    // When an IHI is resolved, the new active one is stored for
                    // the patient, and the old resolved one is stored in the
                    // history. There should never be an IHI with this status on
                    // a patient.
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusResolved;
                    break;

                case IhiStatus.Retired:

                    // When an IHI is Retired, it stays attached
                    // to the patient, marked with this status.
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusRetired;
                    return;

                case IhiStatus.ServiceUnavailable:
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusServiceUnavailable;
                    break;

                case IhiStatus.Unknown:
                    response.Status = HipsResponseIndicator.InvalidIhi;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusUnknown;
                    break;

                default:
                    response.Status = HipsResponseIndicator.SystemError;
                    response.HipsErrorMessage = ResponseStrings.IhiStatusNotHandled;
                    break;
            }
            if (string.IsNullOrEmpty(patientMaster.Ihi))
            {
                response.Status = HipsResponseIndicator.InvalidIhi;
                response.HipsErrorMessage = ResponseStrings.IhiNotSupplied;
                return;
            }
            if (patientMaster.Ihi.Length != 16)
            {
                response.Status = HipsResponseIndicator.InvalidIhi;
                response.HipsErrorMessage = ResponseStrings.IhiLengthNot16;
                return;
            }
            if (patientMaster.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                response.Status = HipsResponseIndicator.InvalidIhi;
                response.HipsErrorMessage = ResponseStrings.IhiRecordStatusNotVerified;
                return;
            }
        }

        /// <summary>
        /// Compares the registered demographics and IHI of a local patient with the incoming patient information.
        /// The IHI, family name, date of birth and sex are all mandatory for the match to be true.
        /// If the registered given name is one of the incoming given names, this is a sufficient match.
        /// If the incoming message has no given names then the local patient must also have no given names.
        /// (In future, HIPS could find the IHI for a patient with only one name, stored in family name.)
        /// </summary>
        /// <param name="incomingConsumer">Details of the consumer who is the subject of care in the received eHealth message.</param>
        /// <param name="ihi">The IHI of the local patient record.</param>
        /// <param name="registeredFamilyName">The registered family name of the local patient record.</param>
        /// <param name="registeredGivenName">The registered given name of the local patient record.</param>
        /// <param name="registeredSex">The registered sex of the local patient record.</param>
        /// <param name="registeredDateOfBirth">The registered date of birth of the local patient record.</param>
        /// <returns>Whether the local patient demographic data matches the incoming demographic data.</returns>
        internal bool IsReceivedPatientMatch(ParticipatingConsumer incomingConsumer, string ihi, string registeredFamilyName, string registeredGivenName, SexEnumerator registeredSex, DateTime registeredDateOfBirth)
        {
            bool familyNameMatch = registeredFamilyName != null && incomingConsumer.FamilyName != null && registeredFamilyName.ToUpper() == incomingConsumer.FamilyName.ToUpper();
            bool dobMatch = incomingConsumer.DateOfBirth.HasValue && registeredDateOfBirth.Date == incomingConsumer.DateOfBirth.Value.Date;
            bool sexMatch = registeredSex == incomingConsumer.Sex;
            bool ihiMatch = !string.IsNullOrEmpty(ihi) && !string.IsNullOrEmpty(incomingConsumer.Ihi) && ihi == incomingConsumer.Ihi;

            string incomingGivenNames = incomingConsumer.GivenNames == null ? string.Empty : incomingConsumer.GivenNames.ToUpper();
            string incomingGivenNamesPattern = string.Format(" {0} ", incomingGivenNames);
            string patientRegisteredGivenName = registeredGivenName == null ? string.Empty : registeredGivenName.ToUpper();
            string patientRegisteredGivenNamePattern = string.Format(" {0} ", patientRegisteredGivenName);
            bool givenNamesMatch = incomingGivenNamesPattern.Contains(patientRegisteredGivenNamePattern);

            return familyNameMatch && givenNamesMatch && sexMatch && dobMatch && ihiMatch;
        }

        /// <summary>
        /// Obtains a set of local patient records that best match the incoming IHI and incoming demographic
        /// data when receiving an e-health message. A list is returned because the consumer may have separate
        /// patient master records in each hospital under the given HPI-O.
        /// The meaning of each response and the actions required are:
        /// OK - One or more records were found whose local IHI and demographics match the incoming message.
        ///      Validate the local records if not validated within the configured validity period (maximum 24 hours).
        /// DemographicMismatchWarning - One or more records were found, but there is a mismatch between the local
        ///      records and the incoming message. Validate the local records, and raise an alert if the mismatch persists.
        /// InvalidIhi - One or more records were found where the local IHI is absent.
        ///      Attempt to obtain the IHI. Raise an alert if the IHI found does not match the incoming IHI.
        /// DatabaseError - There was an error accessing the database. Log the error.
        /// </summary>
        /// <param name="receiverHpio">Identifier of the healthcare provider organisation that received the incoming message.</param>
        /// <param name="incomingConsumer">Details of the consumer who is the subject of care in the received eHealth message.</param>
        /// <param name="bestCandidates">Out parameter: set of PatientMaster records that best match the incoming data.</param>
        /// <returns>HIPS Response, as documented above.</returns>
        public HipsResponse FindLocalPatientsForReceivedMessage(string receiverHpio, ParticipatingConsumer incomingConsumer, out List<PatientMaster> bestCandidates)
            {
            // Set the return patient masters to empty list.
            bestCandidates = new List<PatientMaster>();
            var candidatePatientMastersByIhi = new List<PatientMaster>();
            var candidatePatientMastersByDemographics = new List<PatientMaster>();
            HipsResponse response;

            // Get the local patients by IHI.
            if (!string.IsNullOrEmpty(incomingConsumer.Ihi))
            {
                response = PatientMasterDataAccess.GetByIhiForReceivedMessage(receiverHpio, incomingConsumer.Ihi, out candidatePatientMastersByIhi);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    return response;
            }
            }
            
            // Examine the patient records found by IHI.
            foreach (var patient in candidatePatientMastersByIhi)
            {
                if (this.IsReceivedPatientMatch(incomingConsumer, patient.Ihi, patient.RegisteredFamilyName, patient.RegisteredGivenName, (SexEnumerator)patient.RegisteredSexId, patient.RegisteredDateOfBirth))
                {
                    // The local patient matches the incoming demographic data.
                    bestCandidates.Add(patient);
                }
            }

            if (bestCandidates.Count != 0)
                {
                // We found some local patient records whose IHI and demographics both match perfectly.
                    return new HipsResponse(HipsResponseIndicator.OK);
                }

            if (candidatePatientMastersByIhi.Count != 0)
                {
                // We found some local patients whose IHI matched but demographics did not match.
                // Requirement 023944 says the software must try to validate the local IHI and local demographics against the HI Service in this case.
                // This could generate an alert on the local record, separate to the alert raised because the incoming demographic data did not match.
                bestCandidates = candidatePatientMastersByIhi;
                    return new HipsResponse(
                        HipsResponseIndicator.DemographicMismatchWarning,
                    "Local patient found by IHI, but incoming demographic information does not match.");
            }

            // If patient not found by IHI, search for patient by demographics.
            response = PatientMasterDataAccess.GetByDemographicsForReceivedMessage(receiverHpio, incomingConsumer.FamilyName, incomingConsumer.GivenNames, incomingConsumer.DateOfBirth.Value, (int)incomingConsumer.Sex, null, null, null, out candidatePatientMastersByDemographics);
            if (response.Status != HipsResponseIndicator.OK)
                {
                return response;
            }

            var candidatePatientsWithoutIhi = candidatePatientMastersByDemographics.Where(c => string.IsNullOrEmpty(c.Ihi));
            if (candidatePatientsWithoutIhi.Count() != 0)
                    {
                // Incoming demographic data matches a local patient record and the local IHI is absent.
                // Requirement 023943 says the software shall try to obtain the IHI, using local patient demographics, from the HI Service and store this in the local patient record.
                // If the IHI retrieved from the HI Service does not match with the incoming IHI the system shall raise an alert.
                bestCandidates = candidatePatientsWithoutIhi.ToList();
                        return new HipsResponse(
                            HipsResponseIndicator.InvalidIhi,
                            "Local patient found by demographics, but has no IHI assigned in local system.");
                    }

            if (candidatePatientMastersByDemographics.Count != 0)
                    {
                // The remaining patients found by demographics must have an IHI that differs from the incoming IHI.
                // Requirement 023944 says the software must try to validate the local IHI and local demographics against the HI Service in this case.
                // This could generate an alert on the local record, separate to the alert raised because the incoming IHI did not match.
                bestCandidates = candidatePatientMastersByDemographics;
                        return new HipsResponse(
                            HipsResponseIndicator.DemographicMismatchWarning,
                    "Local patient found by demographics, but incoming IHI does not match.");
                }

            // Neither the incoming demographic data nor the incoming IHI matched any local patient record.
                    return new HipsResponse(
                        HipsResponseIndicator.InvalidPatient,
                "Neither incoming IHI nor demographic information matched any local patient.");
        }

        /// <summary>
        /// Check if the HPIO exists on HIPS
        /// </summary>
        /// <param name="hpio">HPIO</param>
        public HipsResponse ValidateHpio(string hpio)
        {
            Hospital hpioHospital = HospitalDataAccess.GetAll().Where(i => i.HpiO == hpio).FirstOrDefault();

            // Check if any Hospital matched
            if (hpioHospital == null)
            {
                return new HipsResponse(
                    HipsResponseIndicator.InvalidHospital,
                    string.Format("No facility found with the specified HPI-O {0}.", hpio));
            }

            return new HipsResponse(HipsResponseIndicator.OK);
        }

        /// <summary>
        /// Calls the HpiiSearch.HpiiValidation.
        /// If valid and healthProviderIndividual has been provided, will save the details to the HealthProviderIndividualHpii table
        /// </summary>
        /// <param name="query">Hpii Search query (includes Hpii number and Provider Family Name)</param>
        /// <param name="healthProviderIndividual">The HealthProviderIndividual record (can be null)</param>
        /// <param name="facility">The Hospital facility to get the hospitalIdentifier for</param>
        /// <returns>HpiiQueryResponse object. Will raise an HpiiErrorException if not found or any other error occurs.</returns>
        public HpiiQueryResponse ValidateHpii(HpiiIdentifierQuery query, HealthProviderIndividual healthProviderIndividual = null, Hospital facility = null)
        {
            HospitalIdentifier hospitalIdentifier = null;
            
            if (facility != null)
            {
                string hospitalCodeSystem = HL7.ConstantsResource.PatientAdministrationSystemHospitalCodeSystem;
                string hospitalCode = facility.GetCode(hospitalCodeSystem);
                
                hospitalIdentifier = new HospitalIdentifier(hospitalCode, hospitalCodeSystem);
            }
            
            var search = new HpiiSearch();
            var result = search.HpiiValidation(query, this.User, hospitalIdentifier);

            try
            {
                // Call the CheckHPiiStatus this will throw an exception for alerts or warnings
                CheckHpiiStatus(result, facility, healthProviderIndividual, query.HpiiNumber, query.FamilyName);
                // Check if this is a Resolved Hpii (the Hpii number will be different but returned as Active)
                if (query.HpiiNumber != result.HpiiNumber)
                    throw new HpiiInfoException(ValidHpiiResource.ResolvedActiveHpiiStatus);
                // All OK go ahead with save
                SaveHpii(healthProviderIndividual, facility, result);
            }
            catch (HpiiInfoException ex)
            {
                // A HPI-I warning has been raised, but we still can go ahead with save
                SaveHpii(healthProviderIndividual, facility, result);

                // Now set the Hips Response for the Hpii Warning to raise to the user
                result.HipsResponse.Status = HipsResponseIndicator.HpiiWarningRaised; 
                result.HipsResponse.HipsErrorMessage = ex.Message;
            }
            catch (HiServiceException ex)
            {
                // Set the Hips Response for HI Service Unavailable
                result.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                result.HipsResponse.HipsErrorMessage = ex.Message;
                
            }
            catch (HpiiErrorException ex)
            {
                // Set the Hips Response for the Hpii Warning
                result.HipsResponse.Status = HipsResponseIndicator.UnresolvedHpiiAlert; 
                result.HipsResponse.HipsErrorMessage = ex.Message;
            }            
            catch (Exception ex)
            {
                // Set the Hips Response for the Hpii Warning
                result.HipsResponse.Status = HipsResponseIndicator.SystemError;
                result.HipsResponse.HipsErrorMessage = ex.Message;
                result.HipsResponse.ResponseCode = ex.GetType().Name;
                result.HipsResponse.ResponseCodeDescription = ex.Message;
                result.HipsResponse.ResponseCodeDetails = ex.InnerException == null ? null : ex.InnerException.Message;
            }

            return result;
        }

        /// <summary>
        /// Obtains the first hospital using the PatientMasterId and HPIO
        /// </summary>
        /// <param name="patientMasterId">Patient Master Id</param>
        /// <param name="hpio">HPIO</param>
        /// <param name="hospital">Out Parameter: First Hospital Found</param>
        /// <param name="user">The user responsible for this action</param>
        /// <returns>HIPS Response</returns>
        public bool GetHospitalByPatientMasterIdAndHpio(int patientMasterId, string hpio, out Hospital hospital, UserDetails user)
        {
            bool result = false;

            PatientAccess dataAccessHelper = new PatientAccess(user);
            // Get Hospital lists
            List<Hospital> hospitals = dataAccessHelper.HospitalDataAccess.GetHospitalsUsingPatientMasterIdAndHpio(patientMasterId, hpio);

            // Check if any Hospital matched
            if (hospitals.Count > 0)
            {
                result = true;
            }

            hospital = hospitals.FirstOrDefault();

            return result;
        }

        /// <summary>
        /// Checks that the family and given names were provided and the sex code was a recognised code.
        /// Also checks that the IHI is 16 digits and the record status is Verified.
        /// </summary>
        /// <param name="validatedIhi">The supplied validated IHI information</param>
        /// <param name="sex">Outputs the parsed sex code</param>
        /// <returns>Whether the information passed validation</returns>
        private HipsResponse ValidateValidatedIhiInformation(ValidatedIhi validatedIhi, out Sex sex)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            sex = null;
            if (string.IsNullOrEmpty(validatedIhi.FamilyName) || string.IsNullOrEmpty(validatedIhi.GivenName))
            {
                response.Status = HipsResponseIndicator.InvalidPatient;
                response.HipsErrorMessage = ResponseStrings.NameNotSupplied;
                return response;
            }
            sex = ListSingleton.Instance.AllSexes.Where(a => a.SexId == (int)validatedIhi.Sex).FirstOrDefault();
            if (sex == null)
            {
                response.Status = HipsResponseIndicator.InvalidPatient;
                response.HipsErrorMessage = ResponseStrings.InvalidSex;
                return response;
            }
            return response;
        }

        

        

        /// <summary>
        /// This method checks if the message contains PCEHR disclosure information. 
        /// If it doesn’t then it will look-up the PCEHR Advertised status of the patient or if none exists will check the PCEHR Disclosure status within eHISC.
        /// </summary>
        /// <param name="order">The OrderGroup.</param>
        /// <param name="patientMaster">The PatientMaster.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="episodeCreated">Indicates whether a new episode was created</param>
        /// <returns>CheckPcehrExistsResponse which includes the Pcehr disclosure status</returns>
        public CheckPcehrExistsResponse CheckPcehrExists(List<OrderGroup> orders, PatientMaster patientMaster, Hospital hospital, bool episodeCreated)
        {
            CheckPcehrExistsResponse response = new CheckPcehrExistsResponse();
            response.HipsResponse = new HipsResponse(HipsResponseIndicator.OK);

            if (!HIPS.CommonBusinessLogic.User.PopulateAndValidateUser(hospital, this.User))
            {
                // Attempted access with incorrect user details should be brought to system administrator's attention.
                EventLogger.WriteLog(ResponseStrings.InvalidUserDetails, null, this.User, LogMessage.HIPS_MESSAGE_082);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ResponseStrings.MissingUserValues;
                return response;
            }

            // Check each OBR record for an entry in FillerField1. If this field contains an AUSEHR flag then use this as PCEHR disclosure status for the patient.
            foreach (var order in orders)
            {
                if (order.Observation.Any(i => i.ObservationsReportID.FillerField1.Contains("AUSEHR")))
                {
                    // If the field has “AUSEHR=Y” then treat the patient as having a disclosed PCEHR. 
                    if (order.Observation.Any(i => i.ObservationsReportID.FillerField1.Contains("AUSEHR=Y")))
                    {
                        response.PcehrDisclosed = true;
                        response.PCEHRAdvertised = true;
                    }
                    else if (order.Observation.Any(i => i.ObservationsReportID.FillerField1.Contains("AUSEHR=N")))
                    {
                        response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                        response.HipsResponse.HipsErrorMessage = ResponseStrings.AUSEHRSetToN;
                        return response;
                    }
                    else
                    {
                        response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                        response.HipsResponse.HipsErrorMessage = ResponseStrings.AUSEHRNotValid;
                        return response;
                    }
                }
                else
                {
                    // 2. If there is no entry in FillerField1 and the patient has an existing patient master with an IHI
                    PatientMaster currentPatient;
                    PatientMasterDataAccess.GetByIhi(patientMaster.Ihi, hospital.HealthProviderOrganisationNetworkId, out currentPatient);

                    if (currentPatient != null)
                    {
                        //    If a new episode was created then invoke PcehrExists on the DoesPcehrExist class to check the current PCEHR status of the patient.
                        if (episodeCreated)
                        {
                            var hospitalPatientDl = new HospitalPatientDl(User);
                            var hospitalPatient = hospitalPatientDl.GetAll(hospital.HospitalId, patientMaster.PatientMasterId.GetValueOrDefault()).FirstOrDefault();

                            // This part of code will create a patientIdentifierBase needed by the PcehrExists method
                            var hospitalCodeDl = new HospitalCodeDl();
                            var hospitalCode = hospitalCodeDl.GetAll(hospital.HospitalId).FirstOrDefault();

                            var mrn = new Mrn(hospitalPatient.Mrn, hospitalCode.Code, hospitalCode.CodeSystemCode);

                            DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
                            DoesPcehrExistResponse doesPcehrExistResponse = doesPcehrExist.PcehrExists(mrn, hospital, patientMaster, User);
                            if (doesPcehrExistResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                            {
                                response.HipsResponse = doesPcehrExistResponse.HipsResponse;
                                return response;
                            }
                        }

                        // Check the hips.HealthProviderOrganisationPatient table for the PCEHR advertised or disclosed status
                        var healthProviderOrganisationPatientDl = new HealthProviderOrganisationPatientDl(User);
                        HealthProviderOrganisationPatient hpoPatient;
                        healthProviderOrganisationPatientDl.Get(hospital.HpiO, currentPatient.PatientMasterId.GetValueOrDefault(), out hpoPatient);

                        response.PCEHRAdvertised = hpoPatient.PcehrAdvertised.HasValue ? hpoPatient.PcehrAdvertised : false;
                        response.PcehrDisclosed = hpoPatient.PcehrDisclosed;
                        
                        // If both PCEHR advertised and disclosed are false then do not attempt to upload the report and return an error message to the calling system.
                        if (response.PCEHRAdvertised == false && response.PcehrDisclosed == false)
                        {
                            response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                            response.HipsResponse.HipsErrorMessage = ResponseStrings.PCEHRNotDisclosed;
                            return response;
                        }
                    }
                }
            }

            return response;
        }

        /// <summary>
        /// Save the FillerOrderNumber
        /// </summary>
        /// <param name="transaction">The parent SQL transaction.</param>
        /// <param name="episode">Patient Episode.</param>
        /// <param name="orderIdentifier">PDI OrderIdentifier.</param>
        /// <param name="fillerOrderNumberValue">PDI FillerOrderNumber</param>
        /// <param name="cdaSetNumberId">The associated CDA set Number Id</param>
        public void SaveFillerOrderNumber(SqlTransaction transaction, string orderIdentifier, string fillerOrderNumberValue, int cdaSetNumberId)
        {
            using (HIPSTraceWriter trace = new HIPSTraceWriter())
            {
                trace.WriteTrace(string.Format("IN HIPS.CommonBuisnessLogic.PatientAccess.SaveFillerOrderNumber:: fillerOrderNumberValue: {0}, cdaSetNumberId: {1}", fillerOrderNumberValue, cdaSetNumberId), System.Diagnostics.TraceEventType.Information);

                FillerOrderNumberDl fillerOrderNumberDl = new FillerOrderNumberDl(this.User);

                try
                {
                    bool saved = false;
                    using (SqlCommand command = fillerOrderNumberDl.GetCommand())
                    {
                        // Save Filler Order Number
                        FillerOrderNumber fillerOrderNumber = new FillerOrderNumber();
                        fillerOrderNumber.OrderIdentifier = orderIdentifier;
                        fillerOrderNumber.fillerOrderNumber = fillerOrderNumberValue;
                        fillerOrderNumber.CdaSetNumberId = cdaSetNumberId;
                        trace.WriteTrace(string.Format(" - HIPS.CommonBuisnessLogic.PatientAccess.SaveFillerOrderNumber:: fillerOrderNumberValue: {0}, cdaSetNumberId: {1} - Call fillerOrderNumberDl.Save", fillerOrderNumberValue, cdaSetNumberId), System.Diagnostics.TraceEventType.Information);

                        saved = fillerOrderNumberDl.Save(fillerOrderNumber, transaction);
                        if (!saved)
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, "FillerOrderNumberInsert");
                            throw new Exception(message);
                        }
                    }
                }
                catch (Exception ex) // If an exception is caught here then throw it up to be handled
                {
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.PatientAccess.SaveFillerOrderNumber:: fillerOrderNumberValue: {0}, cdaSetNumberId: {1}. Exception details: {2}", fillerOrderNumberValue, cdaSetNumberId, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    
                    throw ex;
                }
                trace.WriteTrace(string.Format("OUT HIPS.CommonBuisnessLogic.PatientAccess.SaveFillerOrderNumber:: fillerOrderNumberValue: {0}, cdaSetNumberId: {1}", fillerOrderNumberValue, cdaSetNumberId), System.Diagnostics.TraceEventType.Information);
            }
        }

        /// <summary>
        /// Saves the CdaSetId based on teh Document Type and Episode Id
        /// </summary>
        /// <param name="transaction">The parent SQL transaction</param>
        /// <param name="metadata">MetaData object</param>
        /// <param name="episode">Patient Episode</param>
        /// <param name="docType">the document Type</param>
        /// <param name="docFormatCode">The document Format</param>
        /// <param name="setNumber">Out. The newly created CDA Set object</param>
        public void SaveCdaSetNumber(SqlTransaction transaction, CdaHeaderMetadata metadata, Episode episode, DocumentType docType, string docFormatCode, out CdaSetNumber setNumber)
        {
            using (HIPSTraceWriter trace = new HIPSTraceWriter())
            {
                trace.WriteTrace(string.Format("IN HIPS.CommonBuisnessLogic.PatientAccess.SaveCdaSetNumber:: episodeId: {0}", episode.EpisodeId.Value), System.Diagnostics.TraceEventType.Information);

                CdaSetNumberDl setNumberDl = new CdaSetNumberDl(this.User);
                DocumentFormat documentFormat = ListSingleton.Instance.AllDocumentFormats.Single(a => a.Code == docFormatCode);

                try
                {
                    trace.WriteTrace(string.Format(" - HIPS.CommonBuisnessLogic.PatientAccess.SaveCdaSetNumber:: episodeId: {0} - Call GetByEpisodeAndDocumentType", episode.EpisodeId.Value), System.Diagnostics.TraceEventType.Information);

                    setNumber = setNumberDl.GetByEpisodeAndDocumentType(episode.EpisodeId.Value, docType.DocumentTypeId.Value, transaction);
                    setNumber.EpisodeId = episode.EpisodeId.Value;
                    setNumber.DocumentTypeId = docType.DocumentTypeId.Value;
                    setNumber.DocumentFormatId = documentFormat.DocumentFormatId.Value;
                    setNumber.ModeOfSeparationId = (int)ModeOfSeparation.None;
                    setNumber.AdmissionDateTime = metadata.AdmissionDateTime;
                    setNumber.DischargeDateTime = metadata.DischargeDateTime;

                    bool saved = false;
                    using (SqlCommand command = setNumberDl.GetCommand())
                    {
                        // Save CdaSetNumber
                        trace.WriteTrace(string.Format(" - HIPS.CommonBuisnessLogic.PatientAccess.SaveCdaSetNumber:: episodeId: {0} - Call setNumberDl.Save", episode.EpisodeId.Value), System.Diagnostics.TraceEventType.Information);

                        saved = setNumberDl.Save(setNumber, transaction);
                        if (!saved)
                        {
                            string message = string.Format(ResponseStrings.DatabaseError, "CdaSetNumberInsert");
                            throw new Exception(message);
                        }

                    }
                }
                catch (Exception ex) // If an exception is caught here then throw it up to be handled
                {
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.PatientAccess.SaveCdaSetNumber:: episodeId: {0}. Exception Details: {3}", episode.EpisodeId.Value, ex.ToString()), System.Diagnostics.TraceEventType.Error);

                    throw ex;
                }
                trace.WriteTrace(string.Format("OUT HIPS.CommonBuisnessLogic.PatientAccess.SaveCdaSetNumber:: episodeId: {0}", episode.EpisodeId.Value), System.Diagnostics.TraceEventType.Information);
            }
        }

        #endregion Methods

        #region Private

        /// <summary>
        /// Creates the patient from demographic data.
        /// </summary>
        /// <param name="demographic">The demographic.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="hospitalPatient">Out. The newly created hospital record for this patient.</param>
        /// <param name="patientMaster">Out. The newly created patient master record for this patient.</param>
        public void CreatePatient(Demographic demographic, Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster)
        {
            using (SqlCommand command = base.Command)
            {
                try
                {
                    SqlTransaction transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;

                    hospitalPatient = new HospitalPatient();
                    patientMaster = new PatientMaster();

                    // Store the key demographic information into the patient master record
                    patientMaster.SetNewCurrentName(-1, demographic.GivenName, demographic.FamilyName, -1);
                    patientMaster.MedicareNumber = demographic.MedicareNumber;
                    patientMaster.MedicareIrn = demographic.MedicareIrn;
                    patientMaster.IsMedicareNumberValid = Medicare.Validate(patientMaster.MedicareNumber);
                    patientMaster.DvaNumber = demographic.DvaNumber;
                    patientMaster.DateOfBirth = demographic.DateOfBirth;
                    patientMaster.RegisteredDateOfBirth = demographic.DateOfBirth;
                    patientMaster.CurrentSexId = (int)demographic.Sex;

                    if (!PatientMasterDataAccess.Insert(patientMaster, transaction))
                    {
                        // Failed
                    }

                    // Create a new hospital patient record with a generated MRN
                    hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;
                    hospitalPatient.HospitalId = hospital.Id.Value;
                    hospitalPatient.Mrn = ("AR" + System.Guid.NewGuid().ToString()).Substring(0, 20);

                    if (!HospitalPatientDataAccess.Insert(hospitalPatient, transaction))
                    {
                        // Failed
                    }

                    transaction.Commit();
                }
                finally
                {
                    command.Connection.Close();
                }
            }
        }

        /// <summary>
        /// Look ups a patient by the demographics within a hospital.
        /// </summary>
        /// <param name="familyName">Name of the family.</param>
        /// <param name="givenNames">The given names.</param>
        /// <param name="dateOfBirth">The date of birth.</param>
        /// <param name="sexId">The sex identifier.</param>
        /// <param name="medicareCardNumber">The Medicare card number.</param>
        /// <param name="medicareIrn">The Medicare IRN.</param>
        /// <param name="dvaFileNumber">The DVA file number.</param>
        /// <param name="patientMaster">The patient master.</param>
        private void LookupByDemographicsWithinHospital(string familyName, string givenNames, DateTime? dateOfBirth, int? sexId, string medicareCardNumber, string medicareIrn, string dvaFileNumber, Hospital hospital, out PatientMaster patientMaster)
        {
            patientMaster = null;
            bool? medicareIsValid = Medicare.Validate(medicareCardNumber);
            if (!medicareIsValid.HasValue || !medicareIsValid.Value)
            {
                medicareCardNumber = null;
                medicareIrn = null;
            }
            if (dateOfBirth.HasValue && sexId.HasValue && (medicareCardNumber != null || dvaFileNumber != null))
            {
                PatientMasterDataAccess.GetByDemographicsWithinHospital(familyName, givenNames, dateOfBirth.Value, sexId.Value, medicareCardNumber, medicareIrn, dvaFileNumber, hospital, out patientMaster);
            }
        }

        /// <summary>
        /// Creates the patient from Patient Master and State Identifier as MRN
        /// Look for the first hospital that matched the HPIO
        /// </summary>
        /// <param name="patientMasterId">Patient Master Id.</param>
        /// <param name="stateIdentifier">State Identifier.</param>
        /// <param name="hpio">hpio.</param>
        /// <param name="hospital">The hospital class to be use in generating Patient Master Id.</param>
        public void CreatePatientFromStateIdentifier(int patientMasterId, string stateIdentifier, string hpio, out Hospital hospital)
        {
            using (SqlCommand command = base.Command)
            {
                try
                {
                    SqlTransaction transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;

                    HospitalPatient hospitalPatient = new HospitalPatient();

                    // Create a new hospital patient record with a generated MRN
                    hospitalPatient.PatientMasterId = patientMasterId;
                    // Get First Hospital using HPIO
                    hospital = HospitalDataAccess.GetAll().Where(i => i.HpiO == hpio).FirstOrDefault();
                    hospitalPatient.HospitalId = (int)hospital.HospitalId;
                    // Use StateIdentifier as MRN
                    hospitalPatient.Mrn = stateIdentifier;

                    if (!HospitalPatientDataAccess.Insert(hospitalPatient, transaction))
                    {
                        // TODO: Failed
                    }

                    transaction.Commit();
                }
                finally
                {
                    command.Connection.Close();
                }
            }
        }

        /// <summary>
        /// Checks the returned HpiiValidation Query results and the HpiiStatus to determine if an Alert Error of Validation Warning should be raised
        /// </summary>
        /// <param name="hpiiValidateResponse">The Hpii Query Response to check</param>
        /// <param name="hospital">The Hospital acessing the Hpii Query</param>
        /// <param name="healthProviderIndividual">The HealthProviderIndividual (if one provided)</param>
        /// <param name="hpiiNumber">The Hpii Number initially validated</param>
        /// <param name="providerFamilyName">The Provider's Family Name</param>
        private void CheckHpiiStatus(HpiiQueryResponse hpiiValidateResponse, Hospital hospital, HealthProviderIndividual healthProviderIndividual, string hpiiNumber, string providerFamilyName)
        {
            // Check the Response Codes for any errors

            // If it has not been found (WSE0035) or an error has been returned then raise HpiiErrorException
            if (hpiiValidateResponse.HipsResponse.ResponseCode == "WSE0035")
                throw new HpiiErrorException(ValidHpiiResource.HpiiNotFound);

            // Check if any other error or warning has been returned from the HIService 
            if (hpiiValidateResponse.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
            {
                if (hpiiValidateResponse.ServiceMessagesType != null) // If the HI Service has returned errors or warnings then raise a HpiiErrorException 
                {
                    List<string> serviceErrorMsgs = hpiiValidateResponse.ServiceMessagesType.serviceMessage
                                                          .Where(s => s.severity != SeverityType.Informational)
                                                          .Select(s => s.reason).ToList();
                    if (serviceErrorMsgs.Count > 0)
                    {
                        string serviceErrorMsg = string.Format(ValidHpiiResource.HiServiceHpiiError, string.Join(" ", serviceErrorMsgs));
                        throw new HpiiErrorException(serviceErrorMsg);
                    }
                }
                // If the HPI-I cannot be validated because the HI service is unavailable, HIPS will
                // roll back the queue transaction, and the operation will be tried again.
                throw new HiServiceException(hpiiValidateResponse.HipsResponse);
            }

            // Check the HpiiStatus
            switch (hpiiValidateResponse.HpiiStatus)
            {
                case HpiiStatus.Active:
                    // First check if Active but different HPI-I returned, then Resolved
                    if (hpiiValidateResponse.HpiiNumber != hpiiNumber)
                    {
                        // Validate new HPI-I with HI Service
                        HpiiIdentifierQuery hpiiQuery = new HpiiIdentifierQuery() { HpiiNumber = hpiiValidateResponse.HpiiNumber, FamilyName = providerFamilyName };
                        hpiiValidateResponse = ValidateHpii(hpiiQuery, healthProviderIndividual, hospital);
                        // If Not Active - then throw an HpiiErrorException
                        if (hpiiValidateResponse.HpiiStatus != HpiiStatus.Active)
                            throw new HpiiErrorException(ValidHpiiResource.ResolvedActiveHpiiStatus);
                    }
                    // Active - Success
                    break;
                case HpiiStatus.Retired:
                    // Retired - Raise HPI-I validation warning
                    throw new HpiiInfoException(ValidHpiiResource.RetiredHpiiStatus);
                case HpiiStatus.Deactivated:
                    // Deactivated - Raise HPI-I validation warning
                    throw new HpiiInfoException(ValidHpiiResource.DeactivatedHpiiStatus);
                default:
                    // No match - Raise HPI-I validation alert 
                    throw new HpiiErrorException(ValidHpiiResource.HpiiStatusUndefined);
            }
        }
        
        /// <summary>
        /// Saves an Health Provider Individual record and their corresponding HPI-I record. Updates the HPI-I History table for Auditing HPI-I changes
        /// </summary>
        /// <param name="healthProviderIndividual">The HealthProviderIndividual Record</param>
        /// <param name="facility">The Health Facility (Hospital)</param>
        /// <param name="hpiiValidationResult">The Hpii validation result that includes HpiiNumber and HpiiStatus</param>
        private void SaveHpii(HealthProviderIndividual healthProviderIndividual, Hospital facility, HpiiQueryResponse hpiiValidationResult)
        {
            // If a health provider individual was specified, then the validated HPI-I needs to be written to the database.
            if (healthProviderIndividual != null)
            {
                using (transaction)
                {
                    HealthProviderIndividualDl healthProviderIndividualDataAccess = new HealthProviderIndividualDl(this.User);
                    // Check if the existing health provider individual has a HPI-I Entry.
                    // If it does we need to update the HPI-I history table before writing the new HPI-I value.
                    HealthProviderIndividualHpiiDl hpiiDataAccess = new HealthProviderIndividualHpiiDl(this.User);
                    HealthProviderIndividualHpii existingHpii = hpiiDataAccess.Get(hpiiValidationResult.HpiiNumber, facility.HealthProviderOrganisationNetworkId);
                    if (existingHpii != null)
                    {
                        // Assign the HealthProviderIndividualId the existing Id, so it does not insert a new one but does an update instead
                        HealthProviderIndividual healthProvider = new HealthProviderIndividual();
                        if (healthProviderIndividualDataAccess.Get(existingHpii.HealthProviderIndividualId.Value, facility.HospitalId.Value, out healthProvider, (transaction != null ? transaction.Connection.CreateCommand() : null)))
                        {
                            healthProviderIndividual.HealthProviderIndividualId = healthProvider.HealthProviderIndividualId;
                            healthProviderIndividual.DateModified = healthProvider.DateModified;
                        }
                        // Update the Hisptory table
                        HealthProviderIndividualHpiiHistoryDl hpiiHistoryDataAccess = new HealthProviderIndividualHpiiHistoryDl(this.User);
                        HealthProviderIndividualHpiiHistory hpiiHistory = new HealthProviderIndividualHpiiHistory();
                        hpiiHistory.HealthProviderIndividualId = existingHpii.HealthProviderIndividualId.Value;
                        hpiiHistory.HealthProviderOrganisationNetworkId = existingHpii.HealthProviderOrganisationNetworkId;
                        hpiiHistory.Hpii = existingHpii.Hpii;
                        hpiiHistory.HpiiStatusId = existingHpii.HpiiStatusId;

                        if (!hpiiHistoryDataAccess.Insert(hpiiHistory, transaction))
                        {
                            throw new Exception(string.Format(ResponseStrings.DatabaseError, hpiiHistoryDataAccess.GetType().FullName));
            }
                      
        }

                    // Update the Hpii table
                    healthProviderIndividual.HpiI = hpiiValidationResult.HpiiNumber;
                    healthProviderIndividual.HpiiLastValidated = DateTime.Now;
                    healthProviderIndividual.HpiiStatusId = (int)hpiiValidationResult.HpiiStatus;
                    healthProviderIndividual.HealthProviderOrganisationNetworkId = facility.HealthProviderOrganisationNetworkId;
                    
                    if (!healthProviderIndividualDataAccess.Save(healthProviderIndividual, transaction))
                    {
                        throw new Exception(string.Format(ResponseStrings.DatabaseError, healthProviderIndividualDataAccess.GetType().FullName));
                    }
                }
            }
        }
        #endregion Private
    }
}